use Carp;

use FileHandle;
use Crypt::SSLeay;
use URI::URL;
use HTTP::Cookies;
use HTTP::Headers;
use HTTP::Request;
use HTTP::Response;
use HTTP::Daemon;

# -------------------------------------------------------------------

use strict;

use Deluge::Agent;
use Deluge::Mcp;
use Deluge::Prereq;
use Deluge::Script;
use Deluge::Stopwatch;
use Deluge::Translator;
use Deluge::Variable;

package Deluge::User;
use vars qw($AUTOLOAD);

# -------------------------------------------------------------------
# Internal Constants

my ($C_VISITED)  = 1;
my ($C_OUTSIDE)  = 2;
my ($C_IGNORED)  = 3;
my ($C_IMAGE)    = 4;
my ($C_SCHEME)   = 5;
my ($C_FILETYPE) = 6;

my ($MODE_DEPTH) = 1;
my ($MODE_BREADTH) = 2;
my ($MODE_WANDER) = 3;
my ($MODE_BOUNCE) = 4;
my ($MODE_PLAYBACK) = 5;

my ($STAT_ACTIVE) = 1;
my ($STAT_SLEEPING) = 2;
my ($STAT_FINISHED)= 3;

# -------------------------------------------------------------------

sub dump_state
{
    my ($self) = @_;
	my ($id) = $self->id();
	my ($prereq);

	print STDERR "User: $id\n";
	print STDERR "\tStatus: $self->{status}\n";

	if ($self->current_page) {
		print STDERR "\tCurrent Page:\n";
		$self->current_page->dump_state();
	}

	if (@{$self->{_page_queue}}) {
		print STDERR "\tPage Queue\n";
		foreach $prereq (@{$self->{_page_queue}}) {
			$prereq->dump_state();
		}
	}

	if (@{$self->{_image_queue}}) {
		print STDERR "\tImage Queue\n";
		foreach $prereq (@{$self->{_image_queue}}) {
			$prereq->dump_state();
		}
	}
}

# -------------------------------------------------------------------

sub _cold_reset
{
	my ($self) = @_;
	my ($prereq, $snooze, $item);
	
	($self->id == -1) &&
		die "ERROR: Trying to reset a def, not an instance\n";

	$self->shut_down_kids;
	$self->instances(0);

	if (! (($self->iters == -1) || ($self->restartable))) {
		($self->mcp->debug_level >= 1) &&
			(print STDERR "===Dead: " . $self->id . "\n");
		$self->status($STAT_FINISHED);
		$self->life_timer(0);
		$self->sub_timer(0);
		$self->cookie_jar(0);
		return;
	}

	$self->iters($self->iters + 1);
	($self->iters == 0) ? 
		($snooze = $self->mcp->user_ramp_time * rand()) :
			($snooze = $self->restart_time +
			 (($self->restart_spread * 2 * rand()) - $self->restart_spread));

	($self->mcp->debug_level >= 1) &&
		(print STDERR "===Restarting: " . $self->id . " " . $self->iters .
		 " ($snooze) \n");

	$self->reset_cookie_jar();
	($self->request_vars) && ($self->request_vars->prep);
	$self->all_trans_prepped(0);

	foreach $item (@{$self->{translators}}) {
		$item->reset;
	}

	# Prime the queue
	if (($self->attack_mode == $MODE_DEPTH) ||
		($self->attack_mode == $MODE_BREADTH) ||
		($self->attack_mode == $MODE_WANDER) ||
		($self->attack_mode == $MODE_BOUNCE)) {
		my ($prereq) = Deluge::Prereq->new("GET", $self->{top_url},
										   "", "a", 0, 1, $self,
										   $self->get_unique_prereq_id);
		$self->queue_page($prereq);
	} elsif ($self->attack_mode == $MODE_PLAYBACK) {
		if ($self->mcp->record_mode) {
			$self->script(Deluge::Script->new($self,
											  $self->script_filename, "a"));
		} else {
			my ($prereq);
			$self->event_num(0);
			$self->queue_playback_full_page();
		}
	}

	$self->life_timer->reset;
	$self->sub_timer->reset;
	
	$self->sleep_until($self->sub_timer->time  + $snooze);
	$self->status($STAT_SLEEPING);
	
	$self->{_per_url_hit_count} = {};
	$self->{_discarded_urls} = {};
	$self->{_pages_traversed_count} = 0;
}

# -------------------------------------------------------------------

sub queue_playback_full_page
{
    my ($self) = @_;
	my ($got_primary) = 0;

	while (1) {
		my ($prereq);

		if ($self->event_num >= $self->def->event_count) {
			$self->{event_num}++;
			last;
		}
			
		$prereq = @{$self->def->{events}}[$self->event_num]->
			clone($self, $self->get_unique_prereq_id);

		if (! $prereq) {
			$self->{event_num}++;
			last;
		}
		
		if ($prereq->is_image) {
			if (! $self->get_images) {
				$self->{event_num}++;
				next;
			}
			$prereq->dest_url($self->
							  insert_values_into_url($prereq->dest_url));
			$prereq->ref_url($self->insert_values_into_url($prereq->ref_url));
			$self->queue_image($prereq);
		} else {
			($got_primary) && last;
			$got_primary = 1;
			$prereq->dest_url($self->
							   insert_values_into_url($prereq->dest_url));
			$prereq->ref_url($self->insert_values_into_url($prereq->ref_url));
			$self->queue_page($prereq);
		}

		$self->{event_num}++;
	}
}

# -------------------------------------------------------------------

sub shut_down
{
    my ($self) = @_;

	$self->restartable(0);
	$self->_cold_reset();
}

# -------------------------------------------------------------------

sub awake_p
{
    my ($self) = @_;

	($self->status == $STAT_ACTIVE) && (return 1);
	# ($self->status == $STAT_FINISHED) && (return 0);
	return 0;
}

# -------------------------------------------------------------------

sub playback_mode
{
    my ($self) = @_;

	return ($self->attack_mode == $MODE_PLAYBACK);
}

# -------------------------------------------------------------------

sub queue_image
{
    my ($self, $prereq) = @_;

	if (($self->_req_ok_to_queue($prereq)) ||
		($self->playback_mode)) {
		push (@{$self->{_image_queue}}, $prereq);
	} else {
		$prereq->shut_down;
	}
}

# -------------------------------------------------------------------

sub get_unique_prereq_id
{
    my ($self) = @_;
	my ($upid) = $self->id . "." . $self->{_prereq_count};

	$self->{_prereq_count}++;
	return ($upid);
}

# -------------------------------------------------------------------

sub ok_to_preshuffle_queue
{
    my ($self) = @_;

	(($self->attack_mode == $MODE_WANDER) ||
	 ($self->attack_mode == $MODE_BOUNCE)) &&
		 return 1;

	return 0;
}

# -------------------------------------------------------------------

sub ok_to_presort_queue
{
    my ($self) = @_;

	(($self->attack_mode == $MODE_DEPTH) ||
	 ($self->attack_mode == $MODE_BREADTH)) &&
		 return 1;

	return 0;
}

# -------------------------------------------------------------------

sub _req_ok_to_queue
{
	my ($self, $prereq) = @_;
	my ($uri, $item);

	# Have we discarded this before?
	(exists ($self->{_discarded_urls}->{$prereq->dest_url})) &&
		(return 0);

	# Are we in too deep?
	($self->limit_depth) &&	($prereq->depth >= $self->limit_depth) &&
			# Don't add to _discarded_urls list.  If we get to it via
			# a shorter path, we still want to use it.
			(return 0);

	# Does it match the mandatory URL regexps?
	if (@{$self->mcp->require_url_regexps}) {
		foreach $item (@{$self->mcp->require_url_regexps}) {
			if ($prereq->dest_url !~ m|$item|) {
				$self->{_discarded_urls}->{$prereq->dest_url} = $C_FILETYPE;
				return 0;
			}
		}
	}

	# Does it match the illegal URL regexps?
	foreach $item (@{$self->mcp->ignore_url_regexps}) {
		if ($prereq->dest_url =~ m|$item|) {
			$self->{_discarded_urls}->{$prereq->dest_url} = $C_FILETYPE;
			return 0;
		}
	}

	# Is it an image tag?  Are we accepting images?
	if ((! $self->get_images) && ($prereq->is_image)) {
		$self->{_discarded_urls}->{$prereq->dest_url} = $C_OUTSIDE;
		return 0;
	}

	# Ok, simple tests out of the way.
	$uri = URI->new($prereq->dest_url);

	# Is it via a legal scheme?
	if (! Deluge::Etc::scheme_is_legal($uri->scheme,
									   $self->mcp->allow_secure)) {
		$self->{_discarded_urls}->{$prereq->dest_url} = $C_SCHEME;
		return 0;
	}

	# Is it a legal domain?
	if ($self->mcp->domain_match) {
		my ($site) = $uri->authority();

		if (($site !~ m|^$self->{domain_limit}$|) &&
			($site !~ m|\.$self->{domain_limit}$|)) {
			$self->{_discarded_urls}->{$prereq->dest_url} = $C_OUTSIDE;
			return 0;
		}
	}

    return 1;
}

# -------------------------------------------------------------------

sub note_prereq_as_queued
{
    my ($self, $prereq) = @_;
	
	$self->{_per_url_hit_count}->{$prereq->dest_url} ++;
	$prereq->in_agent(1);
}

# -------------------------------------------------------------------

sub endstate_check
{
    my ($self) = @_;

	if (($self->limit_pages_traversed) &&
		($self->{_pages_traversed_count} >= $self->limit_pages_traversed)) {
		$self->_cold_reset();
		return 1;
	}

	if (($self->limit_attack_time) &&
		($self->life_timer->elapsed > $self->limit_attack_time)) {
		$self->_cold_reset();
		return 1;
	}

	return 0;
}

# -------------------------------------------------------------------

sub _req_ok_to_register
{
    my ($self, $prereq) = @_;

	($self->playback_mode) && (return 1);
	
	# Have we discarded this before?
	(exists ($self->{_discarded_urls}->{$prereq->dest_url})) &&
		(return 0);

	# Have we hit it too many times already?
	if ($self->limit_hits_per_url) {
		if (exists ($self->{_per_url_hit_count}->{$prereq->dest_url})) {
			if ($self->{_per_url_hit_count}->{$prereq->dest_url} >=
				$self->limit_hits_per_url) {
				$self->{_discarded_urls}->{$prereq->dest_url} = $C_VISITED;
				return 0;
			}
		} else {
			# Haven't hit it at all.  Initialize the counter.
			$self->{_per_url_hit_count}->{$prereq->dest_url} = 0;
		}
	}

	return 1;
}

# -------------------------------------------------------------------

sub _queue_script_events
{
    my ($self) = @_;
	my ($got_primary) = 0;
	
	while (1) {
		my ($prereq);
		
		if ($self->script->event_is_image) {
			($prereq = $self->script->event_to_prereq($self, 0)) || last;
			($self->get_images) || next;
			($self->mcp->debug_level >= 3) &&
				(print STDERR "Queueing image: " . $prereq->dest_url . "\n");
			$self->queue_image($prereq);
		} else {
			($got_primary) && last;
			$got_primary = 1;
			($prereq = $self->script->event_to_prereq($self, 1)) || last;
			$self->queue_page($prereq);
			($self->mcp->debug_level >= 3) &&
				(print STDERR "Queueing PAGE: " . $prereq->dest_url . "\n");
		} 
	}
}

# -------------------------------------------------------------------

sub _get_next_page
{
    my ($self) = @_;
	my ($prereq);
			
	while (1) {
		if (! @{$self->{_page_queue}}) {
			$self->_cold_reset();
			return undef;
		}

		(($self->attack_mode == $MODE_BREADTH) ||
		 ($self->attack_mode == $MODE_WANDER) ||
		 ($self->attack_mode == $MODE_PLAYBACK)) &&
			($prereq = shift(@{$self->{_page_queue}}));

		($self->attack_mode == $MODE_DEPTH) &&
			($prereq = pop(@{$self->{_page_queue}}));

		($self->attack_mode == $MODE_BOUNCE) &&
			($prereq = splice(@{$self->{_page_queue}},
							  int(($#{@{$self->{_page_queue}}}+1)*rand()),1));
	
		($self->_req_ok_to_register($prereq)) ?
			(return $prereq) : ($prereq->shut_down);
	}
}

# -------------------------------------------------------------------
# Return -1 if dead, 0 if sleeping, 1 if active

sub _do_stuff_active
{
    my ($self) = @_;
	my ($prereq);
	my ($active) = 0;

	if (! $self->current_page->executed) {
		if (($self->current_page->failed) && (! $self->playback_mode)) {
			$self->_warm_reset;
			($self->status == $STAT_FINISHED) ? (return -1) : (return 0);
		}

		return 0;
	}
	
	if ($self->attack_mode == $MODE_WANDER) {
		if ($self->current_page->response->code >= 400) {
			while ($prereq = shift(@{$self->{_page_queue}})) {
				$prereq->shut_down;
			}
			$prereq = 0
		}
	}

	if (! $self->get_images) {
		$self->_warm_reset();
		return 0;
	}

	foreach $prereq (@{$self->{_image_queue}}) {
		(($prereq->ignored) || ($prereq->failed) || ($prereq->executed)) &&
			next;

		$active = 1;
	}

	if (! $active) {
		$self->_warm_reset;
		($self->status == $STAT_FINISHED) ? (return -1) : (return 0);
	}
			
	return (1);
}

# -------------------------------------------------------------------

sub images_to_agent
{
    my ($self) = @_;
	my ($prereq);

	foreach $prereq (@{$self->{_image_queue}}) {
		if ($self->_req_ok_to_register($prereq)) {
			$self->mcp->add_req_to_agent($prereq);
			$self->note_prereq_as_queued($prereq);
		} else {
			$prereq->shut_down;
		}
	}
}

# -------------------------------------------------------------------

sub sleeping
{
    my ($self) = @_;

	($self->status == $STAT_FINISHED) && (return 999999);
	
	($self->status == $STAT_SLEEPING) ?
		(return ($self->sleep_until)) : (return 0);
}

# -------------------------------------------------------------------
# Return -1 if dead, 0 if sleeping, 1 if active

sub execute_state
{
    my ($self) = @_;
	my ($prereq);

	($self->status == $STAT_FINISHED) && (return -1);
	($self->endstate_check) && (return 0);

	if ($self->status == $STAT_SLEEPING) {
		my ($timeout) = $self->sub_timer->time() - $self->sleep_until;

		($timeout < 0) && (return 0);

		if (! ($prereq = $self->_get_next_page)) {
			($self->status == $STAT_FINISHED) ? (return -1) : (return 0);
		}

		$prereq->wait_for_click($timeout);
		$self->current_page($prereq);
		$prereq = 0;

		$self->mcp->add_req_to_agent($self->current_page);
		$self->note_prereq_as_queued($self->current_page);
		$self->status($STAT_ACTIVE);
		$self->sleep_until(0);
		return 1;
	}

	($self->status == $STAT_ACTIVE) && (return ($self->_do_stuff_active));
}

# -------------------------------------------------------------------

sub queue_page
{
    my ($self, $prereq) = @_;

	(($self->_req_ok_to_queue($prereq)) || ($self->playback_mode)) ?
		(push (@{$self->{_page_queue}}, $prereq)) :	($prereq->shut_down);
}

# -------------------------------------------------------------------

sub shut_down_kids
{
	my ($self) = @_;
	my ($prereq);
	
	if ($self->current_page) {
		$self->current_page->shut_down;
		$self->current_page(0);
	}

	while ($prereq = shift(@{$self->{_image_queue}})) {
		$prereq->shut_down;
	}
	$prereq = 0;

	$self->{_image_queue} = [];

	while ($prereq = shift(@{$self->{_page_queue}})) {
		$prereq->shut_down;
	}
	$prereq = 0;

	$self->{_page_queue} = [];
}

# -------------------------------------------------------------------

sub _warm_reset
{
	my ($self) = @_;
	my ($prereq);
	
	($self->id == -1) && die "ERROR: Trying to reset a def, not an instance\n";

	($self->mcp->debug_level >= 2) &&
		(print STDERR "---Warm: " . $self->id . "\n");

	$self->current_page(undef);

	while ($prereq = shift @{$self->{_image_queue}}) {};
	$prereq = 0;

	($self->playback_mode) && ($self->queue_playback_full_page);

	$self->sub_timer->reset;

	$self->{_pages_traversed_count} ++;
	$self->status($STAT_SLEEPING);
	$self->sleep_until($self->sub_timer->time + $self->delay_time +
					   (($self->delay_spread *2*rand())-$self->delay_spread));
}

# -------------------------------------------------------------------

sub reset_cookie_jar
{
    my ($self) = @_;

	($self->accept_cookies) && ($self->cookie_jar(HTTP::Cookies->new));
}

# -------------------------------------------------------------------

sub clear_visited_list
{
    my ($self) = @_;

	$self->{_per_url_hit_count} = {};
	$self->{_discarded_urls} = {};
}

# -------------------------------------------------------------------

sub script_filename
{
    my ($self) = @_;

	return $self->script_dir . "/" . $self->script_file;
}

# -------------------------------------------------------------------

sub start_proxy
{
    my ($self) = @_;
	my ($connection, $request, $response);
	
	while ($connection = $self->{_proxy_daemon}->accept) {
		my ($pid);

		$self->proxy_counter($self->proxy_counter + 1);
		
		($pid = fork) || last;

		# i'm the parent.
		close $connection;
		next;
	}

	# i'm the child.
	$request = $connection->get_request();
	($self->accept_cookies) || ($request->remove_header(qw(Cookie)));
	$self->script->basic_request_to_event($request);

	if (Deluge::Etc::scheme_is_legal($request->url->scheme, 0)) {
		$response = $self->{_proxy_agent}->request($request);
		($self->accept_cookies) || ($response->remove_header(qw(Set-Cookie)));
		$self->script->response_to_event($request, $response, $self);
	} else {
		$response = HTTP::Response->new(403, "Forbidden");
		$response->content("bad scheme: " . $request->url->scheme . "\n");
		$self->script->bad_request_to_event($request);
	}

	$connection->send_response($response);
	close ($connection);
	exit (0);
}

# -------------------------------------------------------------------

sub compile_script_into_def
{
    my ($self) = @_;
	my ($counter) = 0;

	($self->instances) || return;
	
	$self->script(Deluge::Script->new($self, $self->script_filename, "r"));

	while (1) {
		my ($prereq);
		my ($is_image) = $self->script->event_is_image;
		
		($prereq = $self->script->event_to_prereq($self,
												  (! $is_image))) || last;
		push (@{$self->{events}}, $prereq);
		
		$counter ++;
	}

	$self->script(0);
	$self->event_count($counter);
}

# -------------------------------------------------------------------

sub make_user_from_def
{
    my ($self, $id) = @_;
	my ($clone) = Deluge::User->new($self->mcp);
	my ($item);

	($self->iters >= 0) &&
		die "ERROR: Trying to make a user from an instance, not a def\n";

	foreach $item (qw(accept_cookies
					  compare_content
					  get_images
					  restartable

					  delay_time
					  delay_spread
					  limit_depth
					  limit_hits_per_url
					  limit_pages_traversed
					  limit_attack_time
					  restart_time
					  restart_spread

					  attack_type
					  script_dir
					  script_file
					  top_url

					  domain_limit
					  attack_mode
					  defname
					 )) {
		$clone->{$item} = $self->{$item};
	}

	$clone->id($id);
	$clone->def($self);

	foreach $item (@{$self->{translators}}) {
		push(@{$clone->{translators}}, $item->clone);
	}

	if ($self->request_vars) {
		$clone->request_vars($self->request_vars->clone);
		$clone->request_vars->user($clone);
	}
	
	$clone->_cold_reset;

    return ($clone);
}

# -------------------------------------------------------------------

sub prep_for_record
{
	my ($self) = @_;
	my ($daemon);

	($self->attack_mode == $MODE_PLAYBACK) ||
		main::usage("User $self->{defname} isn't using attack_type [playback]");

	$self->{_proxy_agent} = Deluge::Agent->new;

	$self->{_proxy_agent}->env_proxy();
	$self->{_proxy_agent}->timeout($self->mcp->timeout);

	$self->{_proxy_daemon} = HTTP::Daemon->new("LocalAddr",
											   $self->mcp->hostname,
											   "LocalPort",
											   $self->mcp->proxy_http_port);

	($self->{_proxy_daemon}) ||
		main::usage("Proxy failed to start.  Port may be unavailable.");

	$SIG{CLD} = $SIG{CHLD} = sub { wait; };
}

# -------------------------------------------------------------------

sub do_secure_serial
{
	# This is only here until I can figure how to get the parallel user
	# agent to do secure transactions.  dammit.
	
    my ($self, $prereq, $request) = @_;
	my ($agent, $response);

	$request->remove_header(qw(User-Agent));

	$agent = Deluge::Agent->new;
	($self->cookie_jar) && ($agent->cookie_jar($self->cookie_jar));
	$agent->timeout($self->mcp->timeout);
	$agent->from($self->mcp->owner_email);

	($self->playback_mode) ?
		($agent->agent($prereq->user_agent_code)) :
			($agent->agent($Deluge::Etc::AgentCode));

	print STDERR "Running secure request outside of queue... ";
	$response = $agent->request($request);
	print STDERR "done\n";
	print STDERR "\t" . $prereq->dest_url . "\n";

	$prereq->preprocess($response);

	return (1);
}

# -------------------------------------------------------------------

sub extract_values_from_url
{
    my ($self, $url) = @_;
	my ($trans);
	my ($ok) = 1;

	foreach $trans (@{$self->{translators}}) {
		$trans->extract_value_from_url($url);
		($trans->val) || ($ok = 0);
	}

	$self->all_trans_prepped($ok);
}

# -------------------------------------------------------------------

sub insert_values_into_url
{
    my ($self, $url) = @_;
	my ($trans);

	foreach $trans (@{$self->{translators}}) {
		$url = $trans->insert_value_into_url($url);
	}

	return ($url);
}

# -------------------------------------------------------------------

sub _check_config_info
{
    my ($self) = @_;
	my ($in) = "in user_def [$self->{defname}]";

	($self->attack_type) ||
		die "ERROR: Missing mandatory [attack_type] assignment $in\n";

	if ($self->attack_type =~ m|^depth$|i) {
		$self->attack_mode($MODE_DEPTH);
	} elsif ($self->attack_type =~ m|^breadth$|i) {
		$self->attack_mode($MODE_BREADTH);
	} elsif ($self->attack_type =~ m|^wander$|i) {
		$self->attack_mode($MODE_WANDER);
	} elsif ($self->attack_type =~ m|^bounce$|i) {
		$self->attack_mode($MODE_BOUNCE);
	} elsif ($self->attack_type =~ m|^playback$|i) {
		$self->attack_mode($MODE_PLAYBACK);
	} else {
		main::usage("Unknown [attack_type] assignment $self->{attack_type} $in");
	}

	if (($self->attack_mode == $MODE_DEPTH) ||
		($self->attack_mode == $MODE_BREADTH) ||
		($self->attack_mode == $MODE_BOUNCE) ||
		($self->attack_mode == $MODE_WANDER)) {
		my ($uri, $dname, $count);
		my (@sitelist) = ();

		($self->top_url) ||
			main::usage("Missing mandatory [top_url] assignment $in");

		$dname = "";

		$uri = URI->new($self->top_url);

		@sitelist = split('\.', $uri->authority());
		$count = $self->mcp->domain_match;

		while ($count) {
			$dname = "\\." . pop(@sitelist) . $dname;
			$count--;
		}
		$dname =~ s|^\\\.||;

		$self->domain_limit($dname);
	}

	if (! $self->mcp->record_mode) {
		($self->{instances} == -1) &&
			main::usage("Missing mandatory [instances] assignment $in");

		($self->{delay_time} == -1) &&
			main::usage("Missing mandatory [delay_time] assignment $in");
		
		($self->{delay_spread} == -1) &&
			main::usage("Missing mandatory [delay_spread] assignment $in");
		
		($self->{restart_time} == -1) &&
			main::usage("Missing mandatory [delay_time] assignment $in");
		
		($self->{restart_spread} == -1) &&
			main::usage("Missing mandatory [delay_spread] assignment $in");
	}
	
	if ($self->attack_mode == $MODE_PLAYBACK) {
		($self->script_dir) ||
			main::usage("Missing [script_dir] assignment $in");

		(-d $self->script_dir) ||
			main::usage("Value for [script_dir] \"$self->{script_dir}\" ".
						"is not a directory $in\n");

		($self->script_file) ||
			main::usage("Missing [script_file] assignment $in");


		if ($self->mcp->record_mode) {
			(-f $self->script_filename) &&
				main::usage("[script_dir] is not empty.");
		} else {
			(-f $self->script_filename) ||
				main::usage("Value for [script_file] \"$self->{script_file}\"".
							" is not a file in directory $self->{script_dir} ".
							"from assignment $in\n");
			
			(-r $self->script_filename) ||
				main::usage("Script file \"$self->{script_file}\" ".
							"is not readable from assignment $in\n");

			$self->compile_script_into_def();
		}
	}
}

# -------------------------------------------------------------------

sub _read_config_file
{
    my ($self, $cfg, $tln) = @_;
	my ($line, $tag);

  LINE:
	while ($line = $cfg->get_next_line) {
		($line eq "END") && (return);
		
		my ($car, $cdr) = $cfg->get_pair($line);

		# Switch values
		foreach $tag (qw(accept_cookies
						 compare_content
						 get_images
						 playback_errors
						 restartable)) {
			if ($car eq $tag) {
				$self->{$tag} = $cfg->get_switch($car, $cdr);
				next LINE;
			}
		}
		
		# Numerical values
		foreach $tag (qw(attack_time
						 instances
						 delay_time
						 delay_spread
						 limit_depth
						 limit_hits_per_url
						 limit_pages_traversed
						 limit_attack_time
						 restart_time
						 restart_spread)) {
			if ($car eq $tag) {
				$self->{$tag} = $cfg->get_number($car, $cdr);
				next LINE;
			}
		}

		# String values
		foreach $tag (qw(attack_type
						 script_dir
						 script_file
						 top_url)) {
			if ($car eq $tag) {
				$self->{$tag} = $cdr;
				next LINE;
			}
		}

		# Objects
		if ($car eq "request_vars") {
			$self->{$car} = Deluge::Variable->new($self, $cfg);
			next LINE;
		}

		if ($car eq "translator") {
			push(@{$self->{translators}},
				 Deluge::Translator->new($cfg));
			next LINE;
		}

		$cfg->error("Unknown variable [$car]");
	}

	main::usage("No END tag found for user_def starting at line $tln");
}

# -------------------------------------------------------------------

sub DESTROY
{
    my ($self) = @;
}

# -------------------------------------------------------------------

sub _initialize
{
    my ($self, $mcp) = @_;

	# -- Public: Config file --
	# Switches
	$self->{accept_cookies} = 0;
	$self->{compare_content} = 0;
	$self->{get_images} = 0;
	$self->{restartable} = 0;
	$self->{playback_errors} = 0;

	# Numerical
	$self->{instances} = -1;
	$self->{delay_time} = -1;
	$self->{delay_spread} = -1;
	$self->{limit_depth} = 0;
	$self->{limit_hits_per_url} = 0;
	$self->{limit_pages_traversed} = 0;
	$self->{limit_attack_time} = 0;
	$self->{restart_time} = 0;
	$self->{restart_spread} = 0;

	# Strings
	$self->{attack_type} = "";
	$self->{script_dir} = "";
	$self->{script_file} = "";
	$self->{top_url} = "";

	# Methods
	$self->{request_vars} = 0;
	$self->{translators} = [];

	# -- Computed --
	$self->{domain_limit} = "";  # From top_url and cfg->domain_limit
	$self->{attack_mode} = 0;
	
	# -- Def specific --
	$self->{defname} = "";
	$self->{events} = [];
	$self->{event_count} = 0;

	# -- Instance specific --
	$self->{def} = 0;
	$self->{id} = -1;
	$self->{iters} = -1;
	$self->{event_num} = 0;
	$self->{all_trans_prepped} = 0;
	$self->{life_timer} = Deluge::Stopwatch->new();
	$self->{sub_timer} = Deluge::Stopwatch->new();
	$self->{page_timer} = Deluge::Stopwatch->new();
	# $self->{page_elapsed} = 0;
	$self->{sleep_until} = -1;
	$self->{status} = $STAT_SLEEPING;

	$self->{cookie_jar} = 0;
	$self->{mcp} = $mcp;
	$self->{current_page} = 0;

	
	# -- Private --
	$self->{_prereq_count} = 0;
	
	$self->{_page_queue} = [];
	$self->{_image_queue} = [];

	$self->{_per_url_hit_count} = {};
	$self->{_discarded_urls} = {};
	$self->{_pages_traversed_count} = 0;

	# -- Proxy stuff --
	$self->{script} = 0;

	$self->{_proxy_agent} = 0;
	$self->{_proxy_daemon} = 0;
	$self->{proxy_counter} = 0;
}

# -------------------------------------------------------------------

sub AUTOLOAD
{
    my ($self) = shift;
	my ($type) = ref($self) || main::confess "$self is not an object\n";
	my ($name) = $AUTOLOAD;

	$name =~ s|.*:||;

	(exists $self->{$name}) || main::confess "$name is not a method here\n";
	($name =~ m|^_|) && main::confess "Access to method $name denied\n";

	(@_) ? (return $self->{$name} = shift) : (return $self->{$name});
}

# -------------------------------------------------------------------

sub new
{
	my ($above, $mcp, $cfg, $defname) = @_;
	my ($class) = ref($above) || $above;
	my ($self) = {};

	bless ($self, $class);

	$self->_initialize($mcp);

	if ($cfg) {
		my ($toplinenum) = $cfg->linenum;
		$self->defname($defname);
		$self->_read_config_file($cfg, $toplinenum);
		$self->_check_config_info($mcp);
	}
	
	return ($self);
}

# -------------------------------------------------------------------

package main;

1;
