use Carp;

use FileHandle;
use Crypt::SSLeay;
use HTML::LinkExtor;
use HTML::TokeParser;
use LWP::Parallel::UserAgent;

# -------------------------------------------------------------------

use strict;

require Deluge::Etc;
require Deluge::Log;
require Deluge::Mcp;
require Deluge::User;

package Deluge::Prereq;
use vars qw($AUTOLOAD);

# -------------------------------------------------------------------
# Internal Constants

my ($C_META_HTTP) = 1;
my ($C_META_NAME) = 2;

# -------------------------------------------------------------------

sub dump_state
{
    my ($self) = @_;

	print STDERR "\t\tPrereq: $self->{id}\n";
	print STDERR "\t\t\tURL: $self->{dest_url}\n";
	print STDERR "\t\t\tIn_agent: $self->{in_agent}\n";
	print STDERR "\t\t\tExecuted: $self->{executed}\n";
	print STDERR "\t\t\tFailed: $self->{failed}\n";
	print STDERR "\t\t\tIgnored: $self->{ignored}\n";
}

# -------------------------------------------------------------------

sub shut_down
{
    my ($self) = @_;

	$self->ignored(1);
}

# -------------------------------------------------------------------

sub agent_callback
{
    my ($self, $content, $response, $protocol, $entry) = @_;
	
	$self->callback(1);
	$self->req_wait->set_mark;
	$self->user->page_timer->set_mark;

	($self->is_image) || ($self->{content} .= $content);
	$self->{content_length} += length($content);
		
	return undef;
}

# -------------------------------------------------------------------

sub preprocess
{
    my ($self, $response) = @_;

	$self->response($response);
	
	if ($self->callback) {
		$self->response->content($self->content);
		$self->{content} = "";
	} else {
		$self->req_wait->set_mark;
		$self->user->page_timer->set_mark;
	}
}

# -------------------------------------------------------------------

sub push_ext_header
{
    my ($self, $car, $cdr) = @_;

	push(@{$self->{_ext_headers}}, $car);
	push(@{$self->{_ext_headers}}, $cdr);
}

# -------------------------------------------------------------------

sub shift_ext_header
{
    my ($self) = @_;
	my ($car, $cdr);

	(@{$self->{_ext_headers}}) || return ("", "");
	
	$car = shift(@{$self->{_ext_headers}});
	$cdr = shift(@{$self->{_ext_headers}});

	return ($car, $cdr);
}

# -------------------------------------------------------------------

sub _cache_link
{
    my ($self, $url, $tag) = @_;
	my ($prereq, $is_image);

	($self->user->playback_mode) &&
		($self->user->extract_values_from_url($url));
	
	if (Deluge::Etc::url_is_image($url, $tag)) {
		if ($self->user->get_images) {
			push (@{$self->{_image_link_cache_urls}}, $url);
			push (@{$self->{_image_link_cache_tags}}, $tag);
		}
	} else {
		push (@{$self->{_page_link_cache_urls}}, $url);
		push (@{$self->{_page_link_cache_tags}}, $tag);
	}
}

# -------------------------------------------------------------------

sub _parse_response_meta_tag_pairs
{
    my ($self, $metatype, $metasubtype, $car, $cdr) = @_;

	if (($metatype == $C_META_HTTP) &&
		(($metasubtype =~ m|^refresh$|i) ||
		 ($metasubtype =~ m|^content-style-type$|i)) &&
		(($car =~ m|^content$|) && ($cdr =~ m|\=(.*)|))) {
		$self->_cache_link($1, "a");
	}

	if (($metatype == $C_META_NAME) &&
		($metasubtype =~ m|^postinfo$|i) &&
		($car =~ m|^content$|) &&
		($cdr =~ m|\=(.*)|)) {
		$self->_cache_link($1, "a");
	}
}

# -------------------------------------------------------------------

sub _parse_response_vis_text
{
	my ($self, $text) = @_;
	my (@temp) = ();
	my ($item);

	foreach $item (@{$self->{_pos_vis}}) {
		($text =~ m|$item|) || (push (@temp, $item));
	}
	$self->{_pos_vis} = [];
	foreach $item (@temp) {
		push (@{$self->{_pos_vis}}, $item);
	}

	@temp = ();
	foreach $item (@{$self->{_neg_vis}}) {
		($text =~ m|$item|) ?
			 ($self->Log->new_tag($Deluge::Log::TAG_NVVR, $item)) :
				 (push (@temp, $item));
	}
	$self->{_neg_vis} = [];
	foreach $item (@temp) {
		push (@{$self->{_neg_vis}}, $item);
	}
}

# -------------------------------------------------------------------

sub _parse_response_invis_text
{
    my ($self, $text) = @_;
	my (@temp) = ();
	my ($item);

	foreach $item (@{$self->{_pos_invis}}) {
		($text =~ m|$item|) || (push (@temp, $item));
	}
	$self->{_pos_invis} = [];
	foreach $item (@temp) {
		push (@{$self->{_pos_invis}}, $item);
	}
	
	@temp = ();
	foreach $item (@{$self->{_neg_invis}}) {
		($text =~ m|$item|) ?
			($self->Log->new_tag($Deluge::Log::TAG_NIVR, $item)) :
				(push (@temp, $item));
	}
	$self->{_neg_invis} = [];
	foreach $item (@temp) {
		push (@{$self->{_neg_invis}}, $item);
	}
}

# -------------------------------------------------------------------

sub _parse_response_tag_start
{
    my ($self, $lump) = @_;
	my ($tag) = $lump->[1];
	my (%attr) = %{$lump->[2]};
	my (@attrseq) = @{$lump->[3]};
	my ($car, $cdr);

	if ($tag eq "meta") {
		my ($metatype) = 0;
		my ($metasubtype);

		if (exists($attr{"http-equiv"})) {
			$metatype = $C_META_HTTP;
			$metasubtype = $attr{"http-equiv"};
		}

		if (exists($attr{"name"})) {
			$metatype = $C_META_NAME;
			$metasubtype = $attr{"name"};
		}

		if (! $metatype) {
			# ERROR: Mandatory META tag type missing
			($self->user->mcp->verbose_logs) &&
				$self->Log->new_tag($Deluge::Log::TAG_XMME, 1);
			return;
		}
		
		foreach $car (keys(%attr)) {
			$cdr = $attr{$car};
			(($car eq "http-equiv") || ($car eq "name")) && next;
			
			$self->_parse_response_meta_tag_pairs($metatype, $metasubtype,
												  $car, $cdr);
		}
	}

    $self->_parse_response_invis_text($lump->[4]);

}

# -------------------------------------------------------------------

sub _parse_response_tag_end
{
    my ($self, $lump) = @_;
	my ($tag) = $lump->[1];

	$self->_parse_response_invis_text($lump->[2]);
}

# -------------------------------------------------------------------

sub _prep_local_expects
{
    my ($self) = @_;
	my ($car, $cdr);

	foreach $car (keys(%{$self->user->mcp->pos_vis_regexps})) {
		($self->dest_url =~ m|$car|) &&
			(push (@{$self->{_pos_vis}},
				   $self->user->mcp->pos_vis_regexps->{$car}));
	}

	foreach $car (keys(%{$self->user->mcp->neg_vis_regexps})) {
		($self->dest_url =~ m|$car|) &&
			(push (@{$self->{_neg_vis}},
				   $self->user->mcp->neg_vis_regexps->{$car}));
	}

	foreach $car (keys(%{$self->user->mcp->pos_invis_regexps})) {
		($self->dest_url =~ m|$car|) &&
			(push (@{$self->{_pos_invis}},
				   $self->user->mcp->pos_invis_regexps->{$car}));
	}

	foreach $car (keys(%{$self->user->mcp->neg_invis_regexps})) {
		($self->dest_url =~ m|$car|) &&
			(push (@{$self->{_neg_invis}},
				   $self->user->neg_invis_regexps->{$car}));
	}
}

# -------------------------------------------------------------------

sub _parse_response
{
    my ($self) = @_;
	my ($parser) = HTML::TokeParser->new(\$self->response->content);
	my ($item, $type);

	if (! $parser) {
		($self->user->mcp->verbose_logs) &&
			$self->Log->new_tag($Deluge::Log::TAG_XNRD, 1);
		return 0;
	}

	$self->_prep_local_expects;

	while (1) {
		if ($item = $parser->get_trimmed_text()) {
			$self->_parse_response_vis_text($item);
			next;
		}

		if ($item = $parser->get_token()) {
			$type = $item->[0];

			if ($type eq "S") {
				$self->_parse_response_tag_start($item);
				next;
			}

			if ($type eq "E") {
				$self->_parse_response_tag_end($item);
				next;
			}

			if ($type eq "T") {
				$self->_parse_response_vis_text($item->[1]);
				next;
			}

			if (($type eq "C") || ($type eq "D") || ($type eq "PI")) {
				$self->_parse_response_invis_text($item->[1]);
				next;
			}

			($self->user->mcp->verbose_logs) &&
				$self->Log->new_tag($Deluge::Log::TAG_XUKT, $type);
			next;
		}

		last;
	}

	foreach $item (@{$self->{_pos_vis}}) {
		$self->{Log}->new_tag($Deluge::Log::TAG_PVVD, $item);
	}

	foreach $item (@{$self->{_pos_invis}}) {
		$self->{Log}->new_tag($Deluge::Log::TAG_PIVD, $item);
	}
}

# -------------------------------------------------------------------

sub _link_finder_callback
{
	my ($self, $tag, %attr) = @_;
	my ($item, $val);
			
	foreach $item (keys(%attr)) {
		$val = $attr{$item};
		$val =~ s|%7e|~|ig;

		(($val =~ m|\#|) && ($tag eq "img")) && next;
		$val =~ s|\#.*||ig;

		$self->_cache_link($val, $tag);
	}
}

# -------------------------------------------------------------------

sub _process_links
{
    my ($self) = @_;
	my ($url, $newurl, $tag, $prereq);
	my (@temp_list);
	
	while (@{$self->{_page_link_cache_urls}}) {
		$url = shift(@{$self->{_page_link_cache_urls}});
		$tag = shift(@{$self->{_page_link_cache_tags}});

		$newurl = main::url($url, $self->response->base)->abs;

		($self->user->mcp->verbose_logs) &&
			$self->Log->new_tag($Deluge::Log::TAG_GOTO, $newurl);

		$prereq = Deluge::Prereq->new("GET", $newurl, $self->dest_url,
									  $tag, $self->depth + 1, 1,
									  $self->user,
									  $self->user->get_unique_prereq_id);
		push (@temp_list, $prereq);
	}

	($self->user->ok_to_presort_queue) && (@temp_list = sort(@temp_list));

	if ($self->user->ok_to_preshuffle_queue) {
		my ($i);
			
		for ($i=0; $i<$#temp_list; $i++) {
			my ($shuffle) = int(($#temp_list + 1) * rand());
			my ($temp_prereq) = $temp_list[$i];
			$temp_list[$i] = $temp_list[$shuffle];
			$temp_list[$shuffle] = $temp_prereq;
		}
	}
	
	while ($prereq = shift(@temp_list)) {
		$self->user->queue_page($prereq);
	}

	if ($self->user->get_images) {
		while (@{$self->{_image_link_cache_urls}}) {
			$url = shift(@{$self->{_image_link_cache_urls}});
			$tag = shift(@{$self->{_image_link_cache_tags}});
			
			$newurl = main::url($url, $self->response->base)->abs;

			($self->user->mcp->verbose_logs) &&
				$self->Log->new_tag($Deluge::Log::TAG_GIMG, $newurl);
			
			$prereq = Deluge::Prereq->new("GET", $newurl, $self->dest_url,
										  $tag, $self->depth, 0,
										  $self->user,
										  $self->user->get_unique_prereq_id);
			$self->user->queue_image($prereq);
		}
	}
}

# -------------------------------------------------------------------

sub _compare_responses
{
    my ($self) = @_;

	($self->user->mcp->debug_level >= 2) &&
		(print STDERR "Need to write _compare_responses code\n");
}

# -------------------------------------------------------------------

sub _dump_response
{
	my ($self) = @_;
	my ($fh);

	($self->is_image) && return;
	
	$fh = new FileHandle $self->id . "_client", "w";
	$fh->print($self->dest_url);
	$fh->print("\n------------------------------------\n");
	$fh->print($self->response->request->content);
	$fh->print("\n------------------------------------\n");
	$fh->print($self->response->request->headers->as_string);
	$fh->close;

	$fh = new FileHandle $self->id . "_server.html", "w";
	$fh->print($self->response->content);
	$fh->close;
}

# -------------------------------------------------------------------

sub process_response
{
    my ($self) = @_;

	($self->executed) ? (return) : ($self->executed(1));

	(defined $self->response->code) || return;
	
	$self->Log->new_tag($Deluge::Log::TAG_DNAM, $self->user->defname);
	$self->Log->new_tag($Deluge::Log::TAG_CODE, $self->response->code);

	($self->wait_for_click) &&
		($self->Log->new_tag($Deluge::Log::TAG_DTIM, $self->wait_for_click));

	($self->user->mcp->debug_level >= 1) &&
		($self->response->code >= 400) &&
			(print STDERR "Code " . $self->response->code . " on URL " .
			 $self->dest_url . "\n");

	($self->user->mcp->dump_responses) && $self->_dump_response();

	if ($self->user->compare_content){
		($self->expected_code) &&
			($self->response->code != $self->expected_code) &&
				$self->Log->new_tag($Deluge::Log::TAG_CDMM,
									$self->expected_code);
		
		($self->expected_size) &&
			($self->content_length != $self->expected_size) &&
				$self->Log->new_tag($Deluge::Log::TAG_LNMM,
									$self->content_length);

		($self->compare_fname) && ($self->_compare_responses);
	}

    if (! $self->user->playback_mode) {     # SPEEDUP
        ($self->is_image) || ($self->_parse_response);
    }

	($self->user->accept_cookies) &&
		($self->user->cookie_jar->extract_cookies($self->response));

	($self->user->playback_mode) && ($self->user->all_trans_prepped) &&
		return;

	# Playback mode has to do this to extract session IDs
    if ($self->primary) {
        my ($links) =
            HTML::LinkExtor->new(sub { $self->_link_finder_callback(@_) },
                                 $self->response->base);
        $links->parse($self->response->content);
        ($self->user->playback_mode) || ($self->_process_links); # SPEEDUP
    }
}

# -------------------------------------------------------------------

sub DESTROY
{
    my ($self) = @_;
	my ($item);

	($self->ignored) && return;
	($self->Log) || return;
	($self->dest_url) || return;
	($self->in_agent) || return;
	
	if (! $self->failed) {
		($self->response) &&
			($self->Log->new_tag($Deluge::Log::TAG_SIZE,
								 $self->content_length));

		#($self->req_wait->get_mark >= 0) &&
		($self->Log->new_tag($Deluge::Log::TAG_ELAP,
							 $self->req_wait->get_mark));

		# ($self->user->page_elapsed >= 0) &&
		if ($self->primary) {
			($self->Log->new_tag($Deluge::Log::TAG_PTIM,
								 $self->user->page_timer->get_mark));
		}
	}
		
	foreach $item ($self->Log->dump_tags()) {
		$self->user->mcp->log_fh->print("$item\n");
	}

	$self->user->mcp->log_fh->print("\n");
	$self->user->mcp->log_fh->flush;
}

# -------------------------------------------------------------------

sub clone
{
    my ($self, $user, $id) = @_;
	my ($vname, $item);

	my ($clone) = Deluge::Prereq->new($self->method, $self->dest_url,
									  $self->ref_url, $self->tag,
									  $self->depth, $self->primary,
									  $user, $id);

	foreach $vname (qw(
					   req_content
					   response
					   content
					   content_length
					   expected_code
					   expected_size
					   user_agent_code
					   compare_fname
					   secure_request
					   
					   executed
					   in_agent
					   callback
					   failed
					   ignored
					   
					   delete_cookies_when_registered
					   clear_visited_list_when_registered
					  )) {
		$clone->{$vname} = $self->{$vname};
	}

	foreach $vname (qw(
					   _page_link_cache_urls
					   _page_link_cache_tags
					   _image_link_cache_urls
					   _image_link_cache_tags

					   _pos_vis
					   _pos_invis
					   _neg_vis
					   _neg_invis

					   _ext_headers
					   ext_content
					  )) {
		foreach $item (@{$self->{$vname}}) {
			push(@{$clone->{$vname}}, $item);
		}
	}

	if ($clone->user->request_vars) {
		my (@templist);
		
		while (@{$clone->{_ext_headers}}) {
			my ($car, $cdr) = $clone->shift_ext_header();
			($cdr = $clone->user->request_vars->replace($cdr));
			push (@templist, $car);
			push (@templist, $cdr);
		}
		while (@templist) {
			$item = shift(@templist);
			push (@{$clone->{_ext_headers}}, $item);
		}

		while (@{$clone->{ext_content}}) {
			$item = shift(@{$clone->{ext_content}});
			$item = $clone->user->request_vars->replace($item);
			push (@templist, $item);
		}
		while (@templist) {
			$item = shift(@templist);
			push (@{$clone->{ext_content}}, $item);
		}
	}

	return ($clone);
}

# -------------------------------------------------------------------

sub _initialize
{
    my ($self) = @_;

	# Various data
	$self->{req_content} = "";
	$self->{is_image} = Deluge::Etc::url_is_image($self->dest_url, $self->tag);
	$self->{response} = 0;  # HTTP::Response, from the response callback
	$self->{content} = "";
	$self->{content_length} = 0;
	$self->{expected_code} = 0;
	$self->{expected_size} = 0;
	$self->{user_agent_code} = 0;
	$self->{compare_fname} = "";
	$self->{secure_request}  = 0;
	$self->{wait_for_click} = 0;

	# Pipeline flags
	$self->{executed} = 0;
	$self->{in_agent} = 0;
	$self->{callback} = 0;
	
	$self->{failed} = 0;
	$self->{ignored} = 0;

	# Timers
	$self->{req_wait} = Deluge::Stopwatch->new();
	# $self->{req_elapsed} = -1;

	# Private
	$self->{_page_link_cache_urls} = [];
	$self->{_page_link_cache_tags} = [];
	$self->{_image_link_cache_urls} = [];
	$self->{_image_link_cache_tags} = [];

	$self->{_pos_vis} = [];
	$self->{_pos_invis} = [];
	$self->{_neg_vis} = [];
	$self->{_neg_invis} = [];

	$self->{_ext_headers} = [];
	$self->{ext_content} = [];

	# Extra fun stuff
	$self->{delete_cookies_when_registered} = 0;
	$self->{clear_visited_list_when_registered} = 0;

	# Log stuff
	$self->{Log} = Deluge::Log->new();

	$self->Log->new_tag($Deluge::Log::TAG_URI,  $self->dest_url);
	$self->Log->new_tag($Deluge::Log::TAG_METH, $self->method);
	$self->Log->new_tag($Deluge::Log::TAG_FROM, $self->ref_url);
	$self->Log->new_tag($Deluge::Log::TAG_HOST, $self->user->mcp->hostname);
	$self->Log->new_tag($Deluge::Log::TAG_PROC, $$);
	$self->Log->new_tag($Deluge::Log::TAG_PREQ, $self->id);
}

# -------------------------------------------------------------------

sub AUTOLOAD
{
    my ($self) = shift;
    my ($type) = ref($self) || main::confess "$self is not an object\n";
    my ($name) = $AUTOLOAD;

    $name =~ s|.*:||;

    (exists $self->{$name}) || main::confess "$name is not a method here\n";
    ($name =~ m|^_|) && main::confess "Access to method $name denied\n";

    (@_) ? (return $self->{$name} = shift) : (return $self->{$name});
}

# -------------------------------------------------------------------

sub new
{
    my ($above) = shift;
    my ($class) = ref($above) || $above;
    my ($self) = {};

    bless ($self, $class);

	# -- Public: Main --
	$self->{method} = shift;
	$self->{dest_url} = shift;
	$self->{ref_url} = shift;
	$self->{tag} = shift;
	$self->{depth} = shift;
	$self->{primary} = shift;

	$self->{user} = shift;
	$self->{id} = shift;

	$self->_initialize();
	
    return ($self);
}

# -------------------------------------------------------------------

package main;

1;
