use Carp;

use Crypt::SSLeay;
use LWP::Parallel::UserAgent;
use LWP::Parallel::Protocol::https;

# -------------------------------------------------------------------

use strict;

use Deluge::User;
use Deluge::Stopwatch;

package Deluge::Mcp;
use vars qw($AUTOLOAD);

# -------------------------------------------------------------------

sub dump_state
{
    my ($self) = @_;
	my ($user, $i);

	print STDERR "---------STATE DUMP----------\n";
	$i = $self->user_count;
	print STDERR "Num standard users: $i\n";
	
	foreach $user (@{$self->{_users}}) {
		$user->dump_state();
	}

	if ($self->proxy_user) {
		print STDERR "Proxy user exists:\n";
		$self->proxy_user->dump_state;
	}

	print STDERR "-----------------------------\n";
}

# -------------------------------------------------------------------

sub shut_down
{
    my ($self);
	my ($user);

	$self->{_agent} = undef;
	$self->{_agent_queue_timer} = undef;

	while ($user = shift (@{$self->{_users}})) { $user->shut_down; }

	$self->{log_fh} = undef;
	$self->{timer} = undef;
}

# -------------------------------------------------------------------

sub add_req_to_agent
{
    my ($self, $prereq) = @_;
	my ($req, $res);

	(! @{$self->{_agent_queue}}) && ($self->{_agent_queue_timer}->reset);
	push (@{$self->{_agent_queue}}, $prereq);
	$self->{_agent_queue_count} ++;
}

# -------------------------------------------------------------------

sub main_attack_loop
{
    my ($self) = @_;

  MAIN_LOOP:
	while (1) {
		my ($dead_count) = 0;
		my ($active_count) = 0;
		my ($min_sleep) = $self->{_agent_queue_timer}->time + 999999;  #hack
		my ($res, $user);
		
		print STDERR "------------------------------------\n";
		
	  SUB_LOOP:
		foreach $user (@{$self->{_users}}) {
			($self->endstate_check()) && return;

			($self->debug_level >= 2) &&
				(print STDERR "Processing user: " . $user->id . "\n");

			# Run the user, check its state
			$res = $user->execute_state;

			($self->debug_level >= 2) &&
				(print STDERR "STATE: $res\n");

			($res == -1) && ($dead_count ++);
			($res == 1) && ($active_count ++);

			($self->debug_level >= 5) &&
				(print STDERR "TIME: " .
				 $self->{_agent_queue_timer}->elapsed . "\n");
			
			# Completely empty the queue if it's full or timed out.
			if (($self->{_agent_queue_count} >= $self->threads_per_proc) ||
				($self->{_agent_queue_timer}->elapsed >=
				 $self->queue_max_delay)) {
				while ($self->{_agent_queue_count}) {
					$self->start_agent();
				}
			}
		}

		($self->debug_level >= 1) &&
			(print STDERR "LOOP:  Users: " . $self->user_count .
			 ", Dead: ", $dead_count,  "\n");

		# All the users are dead.  Quit.
		($dead_count >= $self->user_count) && (return 0);

		# Longest sleep we can do is...
		foreach $user (@{$self->{_users}}) {
			my ($foo) = $user->sleeping;
			# print STDERR "SLEEP " . $user->id . " has $foo\n";
			$min_sleep = Deluge::Etc::min($min_sleep, $user->sleeping);
		}

		if ($active_count) {
			if ($self->{_agent_queue_count}) {
				($self->debug_level >= 1) && (print STDERR "TEST: 1 1\n");
				$self->{_agent_queue_timer}->
					sleep($self->queue_max_delay -
						  $self->{_agent_queue_timer}->elapsed);
			} else {
				($self->debug_level >= 1) && (print STDERR "TEST: 1 0\n");
			}
		} else {
			if ($self->{_agent_queue_count}) {
				($self->debug_level >= 1) && (print STDERR "TEST: 0 1\n");
				$self->start_agent();
			} else {
				($self->debug_level >= 1) && (print STDERR "TEST: 0 0\n");
				$self->{_agent_queue_timer}->
					sleep($min_sleep - $self->timer->time);
			}
		}
	}
}

# -------------------------------------------------------------------

sub register
{
    my ($self, $prereq) = @_;
	my ($request, $entry);

	($prereq->delete_cookies_when_registered) &&
		($prereq->user->reset_cookie_jar);

	($prereq->clear_visited_list_when_registered) &&
		($prereq->user->clear_visited_list);
	
	($prereq->primary) && ($prereq->user->page_timer->reset);
	$prereq->req_wait->reset;
		
	$request = HTTP::Request->new($prereq->method, $prereq->dest_url);
	$request->push_header($Deluge::Etc::HeaderTag => $prereq->id);

	if (! $request) {
		$prereq->Log->new_tag($Deluge::Log::TAG_BREQ, "Invalid request");
		$prereq->failed(1);
		return 0;
	}

	while (1) {
		my ($car, $cdr) = $prereq->shift_ext_header();
		($car) || last;
		$request->push_header($car => "$cdr");
	}

	my ($content);
	foreach $content (@{$prereq->{ext_content}}) {
		$request->add_content($content);
	}

	$prereq->Log->new_tag($Deluge::Log::TAG_TIME, $self->timer->time);

	my ($uri) = URI->new($prereq->dest_url);
	my ($ack) = $uri->scheme;
	
	if ($ack =~ m|^https|) {
		$prereq->user->do_secure_serial($prereq, $request);
		return 0;
	}

	($self->debug_level >= 2) &&
		(print STDERR "\tRegistering: (".$prereq->user->id.") " .
		 "$prereq->{dest_url} ...\n");
	
	$entry = $self->{_agent}->register($request,
									   sub { $prereq->agent_callback(@_) });
	
	if ($entry->isa('HTTP::Response')) {
		### register returned an HTTP::Response object, so it's an error.
		($self->debug_level >= 2) && (print STDERR "failed 1.\n");
		$prereq->Log->new_tag($Deluge::Log::TAG_BREQ,
							  "Registration failed");
		$self->failed(1);
		return 0;
	} elsif ($entry->isa("LWP::Parallel::UserAgent::Entry")) {
		### register returned an LWP::Parallel::UserAgent::Entry
		### object, so the register succeeded.
		
		($self->debug_level >= 2) && (print STDERR "succeeded.\n");
		$request->remove_header(qw(User-Agent));
		($prereq->user->playback_mode) ? 
			($entry->agent($prereq->user_agent_code)) :
				($entry->agent($Deluge::Etc::AgentCode));
		
		($prereq->user->cookie_jar) &&
			($entry->cookie_jar($prereq->user->cookie_jar));
		return 1;
	} else {
		### we should never get here, if register worked correctly.
		($self->debug_level >= 2) && (print STDERR "failed 2.\n");
		$prereq->Log->new_tag($Deluge::Log::TAG_BREQ,
							  "Registration returned unknown object");
		$prereq->failed(1);
		return 0;
	}
}

# -------------------------------------------------------------------

sub start_agent
{
    my ($self) = @_;
	my ($entries, $reqcount, $prereq, $entry);
	my (@temp_queue) = ();

	$reqcount = 0;
	while (1) {
		($reqcount >= $self->threads_per_proc) && last;
		($prereq = shift(@{$self->{_agent_queue}})) || last;
		$self->{_agent_queue_count}--;
		push (@temp_queue, $prereq);
		$reqcount += $self->register($prereq);
	}

	print STDERR "Running queue of size $reqcount... ";
	$entries = $self->{_agent}->wait($self->timeout);
	print STDERR "done.\n";

  ENTRY_LOOP:
	foreach $entry (keys %$entries) {
		my ($item);
		my ($response) = $entries->{$entry}->response;
		my ($pqid) = $response->request->header($Deluge::Etc::HeaderTag);

	  PREREQ_LOOP:
		foreach $prereq (@temp_queue) {
			if ($prereq->id eq $pqid) {
				$prereq->preprocess($response);
				next ENTRY_LOOP;
			}
		}
	}

  PROCESS_LOOP:
	foreach $prereq (@temp_queue) {
		$prereq->process_response;
		($prereq->primary) && ($prereq->user->images_to_agent);
	}
	
	$self->{_agent_queue_timer}->reset;
	$self->{_agent}->initialize;
}

# -------------------------------------------------------------------

sub endstate_check
{
	my ($self) = @_;

	($self->attack_time_length) &&
		($self->timer->elapsed >= $self->attack_time_length) &&
			(return 1);

	return 0;
}

# -------------------------------------------------------------------

sub user_count
{
    my ($self) = @_;

	return ($#{@{$self->{_users}}} + 1);
}

# -------------------------------------------------------------------

sub start_proxy
{
	my ($self) = @_;

	print STDERR "Proxy server running on host " . $self->hostname .
		" on port " . $self->proxy_http_port . "\n";

	$self->proxy_user->start_proxy;
}

# -------------------------------------------------------------------

sub prep_for_attack
{
    my ($self) = @_;
	my ($def, $i);

	$self->open_log_file("w");

	foreach $def (@{$self->{_user_defs}}) {
		for ($i=0; $i<$def->instances; $i++) {
			push(@{$self->{_users}},
				 $def->make_user_from_def($self->user_count));
		}
	}

	$self->{_agent} = LWP::Parallel::UserAgent->new;

	# LWP::Agent standard stuff
	$self->{_agent}->env_proxy();
	$self->{_agent}->timeout($self->timeout);
	$self->{_agent}->from($self->owner_email);

	# LWP::Parallel::Agent specific stuff
	$self->{_agent}->in_order(0);
	$self->{_agent}->duplicates(0);
	# $self->{_agent}->redirect(1);
	$self->{_agent}->redirect(0);
	$self->{_agent}->max_req($self->threads_per_proc);

	$self->{_agent_queue_timer}->reset;
}

# -------------------------------------------------------------------

sub prep_for_record
{
    my ($self, $defname) = @_;
	my ($def, $i);

	foreach $def (@{$self->{_user_defs}}) {
		($defname eq $def->defname) || next;
		$self->proxy_user($def->make_user_from_def($self->user_count, $self));
		$self->proxy_user->prep_for_record($self);
	}

	($self->proxy_user) ||
		(main::usage("No such user [$defname] in config file"));
}

# -------------------------------------------------------------------

sub open_log_file
{
	my ($self, $mode) = @_;

	if ($mode eq "w") {
		($self->{log_fh} = new FileHandle $self->log_filename, "w") ||
			main::usage("Can't open [$self->{log_filename}] for writing");
		return;
	}
}

# -------------------------------------------------------------------

sub _check_config_info
{
	my ($self) = @_;
	
    ($self->owner_email) ||
		main::usage("Missing mandatory [owner_email] assignment");
	
    ($self->log_filename) ||
		main::usage("Missing mandatory [log_filename] assignment");

	($self->{timeout} == -1) &&
		main::usage("Missing mandatory [timeout] assignment");

	($self->{timeout} > 0) ||
		main::usage("Value for [timeout] must be > 0");

	($self->record_mode) &&
		($self->proxy_http_port == -1) &&
			main::usage("Missing mandatory [proxy_http_port] assignment");
	
	($self->threads_per_proc == -1) &&
		main::usage("Missing mandatory [threads_per_proc] assignment");

	($self->threads_per_proc > 0) ||
		main::usage("Value for [threads_per_proc] must be > 0");

	($self->hostname) || ($self->hostname(main::hostname()));
}

# -------------------------------------------------------------------

sub _read_config_file
{
	my ($self, $cfg) = @_;
	my ($line, $tag);

  LINE:
	while ($line = $cfg->get_next_line) {
		my ($car, $cdr) = $cfg->get_pair($line);

		# Switch values
		foreach $tag (qw(dump_responses
						 eval_per_url
						 allow_secure
						 verbose_logs)) {
			if ($car eq $tag) {
				$self->{$tag} = $cfg->get_switch($car, $cdr);
				next LINE;
			}
		}

		# Numerical values
		foreach $tag (qw(attack_time_length
						 debug_level
						 domain_match
						 eval_hist_time_buckets
						 eval_hist_value_buckets
						 proxy_http_port
						 queue_max_delay
						 threads_per_proc
						 timeout
						 user_ramp_time)) {
			if ($car eq $tag) {
				$self->{$tag} = $cfg->get_number($car, $cdr);
				next LINE;
			}
		}

		# String values
		foreach $tag (qw(hostname
						 log_filename
						 owner_email)) {
			if ($car eq $tag) {
				$self->{$tag} = $cdr;
				next LINE;
			}
		}

		# Lists
		foreach $tag (qw(ignore_url_regexps
						 require_url_regexps)) {
			if ($car eq $tag) {
				$self->{$tag} = [ $cfg->get_list($car) ];
				next LINE;
			}
		}

		# Hashes
		foreach $tag (qw(pos_vis_regexps
						 neg_vis_regexps
						 pos_invis_regexps
						 neg_invis_regexps)) {
			if ($car eq $tag) {
				$self->{$tag} = { $cfg->get_hash($car) };
				next LINE;
			}
		}

		# Objects
		if ($car eq "user_def") {
			my ($user) = Deluge::User->new($self, $cfg, $cdr);
			push(@{$self->{_user_defs}}, $user);
			next LINE;
		}

		$cfg->error("Unknown variable [$car]");
	}
}

# -------------------------------------------------------------------

sub DESTROY
{
    my ($self) = @;
}

# -------------------------------------------------------------------

sub _initialize
{
    my ($self, $record_mode) = @_;

	# -- Private --
	$self->{_user_defs} = [];
	$self->{_users} = [];

	$self->{_agent} = 0;
	$self->{_agent_queue} = [];
	$self->{_agent_queue_count} = 0;
	$self->{_agent_queue_timer} = Deluge::Stopwatch->new;

	# -- Public: config file --
	# Switches
	$self->{dump_responses} = 0;
	$self->{eval_per_url} = 0;
	$self->{allow_secure} = 0;
	$self->{verbose_logs} = 0;
	
	# Numerical
	$self->{attack_time_length} = 0;
	$self->{debug_level} = 0;
	$self->{domain_match} = 0;
	$self->{eval_hist_time_buckets} = 15;
	$self->{eval_hist_value_buckets} = 15;
	$self->{proxy_http_port} = -1;
	$self->{queue_max_delay} = 0;
	$self->{threads_per_proc} = -1;
	$self->{timeout} = -1;
	$self->{user_ramp_time} = 0;

	# Strings
	$self->{log_filename} = "";
	$self->{owner_email} = "";
	$self->{hostname} = "";

	# Lists
	$self->{ignore_url_regexps} = [];
	$self->{require_url_regexps} = [];

	# Hashes
	$self->{pos_vis_regexps} = {};
	$self->{neg_vis_regexps} = {};
	$self->{pos_invis_regexps} = {};
	$self->{neg_invis_regexps} = {};

	# -- Public --
	$self->{last_active_user} = 0;
	$self->{log_fh} = 0;

	$self->{timer} = Deluge::Stopwatch->new;

	$self->{record_mode} = $record_mode;

	# -- Proxy stuff --
	$self->{proxy_user} = 0;
}

# -------------------------------------------------------------------

sub AUTOLOAD
{
    my ($self) = shift;
	my ($type) = ref($self) || main::confess "$self is not an object\n";
	my ($name) = $AUTOLOAD;

	$name =~ s|.*:||;

	(exists $self->{$name}) || main::confess "$name is not a method here\n";
	($name =~ m|^_|) && main::confess "Access to method $name denied\n";

	(@_) ? (return $self->{$name} = shift) : (return $self->{$name});
}

# -------------------------------------------------------------------

sub new
{
	my ($above, $cfg, $record_mode) = @_;
	my ($class) = ref($above) || $above;
	my ($self) = {};
	my ($fname) = shift;
	
	bless ($self, $class);

	$self->_initialize($record_mode);
	$self->_read_config_file($cfg);
	$self->_check_config_info;

	return ($self);
}

# -------------------------------------------------------------------

package main;

1;
