use Carp;

# -------------------------------------------------------------------

use strict;

use Deluge::Histogram;

package Deluge::Log;
use vars qw($AUTOLOAD);

# -------------------------------------------------------------------
# External constants

# -- Generated at traversal time --
#    -- Before processing --
$Deluge::Log::TAG_URI  = 101;  # Requested URL
$Deluge::Log::TAG_METH = 102;  # Method of transmit
$Deluge::Log::TAG_OCNT = 103;  # Outgoing content
$Deluge::Log::TAG_FROM = 104;  # Refered from URL
$Deluge::Log::TAG_HOST = 105;  # Host ID
$Deluge::Log::TAG_PROC = 106;  # Process ID
$Deluge::Log::TAG_PREQ = 107;  # User/Prereq ID
$Deluge::Log::TAG_DNAM = 108;  # User_def name
$Deluge::Log::TAG_TIME = 109;  # Time when event started, for merging logs

#    -- After processing (not errors) --
$Deluge::Log::TAG_FWDP = 201;  # If forwarding occurred, list the steps
$Deluge::Log::TAG_CODE = 202;  # Response code
$Deluge::Log::TAG_GOTO = 203;  # Refers to non-image URLs
$Deluge::Log::TAG_GIMG = 204;  # Refers to image URLs
$Deluge::Log::TAG_ELAP = 205;  # Response time
$Deluge::Log::TAG_SIZE = 206;  # Response size
$Deluge::Log::TAG_PTIM = 207;  # Response time for entire page
$Deluge::Log::TAG_DTIM = 208;  # Time from "want to click" to "click"

#    -- Discovered errors --
$Deluge::Log::TAG_BREQ = 301;  # Error: Couldn't complete request
$Deluge::Log::TAG_XMME = 301;  # Error: META tag missing HTTP-EQUIV
                               #        or NAME entry
$Deluge::Log::TAG_XNRD = 302;  # Error: No response data
$Deluge::Log::TAG_XUKT = 303;  # Error: Unknown tag found
$Deluge::Log::TAG_PVVD = 304;  # Error: Positive vis validation denied
                               #        from regexp
$Deluge::Log::TAG_NVVR = 305;  # Error: Negative vis validation received
                               #        from regexp
$Deluge::Log::TAG_PIVD = 306;  # Error: Positive invis validation denied
                               #        from regexp
$Deluge::Log::TAG_NIVR = 307;  # Error: Negative invis validation received
                               #        from regexp
$Deluge::Log::TAG_CDMM = 308;  # Error: Expected code from playback
                               #        script doesn't match actual code
$Deluge::Log::TAG_LNMM = 309;  # Error: Expected length from response
                               #        doesn't match expected from playback
	
#    -- Comments --
$Deluge::Log::TAG_CMNT = 401;  # Comments

# -------------------------------------------------------------------

sub _initialize
{
	my ($self) = @_;
	
	# Log stuff
	$self->{tags} = [];

	$self->{linenum} = 0;

	$self->{event_car} = [];
	$self->{event_cdr} = [];

	$self->{fh} = 0;
}

# -------------------------------------------------------------------

sub new_tag
{
    my ($self, $car, $cdr) = @_;

	($cdr) || return;
	push (@{$self->{tags}}, "$car $cdr");
}

# -------------------------------------------------------------------

sub dump_tags
{
    my ($self) = @_;

	return (@{$self->{tags}});
}

# -------------------------------------------------------------------

sub prep_log_for_eval
{
    my ($self, $filename) = @_;

	($self->{fh} = FileHandle->new) ||
		(main::usage("Can't create new filehandle"));

	($self->{fh}->open($filename)) ||
		(main::usage("Can't open log file [$filename]"));
}


# -------------------------------------------------------------------

sub reset_log
{
    my ($self) = @_;

	$self->{fh}->close();
	$self->{linenum} = 0;
}

# -------------------------------------------------------------------

sub get_next_event
{
    my ($self) = @_;
	my ($started) = 0;
	my ($line, $fh);

	$fh = $self->{fh};
	$self->{event_car} = [];
	$self->{event_cdr} = [];
	
	while ($line = <$fh>) {
		my ($car, $cdr);
		
		$self->{linenum} ++;

		chomp($line);
		if (! $line) { ($started) ? (last) : (next); }

		$started = 1;

		($car, $cdr) = split(' ', $line, 2);
		push (@{$self->{event_car}}, $car);
		push (@{$self->{event_cdr}}, $cdr);
	}

	return ($started);
}

# -------------------------------------------------------------------

sub evaluate
{
    my ($self, $filename, $per_url, $time_buckets, $value_buckets) = @_;
	my ($item);
	my ($url_time_h, $page_time_h, $size_h, $errors_h, $user_wait_h);
	my ($url_time_over_time_h, $page_time_over_time_h);
	my ($user_wait_over_time_h);
	
	$url_time_h = Deluge::Histogram->new($value_buckets, 0);
	$page_time_h = Deluge::Histogram->new($value_buckets, 0);
	$size_h = Deluge::Histogram->new($value_buckets, 0);
	$user_wait_h = Deluge::Histogram->new($value_buckets, 0);
	$errors_h = Deluge::Histogram->new($value_buckets, 1);

	$url_time_over_time_h = Deluge::Histogram->new($time_buckets, 1);
	$page_time_over_time_h = Deluge::Histogram->new($time_buckets, 1);
	$user_wait_over_time_h = Deluge::Histogram->new($time_buckets, 1);

	my (%code_counters);
	my ($url_code_counters);

	# First pass: Find various counts, minimums and maximums, and so forth.
	$self->prep_log_for_eval($filename);

	while ($self->get_next_event) {
		my ($url, $code, $elap, $ptim, $dtim, $size, $wall);

		while ($#{@{$self->{event_car}}} > -1) {
			my ($car, $cdr);

			$car = shift(@{$self->{event_car}});
			$cdr = shift(@{$self->{event_cdr}});

			if ($car == $Deluge::Log::TAG_URI) { $url = $cdr; next; }
			if ($car == $Deluge::Log::TAG_TIME) { $wall = $cdr; next; }
			if ($car == $Deluge::Log::TAG_CODE) { $code = $cdr; next; }
			if ($car == $Deluge::Log::TAG_ELAP) { $elap = $cdr; next; }
			if ($car == $Deluge::Log::TAG_PTIM) { $ptim = $cdr; next; }
			if ($car == $Deluge::Log::TAG_DTIM) { $dtim = $cdr; next; }
			if ($car == $Deluge::Log::TAG_SIZE) { $size = $cdr; next; }
		}

		($wall && $code) || next;
		
		($size) && $size_h->set_min_max_indirectly($size);
		($ptim) && $page_time_h->set_min_max_indirectly($ptim);
		($dtim) && $user_wait_h->set_min_max_indirectly($dtim);
		($elap) && $url_time_h->set_min_max_indirectly($elap);

		$url_time_over_time_h->set_min_max_indirectly($wall);
		$page_time_over_time_h->set_min_max_indirectly($wall);
		$user_wait_over_time_h->set_min_max_indirectly($wall);
		$errors_h->set_min_max_indirectly($wall);

		(exists $code_counters{$code}) || ($code_counters{$code} = 0);
		$code_counters{$code}++;

		if ($per_url) {
			(exists $url_code_counters->{$url}) ||
				($url_code_counters->{$url}->{$code} = 0);
			$url_code_counters->{$url}->{$code} ++;
		}
	}
	
	$url_time_h->prep_for_data();
	$page_time_h->prep_for_data();
	$size_h->prep_for_data();
	$errors_h->prep_for_data();
	$user_wait_h->prep_for_data();

	$url_time_over_time_h->prep_for_data();
	$page_time_over_time_h->prep_for_data();
	$user_wait_over_time_h->prep_for_data();

	# Second pass: Find various counts, minimums and maximums, and so forth.
	$self->prep_log_for_eval($filename);

	while ($self->get_next_event) {
		my ($url, $code, $elap, $dtim, $ptim, $size, $wall);

		while ($#{@{$self->{event_car}}} > -1) {
			my ($car, $cdr);
			
			$car = shift(@{$self->{event_car}});
			$cdr = shift(@{$self->{event_cdr}});

			if ($car == $Deluge::Log::TAG_URI) { $url = $cdr; next; }
			if ($car == $Deluge::Log::TAG_TIME) { $wall = $cdr; next; }
			if ($car == $Deluge::Log::TAG_CODE) { $code = $cdr; next; }
			if ($car == $Deluge::Log::TAG_ELAP) { $elap = $cdr; next; }
			if ($car == $Deluge::Log::TAG_PTIM) { $ptim = $cdr; next; }
			if ($car == $Deluge::Log::TAG_DTIM) { $dtim = $cdr; next; }
			if ($car == $Deluge::Log::TAG_SIZE) { $size = $cdr; next; }
		}

		($wall && $code) || next;
		
		($size) && $size_h->insert_value_in_bucket(1, $size);

		if ($elap) {
			$url_time_h->insert_value_in_bucket(1, $elap);
			$url_time_over_time_h->insert_value_in_bucket($elap, $wall);
		}

		if ($ptim) {
			$page_time_h->insert_value_in_bucket(1, $ptim);
			$page_time_over_time_h->insert_value_in_bucket($ptim, $wall);
		}

		if ($dtim) {
			$user_wait_h->insert_value_in_bucket(1, $dtim);
			$user_wait_over_time_h->insert_value_in_bucket($dtim, $wall);
		}

		($code >= 400) && $errors_h->insert_value_in_bucket(1, $wall);
	}

	$url_time_h->prep_for_dump();
	$page_time_h->prep_for_dump();
	$size_h->prep_for_dump();
	$errors_h->prep_for_dump();
	$user_wait_h->prep_for_dump();

	$url_time_over_time_h->prep_for_dump();
	$page_time_over_time_h->prep_for_dump();
	$user_wait_over_time_h->prep_for_dump();
	
	$url_time_h->dump_results("Load time per URL histogram", 1);
	$url_time_over_time_h->
		dump_results("Average load time per URL over time", 3);

	$page_time_h->dump_results("Load time per page histogram", 1);
	$page_time_over_time_h->
		dump_results("Average load time per page over time", 3);

	$user_wait_h->dump_results("User wait time histogram", 1);
	$user_wait_over_time_h->dump_results("User wait time over time", 3);

	$size_h->dump_results("URL bytes histogram", 1);
	$errors_h->dump_results("Errors over time histogram", 1);
	
	print "Response code counters:\n";
	foreach $item (sort(keys(%code_counters))) {
		print "\t$item: " . $code_counters{$item} . "\n";
	}
	print "\n";

	if ($per_url) {
		print "Response code per URL counters:\n";
		foreach $item (sort(keys(%{$url_code_counters}))) {
			my ($code);
			
			print "\t$item\n";
			foreach $code (keys(%{$url_code_counters->{$item}})) {
				print "\t\t$code: " .
					$url_code_counters->{$item}->{$code} . "\n";
			}
		}
		print "\n";
	}
}

# -------------------------------------------------------------------

sub AUTOLOAD
{
    my ($self) = shift;
	my ($type) = ref($self) || main::confess "$self is not an object\n";
	my ($name) = $AUTOLOAD;

	$name =~ s|.*:||;

	(exists $self->{$name}) || main::confess "$name is not a method here\n";
	($name =~ m|^_|) && main::confess "Access to method $name denied\n";

	(@_) ? (return $self->{$name} = shift) : (return $self->{$name});
}

# -------------------------------------------------------------------

sub DESTROY
{
    my ($self) = @_;
}

# -------------------------------------------------------------------

sub new
{
    my ($above) = shift;
    my ($class) = ref($above) || $above;
    my ($self) = {};

    bless ($self, $class);

	$self->_initialize();
	
    return ($self);
}

# -------------------------------------------------------------------

package main;

1;
