/******************************************************************************
 *                        udptcpscan.h by S/ash [RtC]                         *
 *                        RtC Scan : a network mapper                         *
 *                   Scanning functions for UDP & TCP Ports                   *
 *              This is a part of the RtC Neset Project - RtC Tech            *
 ******************************************************************************/

#include "rtcscan.h"

int localhost_addr;   /* for getting our ip address */

/******************************************************************************
 * Function udptcp_scan_port - scan a port on a host                          *
 * Input  : port      : port to scan                                          *
 *          host      : struct sockaddr_in for host to scan                   *
 *          smeth     : methode for scanning                                  *
 *          scanport  : port to use for scanning                              *
 *          timeout   : time out for syn scanning (in ms)                     *
 * Output : scan_port : port state                                            *
 ******************************************************************************/
int udptcp_scan_port(long port, struct sockaddr_in *host, int smeth, long scanport, long timeout)
{
  int rc, tcpsock, rawsock;
  unsigned int fl;
  fd_set rset;
  struct timeval cur_time, beg_time, tv;
  struct sockaddr_in addr;
  char buf[3000];
  struct iphdr *ip = (struct iphdr *) buf;
  struct tcphdr *tcp = (struct tcphdr *) (buf + sizeof(struct iphdr));
  struct icmphdr *icmp = (struct icmphdr *) (buf + sizeof(struct iphdr));
  struct udphdr  *udp  = (struct udphdr *)(buf + 2*sizeof(struct iphdr) + sizeof(struct icmphdr));
  host->sin_port = htons(port);

  switch(smeth)
    {

    case METHOD_S:         /* SYN scanning */
      FD_ZERO(&rset);
      tcpsock=socket(AF_INET, SOCK_RAW, IPPROTO_TCP);
      if(setsockopt(tcpsock, IPPROTO_IP, IP_HDRINCL, (char *)&rc, sizeof(rc))<0) /* set the socket for making ourselves our ip header */
	{
	  fprintf(stderr, CANTWRITEIPHDR);
	  exit(-1);
	}
      FD_SET(tcpsock, &rset);
      /* sending a SYN packet */
      memcpy(&addr, host, sizeof(addr));
      gettimeofday(&beg_time,NULL);
      tcp_send(tcpsock,&addr,
		 localhost_addr,addr.sin_addr.s_addr,
		 scanport,port,
		 TH_SYN,
		 lrand48()+1, lrand48(),
		 512,
		 NULL,
		 0);
      tv.tv_sec = timeout / 1000; 
      tv.tv_usec = timeout * 1000; 
      /* check for reply */
      if((rc=select(tcpsock+1,&rset, NULL, NULL, &tv))>=1)
      {
	gettimeofday(&cur_time, NULL);
	while(recvfrom(tcpsock,&buf,3000,0, NULL,0)>0)
	  {
#ifdef DEBUGGING
	    printf("\nip header  :\n"); HEXDUMP(sizeof(struct iphdr), (char*)ip);
	    printf("tcp header :\n"); HEXDUMP(sizeof(struct tcphdr), (char*)tcp);
	    printf("sport : %d, dport : %d, seqnb : %08X, acknb : %08X, offset : %01X\n"
		   "flags : %02X, URG : %d, ACK : %d, PUSH : %d, RST : %d, SYN : %d, FIN : %d\n"
		   "win : %d, checksum : %d, urgentp : %d, begin of tcp header : %d\n\n",
		   ntohs(tcp->th_sport), ntohs(tcp->th_dport), tcp->th_seq, tcp->th_ack, tcp->th_off,
		   tcp->th_flags,
		   tcp->th_flags & TH_URG ? 1 : 0,  
		   tcp->th_flags & TH_ACK ? 1 : 0,  
		   tcp->th_flags & TH_PUSH ? 1 : 0,
		   tcp->th_flags & TH_RST ? 1 : 0,  
		   tcp->th_flags & TH_SYN ? 1 : 0,  
		   tcp->th_flags & TH_FIN ? 1 : 0,
		   tcp->th_win, tcp->th_sum, tcp->th_urp, ((unsigned int)tcp) - ((unsigned int)buf));
#endif
	    if ((ip->saddr == host->sin_addr.s_addr) && (ntohs(tcp->th_sport)==port) && (ntohs(tcp->th_dport)==scanport)) /* got a reply */
	      {
		
		if(tcp->th_flags & TH_RST) /* port close */
		  {
		    close (tcpsock);
		    return PORTSTATE_CLOSED;
		  }
		else if(tcp->th_flags & TH_SYN && tcp->th_flags & TH_ACK) /* port open */
		  {
		    /* sending a RST packet */
		    memcpy(&addr, host, sizeof(addr));
		    tcp_send(tcpsock,&addr,
			     localhost_addr,addr.sin_addr.s_addr,
			     scanport,port,
			     TH_RST,
			     tcp->th_seq+1, lrand48(),
			     512,
			     NULL,
			     0);
		    /* end */
		    close (tcpsock);
		    return PORTSTATE_OPENED;
		  }
		else /* port error */
		  {
		    close (tcpsock);
		    return PORTSTATE_ERROR;
		  }
	      }
	    gettimeofday(&cur_time, NULL);
	    if(timeout < DIFFTIME(beg_time, cur_time)) break;
	    tv.tv_sec = (timeout - DIFFTIME(beg_time, cur_time)) / 1000;
	    tv.tv_usec = (timeout - DIFFTIME(beg_time, cur_time)) * 1000;
	    if((rc=select(tcpsock+1,&rset, NULL, NULL, &tv))<1) break;
	  }
      }
      close(tcpsock);
      if(rc>=0) return PORTSTATE_TIMEOUT; /* time out */
      else return PORTSTATE_ERROR;     /* error */

    case METHOD_T:         /* complete tcp connexion */
      tcpsock=socket(AF_INET, SOCK_STREAM, 0);
      rc = connect( tcpsock, (struct sockaddr*)host, sizeof(*host));
      close(tcpsock);
      if(rc < 0) return PORTSTATE_CLOSED;
      else return PORTSTATE_OPENED;

    case METHOD_F:         /* stealth scanning (FIN) */
      FD_ZERO(&rset);
      tcpsock=socket(AF_INET, SOCK_RAW, IPPROTO_TCP);
      if(setsockopt(tcpsock, IPPROTO_IP, IP_HDRINCL, (char *)&rc, sizeof(rc))<0) /* set the socket for making ourselves our ip header */
	{
	  fprintf(stderr, CANTWRITEIPHDR);
	  exit(-1);
	}
      FD_SET(tcpsock, &rset);
      /* sending a FIN packet */
      memcpy(&addr, host, sizeof(addr));
      gettimeofday(&beg_time,NULL);
      tcp_send(tcpsock,&addr,
		 localhost_addr,addr.sin_addr.s_addr,
		 scanport,port,
		 TH_FIN,
		 lrand48()+1, lrand48(),
		 512,
		 NULL,
		 0);
      tv.tv_sec = timeout / 1000; 
      tv.tv_usec = timeout * 1000; 
      /* check for reply */
      if((rc=select(tcpsock+1,&rset, NULL, NULL, &tv))>=1)
      {
	gettimeofday(&cur_time, NULL);
	while(recvfrom(tcpsock,&buf,3000,0, NULL,0)>0)
	  {
#ifdef DEBUGGING
	    printf("\nip header  :\n"); HEXDUMP(sizeof(struct iphdr), (char*)ip);
	    printf("tcp header :\n"); HEXDUMP(sizeof(struct tcphdr), (char*)tcp);
	    printf("sport : %d, dport : %d, seqnb : %08X, acknb : %08X, offset : %01X\n"
		   "flags : %02X, URG : %d, ACK : %d, PUSH : %d, RST : %d, SYN : %d, FIN : %d\n"
		   "win : %d, checksum : %d, urgentp : %d, begin of tcp header : %d\n\n",
		   ntohs(tcp->th_sport), ntohs(tcp->th_dport), tcp->th_seq, tcp->th_ack, tcp->th_off,
		   tcp->th_flags,
		   tcp->th_flags & TH_URG ? 1 : 0,  
		   tcp->th_flags & TH_ACK ? 1 : 0,  
		   tcp->th_flags & TH_PUSH ? 1 : 0,
		   tcp->th_flags & TH_RST ? 1 : 0,  
		   tcp->th_flags & TH_SYN ? 1 : 0,  
		   tcp->th_flags & TH_FIN ? 1 : 0,
		   tcp->th_win, tcp->th_sum, tcp->th_urp, ((unsigned int)tcp) - ((unsigned int)buf));
#endif
	    if ((ip->saddr == host->sin_addr.s_addr) && (ntohs(tcp->th_sport)==port) && (ntohs(tcp->th_dport)==scanport)) /* got a reply */
	      {
		close (tcpsock);
		if(tcp->th_flags & TH_RST) return PORTSTATE_CLOSED;         /* port closed */
		else return PORTSTATE_OPENED; /* port opened */
	      }
	    gettimeofday(&cur_time, NULL);
	    if(timeout < DIFFTIME(beg_time, cur_time)) break;
	    tv.tv_sec = (timeout - DIFFTIME(beg_time, cur_time)) / 1000;
	    tv.tv_usec = (timeout - DIFFTIME(beg_time, cur_time)) * 1000;
	    if((rc=select(tcpsock+1,&rset, NULL, NULL, &tv))<1) break;
	  }
      }
      close(tcpsock);
      if(rc>=0) return PORTSTATE_OPENED; /* time out = opened */
      else return PORTSTATE_CLOSED;      /* error */

    case METHOD_A:         /* stealth scanning (ACK) */
      FD_ZERO(&rset);
      tcpsock=socket(AF_INET, SOCK_RAW, IPPROTO_TCP);
      if(setsockopt(tcpsock, IPPROTO_IP, IP_HDRINCL, (char *)&rc, sizeof(rc))<0) /* set the socket for making ourselves our ip header */
	{
	  fprintf(stderr, CANTWRITEIPHDR);
	  exit(-1);
	}
      FD_SET(tcpsock, &rset);
      /* sending a ACK packet */
      memcpy(&addr, host, sizeof(addr));
      gettimeofday(&beg_time,NULL);
      tcp_send(tcpsock,&addr,
		 localhost_addr,addr.sin_addr.s_addr,
		 scanport,port,
		 TH_ACK,
		 lrand48()+1, lrand48(),
		 512,
		 NULL,
		 0);
      tv.tv_sec = timeout / 1000; 
      tv.tv_usec = timeout * 1000; 
      /* check for reply */
      if((rc=select(tcpsock+1,&rset, NULL, NULL, &tv))>=1)
      {
	gettimeofday(&cur_time, NULL);
	while(recvfrom(tcpsock,&buf,3000,0, NULL,0)>0)
	  {
#ifdef DEBUGGING
	    printf("\nip header  :\n"); HEXDUMP(sizeof(struct iphdr), (char*)ip);
	    printf("tcp header :\n"); HEXDUMP(sizeof(struct tcphdr), (char*)tcp);
	    printf("sport : %d, dport : %d, seqnb : %08X, acknb : %08X, offset : %01X\n"
		   "flags : %02X, URG : %d, ACK : %d, PUSH : %d, RST : %d, SYN : %d, FIN : %d\n"
		   "win : %d, checksum : %d, urgentp : %d, begin of tcp header : %d\n\n",
		   ntohs(tcp->th_sport), ntohs(tcp->th_dport), tcp->th_seq, tcp->th_ack, tcp->th_off,
		   tcp->th_flags,
		   tcp->th_flags & TH_URG ? 1 : 0,  
		   tcp->th_flags & TH_ACK ? 1 : 0,  
		   tcp->th_flags & TH_PUSH ? 1 : 0,
		   tcp->th_flags & TH_RST ? 1 : 0,  
		   tcp->th_flags & TH_SYN ? 1 : 0,  
		   tcp->th_flags & TH_FIN ? 1 : 0,
		   tcp->th_win, tcp->th_sum, tcp->th_urp, ((unsigned int)tcp) - ((unsigned int)buf));
#endif
	    if ((ip->saddr == host->sin_addr.s_addr) && (ntohs(tcp->th_sport)==port) && (ntohs(tcp->th_dport)==scanport) && (tcp->th_flags & TH_RST)) 
	      /* got a reply */
	      {
		close(tcpsock);
		return (ip->ttl < LOW_TTL) || (tcp->th_win > 0) ? PORTSTATE_OPENED : PORTSTATE_CLOSED;
	      }
	    gettimeofday(&cur_time, NULL);
	    if(timeout < DIFFTIME(beg_time, cur_time)) break;
	    tv.tv_sec = (timeout - DIFFTIME(beg_time, cur_time)) / 1000;
	    tv.tv_usec = (timeout - DIFFTIME(beg_time, cur_time)) * 1000;
	    if((rc=select(tcpsock+1,&rset, NULL, NULL, &tv))<1) break;
	  }
      }
      close(tcpsock);
      if(rc>=0) return PORTSTATE_TIMEOUT; /* time out */
      else return PORTSTATE_ERROR;     /* error */

    case METHOD_U: /* udp scanning */
      rawsock=socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
      tcpsock=socket(AF_INET, SOCK_RAW, IPPROTO_UDP); /* in fact, it's a udp socket here ;-) */
      if(setsockopt(tcpsock, IPPROTO_IP, IP_HDRINCL, (char *)&rc, sizeof(rc))<0) /* set the socket for making ourselves our ip header */
	{
	  fprintf(stderr, CANTWRITEIPHDR);
	  exit(-1);
	}
      /* sending a UDP packet */
      memcpy(&addr, host, sizeof(addr));
      gettimeofday(&beg_time,NULL);
      udp_send(tcpsock,&addr, localhost_addr, addr.sin_addr.s_addr,
	       scanport, port, NULL, 0);
      tv.tv_sec = timeout / 1000; 
      tv.tv_usec = timeout * 1000; 
      /* check for reply */
      FD_ZERO(&rset);
      FD_SET(rawsock, &rset);
      if((rc=select(rawsock+1,&rset, NULL, NULL, &tv))>=1)
      {
	gettimeofday(&cur_time, NULL);
	while((rc=read(rawsock,buf,3000))>0)
	  {
#ifdef DEBUGGING
	    printf("\nip header  :\n"); HEXDUMP(sizeof(struct iphdr), (char*)ip);
	    printf("icmp header :\n"); HEXDUMP(sizeof(struct icmphdr), (char*)icmp);
#endif
	    if ((ip->saddr == host->sin_addr.s_addr) && (ip->protocol==IPPROTO_ICMP) && (icmp->type==ICMP_PORT_UNREACH) && (icmp->code==ICMP_PORT_UNREACH) && (ntohs(udp->uh_dport)==port)) /* got a reply */
	      {
		close (tcpsock); close (rawsock);
		return PORTSTATE_CLOSED;
	      }
	    gettimeofday(&cur_time, NULL);
	    if(timeout < DIFFTIME(beg_time, cur_time)) break;
	    tv.tv_sec = (timeout - DIFFTIME(beg_time, cur_time)) / 1000;
	    tv.tv_usec = (timeout - DIFFTIME(beg_time, cur_time)) * 1000;
	    if((rc=select(rawsock+1,&rset, NULL, NULL, &tv))<1) break;
	  }
      }
      close(tcpsock);
      close(rawsock);
      if(rc>=0) return PORTSTATE_OPENED; /* time out = opened */
      else return PORTSTATE_ERROR;     /* error */

    case METHOD_u: /* udp scanning (lame) */
      tcpsock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
      sendto(tcpsock, NULL, 0, 0, (struct sockaddr*)host, sizeof(*host));
      fcntl(tcpsock, F_SETFL, fcntl(tcpsock, F_GETFL) | O_NONBLOCK);
      fl = sizeof(*host);
      while(1)
	{
	  memcpy(&addr, host, fl);
	  rc=recvfrom(tcpsock, buf, 3000, 0, (struct sockaddr*)&addr, &fl);
	  if(addr.sin_addr.s_addr == host->sin_addr.s_addr)
	    {
	      close(tcpsock);
	      return (rc==-1) ? ((errno==ECONNREFUSED) ? PORTSTATE_CLOSED : PORTSTATE_OPENED) : PORTSTATE_OPENED;
	    }
	}
    default:
      return PORTSTATE_NO;
    }
}

/******************************************************************************
 * Function udptcp_fscan_port - scan a port on a host                         *
 * Input  : port      : port to scan                                          *
 *          host      : struct sockaddr_in for host to scan                   *
 *          smeth     : methode for scanning                                  *
 *          scanport  : port to use for scanning                              *
 ******************************************************************************/
void udptcp_fscan_port(long port, struct sockaddr_in *host, int smeth, long scanport)
{
  int sock, rc;
  struct sockaddr_in addr;
  host->sin_port = htons(port);

  switch(smeth)
    {

    case METHOD_S:         /* SYN scanning */
      sock=socket(AF_INET, SOCK_RAW, IPPROTO_TCP);
      if(setsockopt(sock, IPPROTO_IP, IP_HDRINCL, (char *)&rc, sizeof(rc))<0) /* set the socket for making ourselves our ip header */
	{
	  fprintf(stderr, CANTWRITEIPHDR);
	  exit(-1);
	}
      /* sending a SYN packet */
      memcpy(&addr, host, sizeof(addr));
      tcp_send(sock,&addr,
		 localhost_addr,addr.sin_addr.s_addr,
		 scanport,port,
		 TH_SYN,
		 lrand48()+1, lrand48(),
		 512,
		 NULL,
		 0);
      close(sock);
      break;
    case METHOD_F:         /* stealth scanning (FIN) */
      sock=socket(AF_INET, SOCK_RAW, IPPROTO_TCP);
      if(setsockopt(sock, IPPROTO_IP, IP_HDRINCL, (char *)&rc, sizeof(rc))<0) /* set the socket for making ourselves our ip header */
	{
	  fprintf(stderr, CANTWRITEIPHDR);
	  exit(-1);
	}
      /* sending a FIN packet */
      memcpy(&addr, host, sizeof(addr));
      tcp_send(sock,&addr,
		 localhost_addr,addr.sin_addr.s_addr,
		 scanport,port,
		 TH_FIN,
		 lrand48()+1, lrand48(),
		 512,
		 NULL,
		 0);
      close(sock);
      break;
    case METHOD_A:         /* stealth scanning (ACK) */
      sock=socket(AF_INET, SOCK_RAW, IPPROTO_TCP);
      if(setsockopt(sock, IPPROTO_IP, IP_HDRINCL, (char *)&rc, sizeof(rc))<0) /* set the socket for making ourselves our ip header */
	{
	  fprintf(stderr, CANTWRITEIPHDR);
	  exit(-1);
	}
      /* sending a ACK packet */
      memcpy(&addr, host, sizeof(addr));
      tcp_send(sock,&addr,
		 localhost_addr,addr.sin_addr.s_addr,
		 scanport,port,
		 TH_ACK,
		 lrand48()+1, lrand48(),
		 512,
		 NULL,
		 0);
      close(sock);
      break;

    case METHOD_U: /* udp scanning */
      sock=socket(AF_INET, SOCK_RAW, IPPROTO_UDP);
      if(setsockopt(sock, IPPROTO_IP, IP_HDRINCL, (char *)&rc, sizeof(rc))<0) /* set the socket for making ourselves our ip header */
	{
	  fprintf(stderr, CANTWRITEIPHDR);
	  exit(-1);
	}
      /* sending a UDP packet */
      memcpy(&addr, host, sizeof(addr));
      udp_send(sock,&addr, localhost_addr, addr.sin_addr.s_addr,
	       scanport, port, NULL, 0);
      close(sock);
      break;
    }
}

/******************************************************************************
 * Function udptcp_read_ans_port - read the answer of a scan                  *
 * Input  : sock      : socket for read                                       *
 *          host      : struct sockaddr_in for host to scan                   *
 *          smeth     : methode for scanning                                  *
 * Output : portv     : vector for the list of port states                    *
 *          nb_port   : nb_port decremented if an answer was got              *
 ******************************************************************************/
void udptcp_read_ans_port(int sock, struct sockaddr_in *host, int smeth,
			  struct port_result *portv, int *nb_port)
{
  int rc, port, scanport;
  char buf[3000];
  int rsock;
  struct sockaddr_in addr;

  struct iphdr *ip = (struct iphdr *) buf;
  struct tcphdr *tcp = (struct tcphdr *) (buf + sizeof(struct iphdr));
  struct icmphdr *icmp = (struct icmphdr *) (buf + sizeof(struct iphdr));
  struct udphdr *udp;

  if((rc=recv(sock, &buf, 3000,0))>0)
    {
#ifdef DEBUGGING
      printf("Received packet :\n"); HEXDUMP(rc, buf);
#endif
      switch(smeth)
	{
	case METHOD_S: /* SYN scanning */
#ifdef DEBUGGING
	    printf("sport : %d, dport : %d, seqnb : %08X, acknb : %08X, offset : %01X\n"
		   "flags : %02X, URG : %d, ACK : %d, PUSH : %d, RST : %d, SYN : %d, FIN : %d\n"
		   "win : %d, checksum : %d, urgentp : %d, begin of tcp header : %d\n\n",
		   ntohs(tcp->th_sport), ntohs(tcp->th_dport), tcp->th_seq, tcp->th_ack, tcp->th_off,
		   tcp->th_flags,
		   tcp->th_flags & TH_URG ? 1 : 0,  
		   tcp->th_flags & TH_ACK ? 1 : 0,  
		   tcp->th_flags & TH_PUSH ? 1 : 0,
		   tcp->th_flags & TH_RST ? 1 : 0,  
		   tcp->th_flags & TH_SYN ? 1 : 0,  
		   tcp->th_flags & TH_FIN ? 1 : 0,
		   tcp->th_win, tcp->th_sum, tcp->th_urp, ((unsigned int)tcp) - ((unsigned int)buf));
#endif
	  if (ip->saddr == host->sin_addr.s_addr) /* got a reply */
	    {
	      port = ntohs(tcp->th_sport);
	      scanport = portv[port].scanport;
	      if(ntohs(tcp->th_dport)==scanport)
		{
		  if(portv[port].state != PORTSTATE_NO) (*nb_port)--;
		  portv[port].state = (tcp->th_flags & TH_RST) ? PORTSTATE_CLOSED :
		    ((tcp->th_flags & TH_SYN && tcp->th_flags & TH_ACK) ? PORTSTATE_OPENED : PORTSTATE_ERROR);
		  if(portv[port].state == PORTSTATE_OPENED) /* sending a RST packet */
		    {
		      rsock=socket(AF_INET, SOCK_RAW, IPPROTO_TCP);
		      if(setsockopt(rsock, IPPROTO_IP, IP_HDRINCL, (char *)&rc, sizeof(rc))<0) /* set the socket for making ourselves our ip header */
			{
			  fprintf(stderr, CANTWRITEIPHDR);
			  exit(-1);
			}
		      /* sending a RST packet */
		      host->sin_port = htons(port);
		      memcpy(&addr, host, sizeof(addr));
		      tcp_send(rsock,&addr,
			       localhost_addr,addr.sin_addr.s_addr,
			       scanport,port,
			       TH_RST,
			       tcp->th_seq+1, lrand48(),
			       512,
			       NULL,
			       0);
		      close(rsock);
		    }
		}
	    }
	  break;

	case METHOD_F: /* stealth scanning (FIN) */
#ifdef DEBUGGING
	    printf("sport : %d, dport : %d, seqnb : %08X, acknb : %08X, offset : %01X\n"
		   "flags : %02X, URG : %d, ACK : %d, PUSH : %d, RST : %d, SYN : %d, FIN : %d\n"
		   "win : %d, checksum : %d, urgentp : %d, begin of tcp header : %d\n\n",
		   ntohs(tcp->th_sport), ntohs(tcp->th_dport), tcp->th_seq, tcp->th_ack, tcp->th_off,
		   tcp->th_flags,
		   tcp->th_flags & TH_URG ? 1 : 0,  
		   tcp->th_flags & TH_ACK ? 1 : 0,  
		   tcp->th_flags & TH_PUSH ? 1 : 0,
		   tcp->th_flags & TH_RST ? 1 : 0,  
		   tcp->th_flags & TH_SYN ? 1 : 0,  
		   tcp->th_flags & TH_FIN ? 1 : 0,
		   tcp->th_win, tcp->th_sum, tcp->th_urp, ((unsigned int)tcp) - ((unsigned int)buf));
#endif
	  if (ip->saddr == host->sin_addr.s_addr) /* got a reply */
	    {
	      port = ntohs(tcp->th_sport);
	      if(ntohs(tcp->th_dport)==portv[port].scanport)
		{
		  portv[port].state = (tcp->th_flags & TH_RST) ? PORTSTATE_CLOSED : PORTSTATE_OPENED;
		  if(portv[port].state != PORTSTATE_NO) (*nb_port)--;
		}
	    }
	  break;

	case METHOD_A: /* stealth scanning (ACK) */
#ifdef DEBUGGING
	    printf("sport : %d, dport : %d, seqnb : %08X, acknb : %08X, offset : %01X\n"
		   "flags : %02X, URG : %d, ACK : %d, PUSH : %d, RST : %d, SYN : %d, FIN : %d\n"
		   "win : %d, checksum : %d, urgentp : %d, begin of tcp header : %d\n\n",
		   ntohs(tcp->th_sport), ntohs(tcp->th_dport), tcp->th_seq, tcp->th_ack, tcp->th_off,
		   tcp->th_flags,
		   tcp->th_flags & TH_URG ? 1 : 0,  
		   tcp->th_flags & TH_ACK ? 1 : 0,  
		   tcp->th_flags & TH_PUSH ? 1 : 0,
		   tcp->th_flags & TH_RST ? 1 : 0,  
		   tcp->th_flags & TH_SYN ? 1 : 0,  
		   tcp->th_flags & TH_FIN ? 1 : 0,
		   tcp->th_win, tcp->th_sum, tcp->th_urp, ((unsigned int)tcp) - ((unsigned int)buf));
#endif
	  if ((ip->saddr == host->sin_addr.s_addr) && (tcp->th_flags & TH_RST)) /* got a reply */
	    {
	      port = ntohs(tcp->th_sport);
	      if(ntohs(tcp->th_dport)==portv[port].scanport)
		{
		  if(portv[port].state != PORTSTATE_NO) (*nb_port)--;
		  portv[port].state = (ip->ttl < LOW_TTL) || (tcp->th_win > 0) ? PORTSTATE_OPENED : PORTSTATE_CLOSED;
		}
	    }
	  break;

	case METHOD_U: /* udp scanning */
	  if ((ip->saddr == host->sin_addr.s_addr) && (ip->protocol==IPPROTO_ICMP) && (icmp->type==ICMP_PORT_UNREACH) && (icmp->code==ICMP_PORT_UNREACH)) /* got a reply */
	    {
	      udp = (struct udphdr*)(buf + (sizeof(struct iphdr)*2) + sizeof(struct icmphdr));
	      port = ntohs(udp->uh_dport);
	      if(portv[port].scanport==ntohs(udp->uh_sport))
		{
		  if(portv[port].state != PORTSTATE_NO) (*nb_port)--;
		  portv[port].state = PORTSTATE_CLOSED;
		}
	    }
	  break;
	}
    }
}

/******************************************************************************
 * Function get_localhost : get localhost ip                                  *
 * Input  : adr       : struct addr for host to scan                          *
 *          hname     : name of localhost                                     *
 *          timeout   : time out for ping                                     *
 *          isupuse   : allowed to use isup ?                                 *
 * output : return    : error code :                                          *
 *                -1 : localhost not found                                    *
 *                 1 : host down                                              *
 *                 0 : host up (no error)                                     *
 ******************************************************************************/
int get_localhost(struct in_addr adr, char *hname, long timeout, int isupuse)
{
  int sock;
  struct sockaddr_in addr;
  char *myname;
  struct in_addr tmp;

  addr.sin_family = AF_INET;
  addr.sin_addr   = adr;

  if(isupuse)
    {
      if(!isup(addr.sin_addr, &tmp, timeout))
	return 1;
    }
  else tmp.s_addr = INADDR_NONE;

  if(!hname)
    {
      if(tmp.s_addr==INADDR_NONE)
	{
	  myname = (char*)malloc(1000);
	  if (gethostname(myname, 1000))
	    return -1;
	  if ((localhost_addr = (tmp = resolve(myname)).s_addr)==INADDR_NONE)
	    return -1;
	  free(myname);
	}
      else localhost_addr = tmp.s_addr;
    }
  else
    {
      myname = (char*)malloc(strlen(hname)+1);
      strcpy(myname, hname);
      if ((localhost_addr = (tmp = resolve(myname)).s_addr)==INADDR_NONE)
	return -1;
    }
  return 0;
}

/******************************************************************************
 * Function udptcp_scan_ports - scan ports in a list on a host                *
 * Input  : plf      : list of ports to scan                                  *
 *          host     : struct sockaddr_in for host to scan                    *
 *          smeth    : scanning methode                                       *
 *          fastscan : run fast methods ?                                     *
 *          scanport : port to use for scanning                               *
 *          smaxport : max port to use for scanning (0 if not at random)      *
 *          timeout  : time out for syn scanning (in ms)                      *
 *          delay    : delay between two scan                                 *
 * Output : portv    : port vector result                                     *
 *          return   : 0 if success, -1 on port file error                    *
 ******************************************************************************/
#define RAND_SCANPORT(min, max) (max ? ((rand()%(max-min+1)) + min) : min)
int udptcp_scan_ports(struct port_list_file *plf, struct sockaddr_in *host,
		 int smeth, int fastscan, long scanport, long smaxport, long timeout,
		 long delay, struct port_result *portv)
{
  long p;
  char *buf = malloc(20000);
  int sock, nb_port = 0;
  struct timeval cur_time, beg_time;
  int fscan;
  fscan = fastscan;

  /* initializing port vector */
  for(p=0; p<SIZEOF_PORTV; p++) portv[p].state = PORTSTATE_NO;

  /* initializing sockets     */
  if(fastscan && (smeth!=METHOD_u) && (smeth!=METHOD_T))
    {
      sock = socket(AF_INET, SOCK_RAW, smeth==METHOD_U ? IPPROTO_ICMP : IPPROTO_TCP);
      fcntl(sock, F_SETFL, fcntl(sock, F_GETFL) | O_NONBLOCK);
    }
  else fscan = 0;

  while((p=get_next_port(plf, buf)))
    {
      if(p<0)
	{
	  free(buf);
	  return -1;
	}
      portv[p].service = (char*)malloc(strlen(buf)+1);
      strcpy(portv[p].service, buf);
      nb_port++;
      host->sin_family = AF_INET; /* this fucking thing is very important !! */
      portv[p].scanport = RAND_SCANPORT(scanport, smaxport);  /* random the port */
      if(fscan)
	{
	  if((smeth==METHOD_U) || (smeth==METHOD_F))
	    portv[p].state = PORTSTATE_OPENED;
	  else
	    portv[p].state = PORTSTATE_TIMEOUT;
	  udptcp_fscan_port(p, host, smeth, portv[p].scanport);
	  udptcp_read_ans_port(sock, host, smeth, portv, &nb_port);
	}
      else portv[p].state = udptcp_scan_port(p, host, smeth, portv[p].scanport, timeout);
      /* sleep for the delay */
      usleep(delay * 1000);
    }
  if(fscan)
    {
      /* check during the time out */
      gettimeofday(&beg_time, NULL);
      gettimeofday(&cur_time, NULL);
      while((DIFFTIME(beg_time, cur_time) < timeout) && (nb_port>0))
	{
	  udptcp_read_ans_port(sock, host, smeth, portv, &nb_port);
	  gettimeofday(&cur_time, NULL);
	}
    }
  free(buf);
  if(fscan) close(sock);
  return 0;
}
