/******************************************************************************
 *                         portfile.h by S/ash [RtC]                          *
 *                        RtC Scan : a network mapper                         *
 *                 contain functions for reading port list file               *
 *              This is a part of the RtC Neset Project - RtC Tech            *
 ******************************************************************************/
#include "rtcscan.h"

/******************************************************************************
 * Function open_port_list_file - open a port list file                       *
 * Input  : name                : file name                                   *
 * Output : plf                 : port_list_file struct                       *
 *          open_port_list_file : error                                       *
 *          proto               : protocol (1 : UDP, 0 : TCP)                 *
 *          useservice          : use /etc/services to get port               *
 ******************************************************************************/
int open_port_list_file(char *name, struct port_list_file *plf, int proto, int useservice)
{
  if(!plf) return -1;
  plf->readservices = useservice;
  plf->proto = proto;

  if(useservice)
    {
      plf->file = NULL;
      plf->curline = NULL;
      plf->curport = 0; plf->maxport = 0;
    }
  else
    {
      plf->file = fopen(name, "r");
      if(!(plf->file)) return errno;
      plf->curline = NULL;
      plf->curport = 0; plf->maxport = 0;
    }
  return 0;
}

/******************************************************************************
 * Function rst_port_file - reset a port list file                            *
 * Input  : plf : port list file                                              *
 * Output : rst_port_file : return of fseek                                   *
 ******************************************************************************/
int rst_port_file(struct port_list_file *plf)
{
  plf->curline = NULL;
  plf->curport = 0; plf->maxport = 0;
  if(plf->readservices) { setservent(1); return 0; }
  else return fseek(plf->file, 0L, SEEK_SET);
}

/******************************************************************************
 * Function get_next_port - get next port in a port list file                 *
 * Input  : plf     : port list file                                          *
 * Output : comment : commentary for port                                     *
 *          port    : port number (0 if eof, -1 if error)                     *
 ******************************************************************************/
long get_next_port(struct port_list_file *plf, char *comment)
{
  int i; char *tmp; long port;
  struct servent *service;

  if(!plf) return -1;
  if(plf->readservices)
    {
      service = getservent();
      i = 1;
      while(service && i)
	{
	  if(strcmp(service->s_proto, plf->proto ? "udp" : "tcp"))
	    service = getservent();
	  else i = 0;
	}
      if(!service)
	{
	  endservent();
	  return 0;
	}
      strcpy(comment, service->s_name);
      return ntohs(service->s_port);
    }
  if(plf->curport<0) plf->curport=0;
  if(!plf->maxport)
    {
      if(plf->file==NULL) return 0;
      plf->curline = (void*)(malloc(1000));
      if(fgets(plf->curline, 1000, plf->file))
	{
	  /* removing the CR */
	  if(plf->curline[strlen(plf->curline)-1]=='\n') plf->curline[strlen(plf->curline)-1] = 0;
	  for(i=0; (((plf->curline[i]==' ') || (plf->curline[i]=='\t')) && plf->curline[i]); i++); /* jumping whitespaces */ 
	  if(plf->curline[i]=='#' || !(plf->curline[i])) /* commentary or blank line */
	    {
	      free(plf->curline);
	      return get_next_port(plf, comment); /* get the next line */
	    }
	  else if((plf->curline[i]<='9') && (plf->curline[i]>='0')) /* test the file format */
	    {
	      port = 0;
	      for(; (plf->curline[i] != 0) && (plf->curline[i]>='0') && (plf->curline[i]<='9'); i++)
		{
		  port *= 10;
		  port += plf->curline[i] - '0';
		}
	      if(plf->curline[i] != '-')            /* if only one port on line */
		{
		  /* jumping whitespaces */
		  while(((plf->curline[i]==' ') || (plf->curline[i]=='\t')) && plf->curline[i]) i++;
		  strcpy(comment, plf->curline+i);  /* copy comment for current port */
		  free(plf->curline);
		}
	      else                                 /* if a list of port on line */
		{
		  plf->curport = port;
		  for(i++; (plf->curline[i] != 0) && (plf->curline[i]>='0') && (plf->curline[i]<='9'); i++)
		    {
		      plf->maxport *= 10;
		      plf->maxport += plf->curline[i] - '0';
		    }
		  /* jumping whitespaces */
		  while(((plf->curline[i]==' ') || (plf->curline[i]=='\t')) && plf->curline[i]) i++;
		  /* keep only commentary */
		  tmp = (char*)malloc(strlen(plf->curline + i)+1);
		  strcpy(tmp, plf->curline + i);
		  free(plf->curline);
		  plf->curline = tmp;
		  /* copy commentary */
		  if(plf->curline) strcpy(comment, plf->curline);
		  else /* try to get service using getservbyport */
		    {
		      service = getservbyport(htons(port), plf->proto ? "udp" : "tcp");
		      if(service)
			strcpy(comment, service->s_name);
		      else
			{
			  /*
			   * the following line is just asking for echo port (because it must exist
			   * on every system) just for prevent from disabling opened sosckets.
			   */
			  getservbyport(htons(7), plf->proto ? "udp" : "tcp");
			  strcpy(comment, "");
			}
		    }
		}
	    }
	  else 
	    {
	      free(plf->curline);
	      return -1;
	    }
	}
      else
	{
	  free(plf->curline);
	  return 0; /* eof */
	}
    }
  else
    {
      port = ++(plf->curport);
      if(plf->curline) strcpy(comment, plf->curline);
      else /* try to get service using getservbyport */
	{
	  service = getservbyport(htons(port), plf->proto ? "udp" : "tcp");
	  if(service)
	    strcpy(comment, service->s_name);
	  else
	    {
	      /*
	       * the following line is just asking for echo port (because it must exist
	       * on every system) just for prevent from disabling opened sosckets.
	       */
	      getservbyport(htons(7), plf->proto ? "udp" : "tcp");
	      strcpy(comment, "");
	    }
	}
      if(port==plf->maxport || port>=SIZEOF_PORTV)
	{
	  plf->maxport = 0;
	  plf->curport = 0;
	  free(plf->curline);
	}
    }
  return port;
}
