/******************************************************************************
 *                          ipraw.c by S/ash [RtC]                            *
 *                        RtC Scan : a network mapper                         *
 *             Contain functions for creating tcp/ip packets                  *
 *              This is a part of the RtC Neset Project - RtC Tech            *
 ******************************************************************************/
#include "rtcscan.h"

struct timeval last_beg_time;

/* tcp/ip raw functions */

/******************************************************************************
 * Function HEXDUMP for debugging                                             *
 * No comment                                                                 *
 ******************************************************************************/
#ifdef DEBUGGING
void HEXDUMP(unsigned len, unsigned char *buf)
{
  unsigned i;
  for(i=0; i<len; i++)
    printf("%02X%c", *(buf+i), ((i+1)%20) ? ' ' : '\n');
}
#endif

/******************************************************************************
 * in_cksum --                                                                *
 *  Checksum routine for Internet Protocol family headers (C Version)         *
 ******************************************************************************/
unsigned short in_cksum(addr, len)
    u_short *addr;
    int len;
{
    register int nleft = len;
    register u_short *w = addr;
    register int sum = 0;
    u_short answer = 0;
 
    /*
     * Our algorithm is simple, using a 32 bit accumulator (sum), we add
     * sequential 16 bit words to it, and at the end, fold back all the
     * carry bits from the top 16 bits into the lower 16 bits.
     */
    while (nleft > 1)  {
        sum += *w++;
        nleft -= 2;
    }
 
    /* mop up an odd byte, if necessary */
    if (nleft == 1) {
        *(u_char *)(&answer) = *(u_char *)w ;
        sum += answer;
    }
 
    /* add back carry outs from top 16 bits to low 16 bits */
    sum = (sum >> 16) + (sum & 0xffff);   /* add hi 16 to low 16 */
    sum += (sum >> 16);                   /* add carry */
    answer = ~sum;                        /* truncate to 16 bits */
    return(answer);
}


/******************************************************************************
 * Function tcp_send                                                          *
 * send a customized TCP Header                                               *
 * Input  : socket : socket to write                                          *
 *          address : destination address                                     *
 *          s_addr, t_addr : source and destination address for checksum      *
 *          s_port, t_port : source and dest port                             *
 *          tcpflags : flags of tcp header                                    *
 *          seq : sequence number                                             *
 *          ack : ack number                                                  *
 *          win : window size                                                 *
 *          datagram : data to send in tcp packet                             *
 *          datasize : size of datagram                                       *
 * Output : tcp_send : result of send                                         *
 ******************************************************************************/
int tcp_send(int      socket,
	     struct sockaddr_in *address,
	     unsigned long s_addr,
	     unsigned long t_addr,
	     unsigned      s_port,
	     unsigned      t_port,
	     unsigned char tcpflags,
	     unsigned long seq,
	     unsigned long ack,
	     unsigned      win,
	     char          *datagram,
	     unsigned      datasize)
{
  struct pseudohdr  {
    u_int32_t saddr;
    u_int32_t daddr;
    u_int8_t  useless;
    u_int8_t  protocol;
    u_int16_t tcplength;
  };

  unsigned char packet[2048];
  struct iphdr        *ip     = (struct iphdr *)(packet);
  struct pseudohdr    *pseudo = (struct pseudohdr *)(packet+sizeof(struct iphdr)-sizeof(struct pseudohdr));
  struct tcphdr       *tcp    = (struct tcphdr *)(packet+sizeof(struct iphdr));
  unsigned char       *data   = (unsigned char *)(packet+sizeof(struct tcphdr)+sizeof(struct iphdr));      

  /*
   * The above casts will save us a lot of memcpy's later.
   * The pseudo-header makes this way become easier than a union.
   */
	
  memcpy(data,datagram,datasize);
  memset(packet,0,sizeof(struct iphdr)+sizeof(struct tcphdr));

  /* The data is in place, all headers are zeroed. */
  tcp->th_sport   = htons(s_port);
  tcp->th_dport   = htons(t_port);
  tcp->th_off     = 5;          /* 20 bytes, (no options) */
  tcp->th_flags   = tcpflags;
  tcp->th_seq     = htonl(seq);
  tcp->th_ack     = htonl(ack);
  tcp->th_win     = htons(win); /* we don't need any bigger, I guess. */
  
  /* pseudo header for cheksum */
  pseudo->saddr = s_addr;
  pseudo->daddr = t_addr;
  pseudo->protocol = IPPROTO_TCP;   
  pseudo->tcplength = htons(sizeof(struct tcphdr)+datasize);  

  /* The necessary TCP header fields are set. */
  
  tcp->th_sum = in_cksum((u_short*)pseudo, sizeof(struct tcphdr)+sizeof(struct pseudohdr)+datasize);
  
  /* Make the ip header */
  memset(packet, 0, sizeof(struct iphdr));
  ip->saddr    = s_addr;
  ip->daddr    = t_addr;
  ip->version  = 4;
  ip->ihl      = 5;
  ip->ttl      = 255;
  ip->id       = random()%1996;
  ip->protocol = IPPROTO_TCP; /* should be 6 */
  ip->tot_len  = htons(sizeof(struct iphdr) + sizeof(struct tcphdr) + datasize);
  ip->check    = in_cksum((char *)packet,sizeof(struct iphdr));

#ifdef DEBUGGING
  printf("Packet ready. Dump: \n");
#ifdef DEBUGGING_DATA
  HEXDUMP(sizeof(struct iphdr)+sizeof(struct tcphdr)+datasize,packet);
#else
  HEXDUMP(sizeof(struct iphdr)+sizeof(struct tcphdr),packet);
#endif
  printf("\n");
#endif
  
  return sendto(socket, packet, sizeof(struct iphdr)+sizeof(struct tcphdr)+datasize, 0, (struct sockaddr *)address, sizeof(struct sockaddr));
  
  /* And off into the raw socket it goes. */
}

/******************************************************************************
 * Function udp_send                                                          *
 * send a customized UDP Header                                               *
 * Input  : socket : socket to write                                          *
 *          address : destination address                                     *
 *          s_addr, t_addr : source and destination address for checksum      *
 *          s_port, t_port : source and dest port                             *
 *          datagram : data to send in udp packet                             *
 *          datasize : size of datagram                                       *
 * Output : udp_send : result of send                                         *
 ******************************************************************************/
int udp_send(int      socket,
	     struct sockaddr_in *address,
	     unsigned long s_addr,
	     unsigned long t_addr,
	     unsigned      s_port,
	     unsigned      t_port,
	     char          *datagram,
	     unsigned      datasize)
{
  struct pseudohdr  {
    u_int32_t saddr;
    u_int32_t daddr;
    u_int8_t  useless;
    u_int8_t  protocol;
    u_int16_t udplength;
  };

  unsigned char packet[2048];
  struct iphdr        *ip     = (struct iphdr *)(packet);
  struct pseudohdr    *pseudo = (struct pseudohdr *)(packet+sizeof(struct iphdr)-sizeof(struct pseudohdr));
  struct udphdr       *udp    = (struct udphdr *)(packet+sizeof(struct iphdr));
  unsigned char       *data   = (unsigned char *)(packet+sizeof(struct udphdr)+sizeof(struct iphdr));      

  /*
   * The above casts will save us a lot of memcpy's later.
   * The pseudo-header makes this way become easier than a union.
   */
	
  memcpy(data,datagram,datasize);
  memset(packet,0,sizeof(struct iphdr)+sizeof(struct udphdr));

  /* The data is in place, all headers are zeroed. */
  udp->uh_sport   = htons(s_port);
  udp->uh_dport   = htons(t_port);
  udp->uh_ulen    = htons(sizeof(struct udphdr) + datasize);
  
  /* pseudo header for cheksum */
  pseudo->saddr = s_addr;
  pseudo->daddr = t_addr;
  pseudo->protocol = IPPROTO_UDP;   
  pseudo->udplength = htons(sizeof(struct udphdr)+datasize);  

  /* The necessary UDP header fields are set. */
  
  udp->uh_sum = in_cksum((u_short*)pseudo, sizeof(struct udphdr)+sizeof(struct pseudohdr)+datasize);
  
  /* Make the ip header */
  memset(packet, 0, sizeof(struct iphdr));
  ip->saddr    = s_addr;
  ip->daddr    = t_addr;
  ip->version  = 4;
  ip->ihl      = 5;
  ip->ttl      = 255;
  ip->id       = random()%1996;
  ip->protocol = IPPROTO_UDP;
  ip->tot_len  = htons(sizeof(struct iphdr) + sizeof(struct udphdr) + datasize);
  ip->check    = in_cksum((char *)packet,sizeof(struct iphdr));

#ifdef DEBUGGING
  printf("Packet ready. Dump: \n");
#ifdef DEBUGGING_DATA
  HEXDUMP(sizeof(struct iphdr)+sizeof(struct udphdr)+datasize,packet);
#else
  HEXDUMP(sizeof(struct iphdr)+sizeof(struct udphdr),packet);
#endif
  printf("\n");
#endif
  
  return sendto(socket, packet, sizeof(struct iphdr)+sizeof(struct udphdr)+datasize, 0, (struct sockaddr *)address, sizeof(struct sockaddr));
  
  /* And off into the raw socket it goes. */
}

/******************************************************************************
 * A relatively fast (or at least short ;) ping function.  Doesn't require a  *
 *  seperate checksum function. Thanks to Fyodor                              *
 * Input  : target  : ping destination                                        *
 *          timeout : ping time out (ms)                                      *
 * Output : isup    : if host alive (if he has reply to ping query)           *
 *          ouraddr : the returned adress (our address) by ping               *                         
 ******************************************************************************/
int isup(struct in_addr target, struct in_addr *ouraddr, long timeout)
{
  int res, retries = 3;
  struct sockaddr_in sock;
  /*type(8bit)=8, code(8)=0 (echo REQUEST), checksum(16)=34190, id(16)=31337 */
#ifdef __LITTLE_ENDIAN_BITFIELD
  unsigned char ping[64] = {
    0x08, 0x00, 0x41, 0x27, 0x03, 0x8E, 0x00, 0x07, 0x75, 0xCA, 0x69, 0x39, 0x63, 0xB4, 0x0E, 0x00, 0x08, 0x09, 0x0A, 0x0B,
    0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33,
    0x34, 0x35, 0x36, 0x37 
  };/*{ 0x8, 0x0, 0x8e, 0x85, 0x69, 0x7A };*/
#else
  unsigned char ping[64] = {
    0x08, 0x00, 0x27, 0x41, 0x8E, 0x03, 0x07, 0x00, 0x75, 0xCA, 0x69, 0x39, 0x63, 0xB4, 0x0E, 0x00, 0x08, 0x09, 0x0A, 0x0B,
    0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33,
    0x34, 0x35, 0x36, 0x37 
  };/*{ 0x8, 0x0, 0x85, 0x8e, 0x7A, 0x69 };*/
#endif
  int sd;
  struct timeval tv;
  struct timeval start, end;
  fd_set fd_read;
  struct {
    struct iphdr ip;
    unsigned char type;
    unsigned char code;
    unsigned short checksum;
    unsigned short identifier;
    char crap[16536];
  }  response;
  
  sd = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
  
  bzero((char *)&sock,sizeof(struct sockaddr_in));
  sock.sin_family=AF_INET;
  sock.sin_addr = target;
#ifdef DEBUGGING
  printf(" Sending 3 64 byte raw pings to host.\n");
#endif
  gettimeofday(&start, NULL);
  while(--retries) 
    {
      if ((res = sendto(sd,(char *) ping,64,0,(struct sockaddr *)&sock,
			sizeof(struct sockaddr))) != 64) 
	{
	  perror("sendto");
	  return 0;
	} 
      FD_ZERO(&fd_read);
      FD_SET(sd, &fd_read);
      tv.tv_sec = timeout / 1000; 
      tv.tv_usec = timeout * 1000;
      while(1)
	{
	  if ((res = select(sd + 1, &fd_read, NULL, NULL, &tv)) != 1) 
	    break;
	  else 
	    {
	      read(sd,&response,sizeof(response));
#ifdef DEBUGGING
	      printf("ping recv :\n");
#ifdef DEBUGGING_DATA
	      HEXDUMP(84, (char*)&response);
#else
	      HEXDUMP(28, (char*)&response);
#endif
	      printf("\ntype : %d; code : %d; ident : %u\n", response.type, response.code, response.identifier);
#endif
	      if  (response.ip.saddr == target.s_addr &&  !response.type 
		   && !response.code   && response.identifier == 910) 
		{
		  gettimeofday(&end, NULL);
		  ouraddr->s_addr = response.ip.daddr;
		  close(sd);
		  return 1;       
		}
	    }
	}
    }
  close(sd);
  return 0;
}

/******************************************************************************
 * send_icmp_echo -- Send an icmp echo request                                *
 * Input  : sock   : socket descriptor                                        *
 *          dest   : destination address                                      *
 *          ident  : icmp ident                                               *
 *          seq_nb : icmp sequence number                                     *
 *          data   : optional data                                            *
 *          sdata  : size of data                                             *
 * Output : return : success or failed                                        *
 ******************************************************************************/
int send_icmp_echo(int sock, struct sockaddr_in dest, u_short ident, u_short seq_nb, char *data, int sdata)
{
  char packet[2048];
  struct icmphdr *icmp     = (struct icmphdr*)packet;
  unsigned char  *datagram = (unsigned char*)(packet + 8);

  /* making the icmp header */
  memcpy(datagram, data, sdata);
  memset(packet, 0, 8);

  icmp->type = ICMP_ECHO;
  icmp->code = 0;
  icmp->un.echo.id = ident;
  icmp->un.echo.sequence = seq_nb;

  /* calculate the checksum */
  icmp->checksum = in_cksum((u_short*)packet, sdata+8);

#ifdef DEBUGGING
#ifdef DEBUGGING_DATA
  printf("ICMP Echo packet ready, Dump : \n");
  HEXDUMP(sdata+8, packet);
  printf("\n");
#else
  printf("ICMP Echo packet ready, Dump : \n");
  HEXDUMP(8, packet);
  printf("\n");
#endif
#endif
  return sendto(sock, packet, 8+sdata, 0, (struct sockaddr*)&dest, sizeof(struct sockaddr));
}

/******************************************************************************
 * ping -- Send an icmp echo request and stand the reply                      *
 * Input  : sock     : socket descriptor                                      *
 *          dest     : destination address                                    *
 *          timeout  : time out for echo request                              *
 *          packsize : size of packet to send                                 *
 *          ident    : icmp ident                                             *
 *          seq_nb   : icmp sequence number                                   *
 * Output : return   : see section 'ping return values' in rtcipscan.h        *
 ******************************************************************************/
int ping(int sock, struct sockaddr_in dest, int timeout, int packsize, u_short ident, u_short seq_nb)
{
  int             rc;
  fd_set          rset;
  struct timeval  cur_time, beg_time, tv;
  char            buf[2048];
  struct iphdr   *ip   = (struct iphdr *) (buf);
  struct icmphdr *icmp = (struct icmphdr*)(buf + sizeof(struct iphdr));

  for(rc=0; rc<packsize-8; rc++) buf[rc] = rc; /* initialize packets */
  /* sending a ICMP echo request */
  gettimeofday(&beg_time,NULL);
  send_icmp_echo(sock,dest,ident,seq_nb, buf, packsize-8);
  tv.tv_sec = timeout / 1000; 
  tv.tv_usec = timeout * 1000; 

  /* check for reply */
  FD_ZERO(&rset);
  FD_SET(sock, &rset);
  tv.tv_sec = timeout / 1000; 
  tv.tv_usec = timeout * 1000;
  gettimeofday(&cur_time,NULL);
  while(timeout > DIFFTIME(beg_time, cur_time))
    {
      if ((rc=select(sock + 1, &rset, NULL, NULL, &tv)) == 0) 
	return HOST_TIMEDOUT;  /* time out */
      else if (rc < 0) return PING_ERROR;
      else 
	{
	  rc = read(sock,buf,2048);

	  if(rc==-1) return PING_ERROR; /* error in reading */
	  else if(rc>0)
	    {
#ifdef DEBUGGING
	      printf("\nip header   :\n"); HEXDUMP(sizeof(struct iphdr), (char*)ip);
	      printf("\nsource : %u.%u.%u.%u, dest : %u.%u.%u.%u\n",
		     ip->saddr % 256, (ip->saddr >> 8) % 256, (ip->saddr >> 16) % 256, ip->saddr >> 24,
		     ip->daddr % 256, (ip->daddr >> 8) % 256, (ip->daddr >> 16) % 256, ip->daddr >> 24);
	      printf("\nicmp header :\n"); HEXDUMP(8, (char*)icmp);
#ifdef DEBUGGING_DATA
	      printf("\nicmp data   :\n"); HEXDUMP(rc-8-sizeof(struct iphdr), (char*)(icmp+8));
#endif
	      printf("\ntype : %02X, code : %02X, checksum : %04X, ident : %04X, seq_nb : %u\n",
		     icmp->type, icmp->code, icmp->checksum, icmp->un.echo.id, icmp->un.echo.sequence);
#endif
	      if (ip->saddr == dest.sin_addr.s_addr) /* got a reply */
		{
		  if((icmp->type == ICMP_ECHOREPLY) && (icmp->un.echo.id == ident) && (icmp->un.echo.sequence == seq_nb))
		    return HOST_ANSWER;
		  else return HOST_UNKNOW;
		}
	      else if(icmp->type == ICMP_DEST_UNREACH) return HOST_UNREACHABLE;
	      gettimeofday(&cur_time, NULL);
	    }
	}
    }
  return HOST_TIMEDOUT; /* time out */
}


/******************************************************************************
 * fast_ping -- Send an icmp echo request without standing the reply          *
 * Input  : sock     : socket descriptor                                      *
 *          dest     : destination address                                    *
 *          packsize : size of packet to send                                 *
 *          ident    : icmp ident                                             *
 *          seq_nb   : icmp sequence number                                   *
 * Output : return   : return of send                                         *
 ******************************************************************************/
int fast_ping(int sock, struct sockaddr_in dest, int packsize, u_short ident, u_short seq_nb)
{
  char            buf[2048];
  int i;
  for(i=0; i<packsize-8; i++) buf[i] = i; /* initialize packets */
  /* sending a ICMP echo request */
  gettimeofday(&last_beg_time,NULL);
  return send_icmp_echo(sock,dest,ident,seq_nb, buf, packsize-8);
}


/******************************************************************************
 * scan_for_pong -- Scan for an icmp ping reply                               *
 * Input  : sock     : socket descriptor                                      *
 *          source   : buffer to put source address if get a reply            *
 *          timeout  : the time out                                           *
 * Output : return   : cf ping                                                *
 ******************************************************************************/
int scan_for_pong(int sock, struct in_addr *source, int timeout)
{
  int             rc;
  fd_set          rset;
  struct timeval  cur_time, tv;
  char            buf[2048];
  struct iphdr   *ip     = (struct iphdr *) (buf);
  struct icmphdr *icmp   = (struct icmphdr*)(buf + sizeof(struct iphdr));
  struct iphdr   *icmpip = (struct iphdr*)(buf + sizeof(struct iphdr) + sizeof(struct icmphdr));
  /* check for reply */
  FD_ZERO(&rset);
  FD_SET(sock, &rset);
  tv.tv_sec = 0; 
  tv.tv_usec = 10; /* 10 microseconds */

  gettimeofday(&cur_time,NULL);

  if(timeout < DIFFTIME(last_beg_time, cur_time)) /* time out */
    return HOST_TIMEDOUT;

  if ((rc=select(sock + 1, &rset, NULL, NULL, &tv)) == 0) return 2;
  else if (rc<0) return -1; /* error in reading */

  rc = read(sock,buf,2048);

  if(rc==-1) return PING_ERROR; /* error in reading */
  else if(rc>0)
    {
#ifdef DEBUGGING
      printf("\nip header   :\n"); HEXDUMP(sizeof(struct iphdr), (char*)ip);
      printf("\nsource : %u.%u.%u.%u, dest : %u.%u.%u.%u\n",
	     ip->saddr % 256, (ip->saddr >> 8) % 256, (ip->saddr >> 16) % 256, ip->saddr >> 24,
	     ip->daddr % 256, (ip->daddr >> 8) % 256, (ip->daddr >> 16) % 256, ip->daddr >> 24);
      printf("\nicmp header :\n"); HEXDUMP(8, (char*)icmp);
#ifdef DEBUGGING_DATA
      printf("\nicmp data   :\n"); HEXDUMP(rc-8-sizeof(struct iphdr), (char*)(icmp+8));
#endif
      printf("\ntype : %02X, code : %02X, checksum : %04X, ident : %04X, seq_nb : %u\n",
	     icmp->type, icmp->code, icmp->checksum, icmp->un.echo.id, icmp->un.echo.sequence);
#endif
      source->s_addr = ip->saddr;
      if(icmp->type == ICMP_ECHOREPLY && !(icmp->code)) /* got an icmp echo reply */
	return HOST_ANSWER;
      if(icmp->type == ICMP_DEST_UNREACH)
	{
	  source->s_addr = icmpip->saddr;
	  return HOST_UNREACHABLE;
	}
      return HOST_UNKNOW;
    }
  return PING_RAS;
}

/******************************************************************************
 * Function resolve - resolve an host name                                    *
 * Input  : name : host name                                                  *
 * Output : resolve : in_addr struct for hostname                             *
 ******************************************************************************/
struct in_addr resolve(char *name)
{
  struct hostent *ent;
  struct in_addr in;

  if (!(ent=gethostbyname(name)))
    {
      in.s_addr=INADDR_NONE;
      return in;
    }

  return *(struct in_addr *)ent->h_addr; 
}
