/******************************************************************************
 *                          ftpscan.c by S/ash [RtC]                          *
 *                        RtC Scan : a network mapper                         *
 *                 contain functions for FTP bounce scanning                  *
 *              This is a part of the RtC Neset Project - RtC Tech            *
 ******************************************************************************/
#include "rtcscan.h"

/*
 * These constants aren't in constant.h because it only concern FTP scan
 */
/* ftp commands used here */
#define FTP_PORT_CMD   "PORT %u,%u,%u,%u,%u,%u\n"
#define FTP_USER_CMD   "USER %s\n"
#define FTP_PASS_CMD   "PASS %s\n"
#define FTP_QUIT_CMD   "QUIT\n"
#define FTP_LIST_CMD   "LIST\n"

/******************************************************************************
 * Function own_recv - modified recv : it's just add a 0 at the end of the    *
 *                      string and separate string using enters               *
 *                      (in fact read one answer from a ftp host)             *
 * cf : man recv... output have change to ftp answer                          *
 ******************************************************************************/
char recv_buffer[4096];
int  recv_index;
int  recv_length;
int own_recv(int s, char *buf, size_t len, int flags)
{
  int i, j, cont, n1, n2, multiline;
  i = 0;
  if(recv_index==recv_length)
    {
      recv_index=0;
      recv_length = recv(s, recv_buffer, 4096, flags);
#ifdef DEBUGGING_DATA
      if(recv_length>0)
	{
	  recv_buffer[recv_length<4096 ? recv_length : 4095] = 0;
	  printf("%s\n", recv_buffer);
	}
#endif
    }
  if(recv_length<0) return recv_length;
  cont = 1; multiline = 0;
  buf[i++] = '>';
  buf[i++] = ' ';
  j = 0;
  while(cont && (i<len-1))
    {
      if(i==5)
	{
	  buf[i] = 0;
	  n1 = atoi(buf+2);
	  if((n1>100) && (recv_buffer[recv_index]=='-'))
	    multiline = 1;
	  else multiline = 0;
	}
      else if(j==4)
	{
	  buf[i] = 0;
	  n2 = atoi(buf+i-3);
	  if(n1==n2) multiline = 0;
	}
      if(recv_buffer[recv_index]=='\n')
	{
	  if(multiline)
	    {
	      buf[i++] = '\n';
	      buf[i++] = '>';
	      buf[i++] = ' ';
	    }
	  else cont = 0;
	  j = 0;
	  recv_index++;
	}
      else buf[i++] = recv_buffer[recv_index++];
      j++;
      if((recv_index==recv_length) && cont && (i<len-1))
	{
	  if(n1>100)
	    {
	      recv_index=0;
	      recv_length = recv(s, recv_buffer, 4096, flags);
	      if(recv_length<0) return recv_length;
	    }
	  else cont = 0;
	}
    }
  buf[i] = 0;
#ifdef DEBUGGING
  printf("\n%s\n", buf);
  fflush(stdout);
#endif
  return n1;
}

/******************************************************************************
 * Function ftp_connect - connect to an ftp host                              *
 * Input  : host     : ftp host                                               *
 *          user     : ftp username                                           *
 *          pass     : ftp password                                           *
 *          port     : ftp port                                               *
 * Output : return   : error code : >0 socket, -1 : connexion refused,        *
 *                     -2 : pass & user refused, -3 : host command refused    *
 *        -3 include closed connexions (it mustn't be an ftp host)            *
 ******************************************************************************/
int ftp_connect(struct in_addr host, char *user, char *pass, int port)
{
  char buf[1024];
  struct sockaddr_in adr;
  int rc, i;
  int sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP); /* huh, it's just a classical TCP socket */

  recv_index  = 0;
  recv_length = 0;

  if(sock<0) return -1;
  adr.sin_family      = AF_INET;
  adr.sin_port        = htons(port);
  adr.sin_addr.s_addr = host.s_addr;
  if(connect(sock, (struct sockaddr*)(&adr), sizeof(adr))<0) return -1;
  /* well we're connected */
  rc = own_recv(sock, buf, 1024, 0);
  
  /* login */
  if(strlen(user)<1000) /* prevent buffer overflow */
    sprintf(buf, FTP_USER_CMD, user);
  else
    {
      close(sock);
      return -2;
    }
#ifdef DEBUGGING
  printf("< %s", buf);
#endif
  if(send(sock, buf, strlen(buf), 0)<0)
    {
      close(sock);
      return -1; /* we consider this as a connexion refused */
    }
  rc = own_recv(sock, buf, 1024, 0);
  if(rc<0)
    {
      close(sock);
      return -3;
    }
  while((rc/100) == 2)
    {
      rc = own_recv(sock, buf, 1024, 0);
      if(rc<0)
	{
	  close(sock);
	  return -3;
	}
    }
  if(rc!=331)
    {
      close(sock);
      return -3;
    }
  /* sending password */
  if(strlen(pass)<1000) /* prevent buffer overflow */
    sprintf(buf, FTP_PASS_CMD, pass);
  else
    {
      close(sock);
      return -2;
    }
#ifdef DEBUGGING
  printf("< %s", buf);
#endif
  if(send(sock, buf, strlen(buf), 0)<0)
    {
      close(sock);
      return -1; /* we consider this as a connexion refused */
    }
  rc = own_recv(sock, buf, 1024, 0);
  if(rc<0)
    {
      close(sock);
      return -3;
    }
  if(rc==530)
    {
      close(sock);
      return -2;
    }
  else if(rc==230)
    return sock;
  else
    {
      close(sock);
      return -3;
    }
}

/******************************************************************************
 * Function ftp_close - close a previously open connexion to an ftp host      *
 * Input  : sock : the connexion socket                                       *
 ******************************************************************************/
void ftp_close(int sock)
{
  char buf[1024];
  int rc;
#ifdef DEBUGGING
  printf("< %s", FTP_QUIT_CMD);
#endif
  if(send(sock, FTP_QUIT_CMD, strlen(FTP_QUIT_CMD), 0)<0)
    {
      close(sock);
      return; 
    }
  rc = own_recv(sock, buf, 1024, 0);
  close(sock);
}

/******************************************************************************
 * Function ftp_send_port : send a port command                               *
 * Input  : sock     : connexion socket                                       *
 *          port     : the port to check                                      *
 *          host     : on which host ?                                        *
 * Output : return   : 0 : port successful, -1 : conn error, 1 : port failed  *
 *                     -2 : unwanted answer                                   *
 *         if you get 1 there's chance that you can't use this ftp as a proxy *
 ******************************************************************************/
int ftp_send_port(int sock, int port, struct in_addr host)
{
  char buf[1024];
  char *tmp;
  int rc;

  tmp = inaddr_to_inet(host);
  sprintf(buf, FTP_PORT_CMD, tmp[0], tmp[1], tmp[2], tmp[3], port / 256, port % 256);
#ifdef DEBUGGING
  printf("< %s", buf);
#endif
  if(send(sock, buf, strlen(buf), 0)<0)
    return -1; 
  rc = own_recv(sock, buf, 1024, 0);
  if(rc<0)
    return -1;
  else if(rc==200) /* all good :-) */
    return 0;
  else if(rc==500) /* this fucking server don't want us own PORT command */
    return 1;
  else return -2; /* what he said ? */
  close(sock);
}

/******************************************************************************
 * Function ftp_check_port : check for port opened                            *
 * Input  : sock     : connexion socket                                       *
 * Output : return   : 0 : port closed, 1 : port opened, -1 : conn error      *
 *                     -2 : unwanted answer                                   *
 ******************************************************************************/
int  ftp_check_port(int sock)
{
  char buf[1024];
  char *tmp;
  int rc, i;

#ifdef DEBUGGING
  printf("< %s", FTP_LIST_CMD);
#endif
  
  if(send(sock, FTP_LIST_CMD, strlen(FTP_LIST_CMD), 0)<0)
    return -1; 
  rc = own_recv(sock, buf, 1024, 0);
  if(rc<0)
    return -1;
  else if((rc==150) || (rc==226)) /* good :-) */
    {
      rc = own_recv(sock, buf, 1024, 0);
      if(rc<0)
	return -1;
      return 1;
    }
  else if(rc==425) /* port closed */
    return 0;
  else return -2; /* what he said ? */
  close(sock);
}

/******************************************************************************
 * Function ftp_scan_port : scan a port                                       *
 * Input  : sock     : connexion socket                                       *
 *          host     : host to scan                                           *
 *          port     : port to scan                                           *
 *          retries  : number of tries to do when encountering a port error   *
 * Output : return   : 0 : port closed, 1 : port opened, -1 : conn error      *
 *                     -2 : unwanted answer, -3 port failed                   *
 ******************************************************************************/
int  ftp_scan_port(int sock, struct in_addr host, int port, int retries)
{
  int rc, i;
  for(i=0; (i<retries) && ((rc = ftp_send_port(sock, port, host))==1); i++);
  if (rc!=0) return rc==1 ? -3 : i;
  return ftp_check_port(sock);
}

/******************************************************************************
 * Function ftp_scan_ports - scan ports in a list on a host                   *
 * Input  : sock     : connexion socket                                       *
 *          host     : host to scan                                           *
 *          tries    : number of tries to do before considering that host     *
 *                     don't accept bogus PORT command                        *
 *          plf      : list of ports to scan                                  *
 * Output : portv    : port vector result                                     *
 *          return   : 0 if success, <0 on scanning error, 1 on port file err *
 ******************************************************************************/
int ftp_scan_ports(int sock, struct in_addr host, int tries,
		   struct port_list_file *plf, struct port_result *portv)
{
  long p,rc;
  char *buf = malloc(20000);

  /* initializing port vector */
  for(p=0; p<SIZEOF_PORTV; p++) portv[p].state = PORTSTATE_NO;


  while((p=get_next_port(plf, buf)))
    {
      if(p<0)
	{
	  free(buf);
	  return 1;
	}
      portv[p].service = (char*)malloc(strlen(buf)+1);
      strcpy(portv[p].service, buf);
      rc = ftp_scan_port(sock, host, p, tries);
      if(rc<0) return rc;
      else portv[p].state = rc ? PORTSTATE_OPENED : PORTSTATE_CLOSED;
    }
  return 0;
}
