/******************************************************************************
 *                        arguments.c by S/ash [RtC]                          *
 *                        RtC Scan : a network mapper                         *
 *       contain functions for argument and ip strings parsing                *
 *              This is a part of the RtC Neset Project - RtC Tech            *
 ******************************************************************************/


#include "rtcscan.h"


/******************************************************************************
 * Function print_help : print the help screen                                *
 * Input  : name       : program name                                         *
 *          longformat : boolean in integer : print the long format help ?    *
 ******************************************************************************/
void print_help(char *name, int longformat)
{
  if(longformat)
    printf(SYNTAXSTR, name);
  else fprintf(stderr, SHORTSYNTAXSTR, name, name);
  exit(-1);
}

/******************************************************************************
 * Function parse_arg : parse command line arguments                          *
 * Input  : argc   : argument counter                                         *
 *          argv   : argument vector                                          *
 * Output : flags  : option flags                                             *
 *          argtab : settings                                                 *
 * I don't use getopt in this function because it don't correspond to what i  *
 * need                                                                       *
 ******************************************************************************/
void parse_arg(int argc, char **argv, int *flags, long *argtab)
{
  int i,j, tmpi;
  argtab[IP_IND]            = 0;
  argtab[METHOD_IND]        = METHOD_T;
  argtab[DELAY_IND]         = DEFDELAY;
  argtab[OWNIP_IND]         = 0;
  argtab[TIMEOUT_IND]       = DEFTIMEOUT;
  argtab[OWNPORT_IND]       = DEFSTCP_PORT;
  argtab[OWNPORTMAX_IND]    = 0;
  argtab[MIN_FILE_PORT_IND] = 0;
  argtab[MAX_PORT_IND]      = 0;
  argtab[FTPHOST_IND]       = 0;
  argtab[FTP_USERNAME_IND]  = 0;
  argtab[FTP_PASSWD_IND]    = 0;
  argtab[FTP_PORT_IND]      = DEFSFTP_PORT;
  argtab[PACKETSIZE_IND]    = DEFAULT_PACKETSIZE;
  argtab[TRIES_IND]         = DEFAULT_TRIES;
  argtab[OUTFILE_IND]       = 0;
  *flags = 0;
  for(i=1; i<argc; i++)                                                 /* getting arguments   */
    {
      if(argv[i][0] == '-')                                             /* options             */
	{
	  tmpi=i;
	  for(j=1; argv[tmpi][j]; j++)
	    switch(argv[tmpi][j])
	      {
	      case 'l':
		if(argtab[IP_IND]) print_help(argv[0], 0);
		if((++i)<argc)
		  argtab[IP_IND] = i;
		else print_help(argv[0], 0);
		*flags |= IPFILE_FLAG;
		break;
	      case 's':
                if(argv[tmpi][++j])
		  switch(argv[tmpi][j])
		    {
		    case 'S':
		      argtab[METHOD_IND]=METHOD_S;
		      break;
		    case 'T':
		      argtab[METHOD_IND]=METHOD_T;
		      break;
		    case 'F':
		      argtab[METHOD_IND]=METHOD_F;
		      break;
		    case 'A':
		      argtab[METHOD_IND]=METHOD_A;
		      break;
		    case 'B':
		      argtab[METHOD_IND]=METHOD_B;
		      if((++i) < argc)
			argtab[FTPHOST_IND] = i;
		      else print_help(argv[0], 0);
		      break;
		    case 'E':
		      argtab[METHOD_IND]=METHOD_E;
		      break;
		    case 'U':
		      argtab[METHOD_IND]=METHOD_U;
		      break;
		    case 'u':
		      argtab[METHOD_IND]=METHOD_u;
		      break;
		    default:
		      print_help(argv[0], 0);
		    }
		else print_help(argv[0], 0);
		break;
	      case 'f':
		*flags |= SLOW_FLAG;
		break;
	      case 'd':
		if((++i)<argc)
		  argtab[DELAY_IND] = atoi(argv[i]);
		else print_help(argv[0], 0);
		break;
	      case 'i':
		if((++i)<argc)
		  argtab[OWNIP_IND] = i;
		else print_help(argv[0], 0);
		break;
	      case 'u':
		*flags |= ISUP_FLAG;
		break;
	      case 't':
		if((++i)<argc)
		  argtab[TIMEOUT_IND] = atoi(argv[i]);
		else print_help(argv[0], 0);
		break;
	      case 'r':
		if((++i)<argc)
		  argtab[OWNPORT_IND] = atoi(argv[i]);
		else print_help(argv[0], 0);
		break;
	      case 'H':
		if((++i)<argc)
	          sscanf(argv[i], "%d-%d", &(argtab[OWNPORT_IND]), &(argtab[OWNPORTMAX_IND]));
		else print_help(argv[0], 0);
		break;
	      case 'm':
		if((++i)<argc)
		  sscanf(argv[i], "%d-%d", &(argtab[MIN_FILE_PORT_IND]), &(argtab[MAX_PORT_IND]));
		else print_help(argv[0], 0);
		*flags |= USESERVICES_FLAG;
		break;
	      case 'p':
		if((++i)<argc)
		  argtab[MIN_FILE_PORT_IND] = i;
		else print_help(argv[0], 0);
		*flags |= PORTFILE_FLAG;
		*flags |= USESERVICES_FLAG;
		break;
	      case 'T':
		if((++i)<argc)
		  argtab[TRIES_IND] = atoi(argv[i]);
		else print_help(argv[0], 0);
		break;
	      case 'S':
		if((++i)<argc)
		  argtab[PACKETSIZE_IND] = atoi(argv[i]);
		else print_help(argv[0], 0);
		break;
	      case 'o':
		if((++i)<argc)
		  argtab[OUTFILE_IND]   = i;
		else print_help(argv[0], 0);
		break;		
	      case 'h':
		if(((++i)+1)<argc)
		  {
		    argtab[FTP_USERNAME_IND] = i;
		    argtab[FTP_PASSWD_IND]   = ++i;
		  }
		else print_help(argv[0], 0);
		break;
	      case 'R':
		if((++i)<argc)
		  argtab[FTP_PORT_IND] = atoi(argv[i]);
		else print_help(argv[0], 0);
		break;
	      case 'a':
		*flags |= SHOWALL_FLAG;
		break;
	      case 'e':
		*flags |= NOECHO_FLAG;
		break;
	      case 'n':
		*flags |= PROMPT_FLAG;
		break;
	      case '?':
		print_help(argv[0], 1);
	      default:
		print_help(argv[0], 0);
	      } 
	}
      else if(!argtab[IP_IND]) argtab[IP_IND] = i;
      else print_help(argv[0], 0);
    }
  if(!argtab[IP_IND]) print_help(argv[0], 0);
  /* parsing completed */
  /*
   * no check for options compatibility is made ! Beware : you could get
   * strange results if you made an error in giving arguments (such as using
   * simultaneously the -m and the -p options)
   */
}  


/******************************************************************************
 * Function inet_to_str - convert an ip in 4-bytes format to a string         *
 * Input  : inetaddr : ip in 4-bytes format                                   *
 * Output : return   : ip in dotted-string format                             *
 ******************************************************************************/
char *inet_to_str(char *inetaddr)
{
  char *res = (char*)malloc(16);
  sprintf(res, "%u.%u.%u.%u", (unsigned char)inetaddr[0], (unsigned char)inetaddr[1], 
	  (unsigned char)inetaddr[2], (unsigned char)inetaddr[3]);
  return res;
}

/******************************************************************************
 * Function inaddr_to_inet - convert an inaddr struct to an ip in 4-bytes fmt *
 * Input  : inetaddr : inaddr struct                                          *
 * Output : return   : ip in 4-bytes format                                   *
 ******************************************************************************/
char *inaddr_to_inet(struct in_addr inetaddr)
{
  char *res = (char*)malloc(4);
  memcpy(res, &(inetaddr.s_addr), 4);
  return res;
}

/******************************************************************************
 * Function inet_to_inaddr - convert an ip in 4-bytes fmt to an inaddr struct *
 * Input  : inetaddr : ip in 4-bytes format                                   *
 * Output : return   : inaddr struct                                          *
 ******************************************************************************/
struct in_addr inet_to_inaddr(char *inetaddr)
{
  struct in_addr res;
  memcpy(&(res.s_addr), inetaddr, 4);
  return res;
}


/******************************************************************************
 * Function open_ip_file - open an ip list file                               *
 * Input  : fn       : ip list filename                                       *
 * Output : ilf      : ip list file                                           *
 *          return   : if success                                             *
 ******************************************************************************/
int open_ip_file(char *fn, struct ip_list_file *ilf)
{
  int i;
  if(!(ilf->file = fopen(fn, "r"))) /* open the file */
    return 0;                /* if error      */
  for(i=0; i<4; i++)         /* clear the ips */
    {
      ilf->maxip[i] = 0;
      ilf->minip[i] = 0;
      ilf->curip[i] = 0;
    }
  ilf->curcomment = NULL;
  ilf->name       = NULL;

  /* all good, returning 1 */
  return 1;
}


/******************************************************************************
 * Function get_next_ip - get the next ip in ilf                              *
 * Input  : ilf      : ip list file                                           *
 * Output : ilf      : ip list file                                           *
 *          return   : if success (0 : success; 1 : eof; -1 : error)          *
 ******************************************************************************/
int get_next_ip(struct ip_list_file *ilf)
{
  char *buf; int i,j,k,r;
  if((ilf->curip[0]==ilf->maxip[0]) && (ilf->curip[1]==ilf->maxip[1]) 
     && (ilf->curip[2]==ilf->maxip[2]) && (ilf->curip[3]==ilf->maxip[3])) /* if no more ip in current line */
    {
      if(ilf->curcomment) free(ilf->curcomment); /* freeing curcomment */
      if(ilf->name) free(ilf->name);             /* freeing name       */
      ilf->name = NULL;
      if(!(ilf->file))                           /* prevent to reading on closed file */
	return 1;
      buf = (char*)malloc(1000);
      if(fgets(buf, 1000, ilf->file))            /* reading newline    */
	{
	  if(buf[strlen(buf)-1]=='\n') buf[strlen(buf)-1]=0;  /* removing the final CR if needed */
	  for(i=0; ((buf[i]==' ') || (buf[i]=='\t')) && buf[i] && (i<1000); i++); /* jumping whitespaces             */
	  if(i==1000)                                     /* if commentary or blank line     */
	    {
	      free(buf);
	      return get_next_ip(ilf);
	    }
	  else if((*(buf+i)=='#') || !(*(buf+i)))
	    {
	      free(buf);
	      return get_next_ip(ilf);
	    }
	  /* keep only ip */
	  j=i; k=0;
	  while((buf[i]!=0) && (buf[i]!=' ') && (buf[i]!='\t') && (i<1000)) i++;

	  if(i<1000)
	    {
	      if(buf[i]) k=1;
	      buf[i] = 0;
	    }
	  else
	    {
	      fprintf(stderr, IPLIST_ERROR);
	      free(buf);
	      return -1;
	    }
	  if((r=parse_ip_string(ilf, buf+j)))
	    { 
	      if (r==1)   /* ip string incorrect */
		{
		  fprintf(stderr, IPSTR_INVALID);
		  free(buf);
		  return -1;
		}
	      else        /* error resolving host */
		{
		  fprintf(stderr, INADDR_ERROR);
		  free(buf);
		  return -1;
		}
	    }

	  i++;
	  if((i<1000) && k)
	    {
	      /* get the commentary */
	      while(((buf[i]==' ') || (buf[i]=='\t')) && buf[i] && (i<1000)) i++;
	      if(i<1000)
		{
		  ilf->curcomment = malloc(strlen(buf+i)+1);
		  strcpy(ilf->curcomment, buf+i);
		}
	      else ilf->curcomment = NULL;
	    }
	  else ilf->curcomment = NULL;
	  /* All good, going on */
	  free(buf);
	  return 0;
	}
      else   /* terminating file */
	{
	  free(buf);
	  fclose(ilf->file);
	  return 1;
	}
    }
  else     /* more ip in current line */
    {
      if(ilf->curip[3] != ilf->maxip[3])
	{
	  ilf->curip[3]++;
	  /* all good, going on */
	  return 0;
	}
      else
	{
	  ilf->curip[3] = ilf->minip[3];
	  if(ilf->curip[2] != ilf->maxip[2])
	    {
	      ilf->curip[2]++;
	      /* all good, going on */
	      return 0;
	    }
	  else
	    {
	      ilf->curip[2] = ilf->minip[2];
	      if(ilf->curip[1] != ilf->maxip[1])
		{
		  ilf->curip[1]++;
		  /* all good, going on */
		  return 0;
		}
	      else
		{
		  ilf->curip[1] = ilf->minip[1];
		  ilf->curip[0]++;
		  /* all good, going on */
		  return 0;
		}
	      
	    }
	}
    }
}


/******************************************************************************
 * Function parse_ip_string - parse a readed ip string                        *
 * Input  : ip_str   : ip string to parse                                     *
 * Output : ilf      : ip list file to parse in                               *
 *          return   : if success (-1 on error resolving host)                *
 ******************************************************************************/
int parse_ip_string(struct ip_list_file *ilf, char *ip_str)
{
  int l, k, j, i=0;
  struct in_addr adr;
  char *tmp;
  /* checking if it's a host name */
  if(!isdigit(ip_str[0]))
    {
      /* Well, resolving host */
      adr = resolve(ip_str);
      if(adr.s_addr==INADDR_NONE) return -1;
      tmp = inaddr_to_inet(adr);
      memcpy(ilf->maxip, tmp, 4);
      memcpy(ilf->minip, tmp, 4);
      free(tmp);
      ilf->curip[0] = ilf->minip[0];
      ilf->curip[1] = ilf->minip[1];
      ilf->curip[2] = ilf->minip[2];
      ilf->curip[3] = ilf->minip[3];
      ilf->name     = (char*)malloc(strlen(ip_str)+1);
      strcpy(ilf->name, ip_str);
      return 0;
    }
  /* No, huh, it's a True IP String */
  /* going to 1st dot */
  ilf->name = NULL;
  while(ip_str[i]!=0 && ip_str[i]!='.') i++;
  k=0;
  for(j=0; j<4; j++)
    { 
      if(ip_str[k] == 0) return 1;              /* string invalid */
      ip_str[i] = 0;
      switch(ip_str[k])
	{
	case '0':                               /* single ip      */
	case '1':
	case '2': 
	case '3':
	case '4':
	case '5':
	case '6': 
	case '7':
	case '8':
	case '9':
	  if(atoi(ip_str+k) > 255) return 1;
	  ilf->maxip[j] = atoi(ip_str+k);
	  ilf->minip[j] = ilf->maxip[j];
	  if(ilf->maxip[j]/100)                 /* check for bad string */
	    {
	      if((i-k)!=3) return 1;
	    }
	  else if(ilf->maxip[j]/10)
	    {
	      if((i-k)!=2) return 1;
	    }
	  else if((i-k)!=1) return 1;
	  break;
	case '*':                                /* all ip */
	  if((i-k)!=1) return 1;
	  ilf->maxip[j] = 255;
	  ilf->minip[j] = 0;
	  break;
	case '[':                               /* part of ip */
	  if((i-k)<5) return 1;                 /* check for bad string */
	  if(ip_str[i-1]!=']') return 1;
	  for(l=k; ip_str[l]!=0 && ip_str[l]!='-'; l++); /* going to '-' */
	  if(ip_str[l] == 0) return 1;              /* string invalid */
	  ip_str[l] = 1;
	  ip_str[i-1] = 1;
	  l++; k++;
	  if((atoi(ip_str+k+1)>255) || (atoi(ip_str+l)>255)) return 0;
	  ilf->minip[j] = atoi(ip_str+k);
	  ilf->maxip[j] = atoi(ip_str+l);
	  if(ilf->minip[j]/100)                 /* check for bad string */
	    {
	      if((l-k)!=4) return 1;
	    }
	  else if(ilf->minip[j]/10)
	    {
	      if((l-k)!=3) return 1;
	    }
	  else if((l-k)!=2) return 1;
	  if(ilf->maxip[j]/100)
	    {
	      if((i-l)!=4) return 1;
	    }
	  else if(ilf->maxip[j]/10)
	    {
	      if((i-l)!=3) return 1;
	    }
	  else if((i-l)!=2) return 1;
	  break;
	default:
	  return 1;
	}
      k = ++i;
      /* going to next dot */
      while(ip_str[i]!=0 && ip_str[i]!='.') i++;
    }
  if(*(ip_str+i)) return 1;
  /* All is good, going on */
  ilf->curip[0] = ilf->minip[0];
  ilf->curip[1] = ilf->minip[1];
  ilf->curip[2] = ilf->minip[2];
  ilf->curip[3] = ilf->minip[3];
  return 0;
}
