/*  TheTrIaX'sSoundSystem  */

#include <stdio.h>
#include <stdlib.h>
#include "sb.h"
#include "tss.h"
#include "types.h"

/* Variables globales */
CHANNEL		*tss_channels;		/* Pointeur sur les canaux */
WORD		tss_nbrchannels;	/* Nombre de canaux */
BYTE		tss_mainvol;		/* Volume de mixage */
WORD		tss_mainfreq;		/* Frquence de mixage */
BYTE		*tss_voltable;		/* Table de conversion du vol */
BYTE		*tss_mixtable;		/* Table de mixage des canaux */
WORD		tss_buffersize;		/* Taille du buffer de mixage */
WORD		*tss_mixbuffer;		/* Ptr sur le buffer de mixage */
BYTE		*tss_sbbuffer;		/* Ptr sur le buffer de la SB */

/************************************************************************
 *   Rgle le nombre de canaux:                                        *
 *  - Met  jour tss_nbrchannels                                        *
 *  - Si des canaux ont dj t allous, alors:                        *
 *      - Libre la mmoire de ces canaux                               *
 *  - Alloue de la mmoire pour les nouveaux canaux                     *
 ************************************************************************/
int tss_setnbrchn (WORD nbrchn)
{
	tss_nbrchannels = nbrchn;

	/* Libre et alloue un nouveau buffer */
	if (tss_channels != NULL)
		free(tss_channels);
	if ((tss_channels = calloc(tss_nbrchannels,sizeof(*tss_channels))) == NULL)
		return 0;

	return 1;
}

/************************************************************************
 *   Rgle le volume principal:                                        *
 *  - Met  jour tss_mainvol                                            *
 *  - Libre la mmoire de la table de mixage si dj allou            *
 *  - Alloue de la mmoire pour une nouvelle table de mixage            *
 *  - Cre la table de mixage des canaux                                *
 ************************************************************************/
int tss_setmainvol (BYTE mainvol)
{
	WORD	smp;

	tss_mainvol = mainvol;

	/* Libre et alloue une nouvelle table */
	if (tss_mixtable != NULL)
		free(tss_mixtable);
	if ((tss_mixtable = malloc(tss_nbrchannels*256)) == NULL)
		return 0;

	/* Cre la nouvelle table de mixage des canaux */
	for (smp=0; smp<tss_nbrchannels*256; smp++)
		tss_mixtable[smp] = ((DWORD)smp*tss_mainvol)/(tss_nbrchannels*64);

	return 1;
}

/************************************************************************
 *   Rgle la frquence de mixage principale:                          *
 *  - Met  jour tss_mainfreq                                           *
 *  - Calcule la taille du buffer de mixage (tss_buffersize)            *
 *  - Libre la mmoire du buffer de mixage si dj allou              *
 *  - Libre la mmoire du buffer de la SB si dj allou               *
 *  - Alloue de la mmoire pour le buffer de mixage                     *
 *  - Alloue de la mmoire pour le buffer de la SB                      *
 *  - Rgle la frquence de mixage sur la SB (sb_setfreq())             *
 ************************************************************************/
int tss_setmainfreq (WORD mainfreq)
{
	tss_mainfreq = mainfreq;
	tss_buffersize = tss_mainfreq/50;  /* 50 appels par seconde */

	/* Libre buffers */
	if (tss_mixbuffer != NULL)
		free(tss_mixbuffer);
	if (tss_sbbuffer != NULL)
		free(tss_sbbuffer);

	/* Alloue buffers */
	if ((tss_mixbuffer = malloc(tss_buffersize<<1)) == NULL)
		return 0;
	if ((tss_sbbuffer = malloc(tss_buffersize)) == NULL)
		return 0;

	sb_setfreq(tss_mainfreq);
	return 1;
}

/************************************************************************
 *   Rgle le volume d'un canal:                                       *
 *  - Calcule l'adresse du canal                                        *
 *  - Met  jour le champ "vol"                                         *
 *  - Calcule l'index dans la table de volume                           *
 ************************************************************************/
void tss_setchnvol (WORD chn, BYTE vol)
{
	CHANNEL	*pchn = &tss_channels[chn];

	pchn->vol = vol;
	pchn->pvol = tss_voltable+(vol<<8);
}

/************************************************************************
 *   Rgle la frquence d'un canal:                                    *
 *  - Calcule l'adresse du canal                                        *
 *  - Met  jour le champ "freq"                                        *
 *  - Calcule l'incrment du pointeur de sample                         *
 ************************************************************************/
void tss_setchnfreq (WORD chn, WORD freq)
{
	CHANNEL	*pchn = &tss_channels[chn];

	pchn->freq = freq;
	pchn->incr = ((DWORD)freq<<16)/tss_mainfreq;
}

/************************************************************************
 *   Commence la sortie d'un sample:                                   *
 *  - Calcule l'adresse du canal                                        *
 *  - Rgle le volume du canal (tss_setchnvol)                          *
 *  - Rgle la frquence du canal (tss_setchnfreq)                      *
 *  - Rgle le reste des champs " la main"                             *
 ************************************************************************/
void tss_playsample (WORD chn, SAMPLE *smp)
{
	CHANNEL	*pchn = &tss_channels[chn];

	tss_setchnvol(chn,smp->vol);
	tss_setchnfreq(chn,smp->freq);

	pchn->flags	= smp->flags + TSS_ACTIVE;
	pchn->length	= smp->length;
	pchn->loopstart	= smp->loopstart;
	pchn->loopend	= smp->loopend;
	pchn->sample	= smp->sample;
	pchn->start	= 0;
	pchn->end	= (DWORD)smp->length<<16;
	pchn->pos	= 0;
}

/************************************************************************
 *   Stoppe la sortie d'un sample:                                     *
 *  - Calcule l'adresse du canal                                        *
 *  - Met le bit d'activit du canal  0                                *
 ************************************************************************/
void tss_stopsample (WORD chn)
{
	CHANNEL	*pchn = &tss_channels[chn];

	pchn->flags &= ~TSS_ACTIVE;
}

/************************************************************************
 *   Routine de mixage des canaux:                                     *
 *  - Vide buffer de mixage                                             *
 *  > - Pour chn de 0  tss_nbrchannels                               *
 *     - Initialise mixleft = tss_buffersize;                          *
 *              et pmix = tss_mixbuffer;                               *
 *     > - S'il reste des octets  mixer (mixleft!=0) ET             *
 *            Si le canal est actif, alors:                           *
 *            - Calcule le nombre d'octets  mixer:                   *
 *                          pchn->end - pchn->pos                     *
 *                tomix =                    *
 *                                pchn->incr                          *
 *            - Si le nombre d'octets  mixer est suprieur           *
 *              au nombre d'octets restants, alors:                   *
 *                    - tomix = mixleft;                              *
 *            - Sinon met le bit TSS_ASKFORLOOP du canal  1          *
 *                  (demande de looping du canal)                     *
 *            - Soustrait le nombre d'octets  mixer du               *
 *              nombre restant d'octets  mixer, en clair:            *
 *                    mixleft -= tomix;                               *
 *        > - S'il reste des octets  mixer (tomix!=0), alors:      *
 *               - Additionne l'octet du sample au buffer de mixage	*
 *               - Incrmente le pointeur de position du canal       *
 *                   (chn->pos += chn->incr)                         *
 *               - Dcrmente tomix                                  *
 *         - Next                                              *
 *            - Si le canal doit tre loop (TSS_LOOPED  1), alors:	*
 *                - Si le bit de looping de sample est  1, alors:	*
 *                    - Efface le bit TSS_ASKFORLOOP                  *
 *                    - chn->start = (DWORD)chn->loopstart<<16;       *
 *                    - chn->end = (DWORD)chn->loopend<<16;           *
 *                    - chn->pos = chn->start                         *
 *            - Sinon stoppe le canal                                 *
 *      - Next                                                  *
 *   - Next                                                      *
 *  - Initialise le nombre d'octets  convertir mixleft                 *
 *  - Initialise le pointeur pmix avec le buffer de mixage              *
 *  - Initialise le pointeur psb avec le buffer de la SB                *
 *  > - S'il reste des octets  mixer (mixleft!=0), alors:            *
 *         - Copie le sample du buffer de mixage dans le buffer        *
 *              de la SB en l'ayant pralablement pass dans la        *
 *              moulinette de la table prcalcule de mixage:          *
 *                 *psb++ = tss_mixtable[*pmix++];                     *
 *         - Dcrmente mixleft                                        *
 *   - Next                                                      *
 *  - Envoie le buffer calcul  la SB                                  *
 ************************************************************************/
void tss_mixhandler ()
{
	CHANNEL	*pchn;		/* Pointeur sur le canal courant */
	WORD	chn;		/* Numro du canal courant */
	WORD	mixleft;	/* Nombre d'octets restants  mixer */
	WORD	tomix;		/* Nombre d'octets devant tre mixs */
	WORD	*pmix;		/* Pointeur sur le buffer de mixage */
	BYTE	*psb;		/* Pointeur sur le buffer de la SB */

	/* Vide buffer de mixage */
	memset(tss_mixbuffer,0,tss_buffersize<<1);

	/* Mixe les canaux dans le buffer de mixage */
	for (chn=0,pchn=tss_channels; chn<tss_nbrchannels; chn++,pchn++)
	{
		mixleft = tss_buffersize;
		pmix = tss_mixbuffer;

		/* Si le canal est actif, mixe-le */
		while (mixleft && (pchn->flags & TSS_ACTIVE))
		{
			/* Teste si le canal va devoir tre loop */
			if ((tomix = (pchn->end-pchn->pos)/pchn->incr) > mixleft)
				tomix = mixleft;
			else
				pchn->flags |= TSS_ASKFORLOOP;

			/* Soustrait le nombre d'octets que l'on va mixer */
			mixleft -= tomix;

			/* Mixe le canal */
			while (tomix)
			{
				*pmix++ += pchn->pvol[pchn->sample[pchn->pos>>16]];
				pchn->pos += pchn->incr;
				tomix--;
			}

			/* Si le canal doit tre loop, alors: */
			if (pchn->flags & TSS_ASKFORLOOP)
				if (pchn->flags & TSS_LOOPON)
				{
					/* Si le bit TSS_LOOPON est mis, loope le canal */
					pchn->flags &= ~TSS_ASKFORLOOP;
					pchn->start = (DWORD)pchn->loopstart<<16;
					pchn->end = (DWORD)pchn->loopend<<16;
					pchn->pos = pchn->start;
				}
				else
					/* Sinon stoppe le canal */
					tss_stopsample(chn);
		}
	}

	/* Convertit le buffer de mixage en sample  envoyer  la SB */
	mixleft = tss_buffersize;
	pmix = tss_mixbuffer;
	psb = tss_sbbuffer;
	while (mixleft)
	{
		*psb++ = tss_mixtable[*pmix++];
		mixleft--;
	}

	/* Envoie le sample calcul  la SB */
	sb_dmaout(tss_buffersize,tss_sbbuffer);
}

/************************************************************************
 *   Initialise le module TSS:                                         *
 *  - Initialise la SB                                                  *
 *  - Installe le handler de mixage                                     *
 *  - Initialise les pointeurs des buffers                              *
 *  - Alloue mmoire pour la table de conversion du volume              *
 *  - Cre la table de conversion du volume                             *
 *  - Rgle le nombre de canaux                                         *
 *  - Rgle le volume principal                                         *
 *  - Rgle la frquence principale                                     *
 *  - Lance la boucle de mixage (appel de tss_mixhandler())             *
 ************************************************************************/
int tss_init (WORD nbrchn, WORD mainfreq)
{
	BYTE	vol;
	WORD	smp;

	if (!sb_init())
		return 0;
	sb_setuserhandler(tss_mixhandler);

	/* Initialise pointeurs */
	tss_channels = NULL;
	tss_voltable = NULL;
	tss_mixtable = NULL;
	tss_mixbuffer = NULL;
	tss_sbbuffer = NULL;

	/* Alloue mmoire pour la table de volume */
	if ((tss_voltable = malloc(65*256)) == NULL)
		return 0;

	/* Cre la table de volume */
	for (vol=0; vol<=64; vol++)
		for (smp=0; smp<256; smp++)
			tss_voltable[(vol<<8)+smp] = 128+(smp-128)*vol/64;

	/* Rgle nombre de canaux, volume et frquence principals */
	if (!tss_setnbrchn(nbrchn))
		return 0;
	if (!tss_setmainvol(64))
		return 0;
	if (!tss_setmainfreq(mainfreq))
		return 0;

	tss_mixhandler();	/* Lance la boucle de mixage */
	return 1;
}

/************************************************************************
 *   Uninitialise le module TSS:                                       *
 *  - Stoppe le transfert DMA                                           *
 *  - Uninitialise la SB                                                *
 *  - Libre tous les buffers:                                          *
 *      - Canaux (tss_channels)                                         *
 *      - Table de conversion du volume (tss_voltable)                  *
 *      - Table de mixage des canaux (tss_mixtable)                     *
 *      - Buffer de mixage (tss_mixbuffer)                              *
 *      - Buffer de la SB (tss_sbbuffer)                                *
 ************************************************************************/
void tss_done ()
{
	sb_dmastop();
	sb_done();

	/* Libre tous les buffers */
	free(tss_channels);
	free(tss_voltable);
	free(tss_mixtable);
	free(tss_mixbuffer);
	free(tss_sbbuffer);
}
