/*  Gestion de la SB  */

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include "dma.h"
#include "sb.h"
#include "types.h"

/* Constantes */
#define SB_DMAREAD      DMA_READ + DMA_INCREMENT + DMA_SINGLE
#define SB_DMAWRITE     DMA_WRITE + DMA_INCREMENT + DMA_SINGLE

#define DSP_RESET       0x06    /* Port de reset */
#define DSP_READ        0x0a    /* Port de lecture */
#define DSP_WRITE       0x0c    /* Port d'criture */
#define DSP_READY	0x0e    /* Port d'tat des donnes */

/* Constantes pour l'autodtection SB */
#define SB_NBRIRQ	5	/* Nombre d'irqs possibles */
#define SB_NBRDMA	3	/* Nombre de canaux dma possibles */
WORD	sb_irqavail[SB_NBRIRQ]	= {2,3,5,7,10};	/* Irqs possibles */
WORD	sb_dmaavail[SB_NBRDMA]	= {1,3,5};	/* Canaux dma possibles */

/* Variables globales */
WORD		sb_port;		/* Port de base SB */
WORD		sb_irq;			/* Ligne d'irq SB */
WORD		sb_dma;			/* Canal dma SB */
int		sb_dmadone;		/* Si 1, transfert DMA termin */
int		sb_dmahold;		/* Si 1, transfert DMA en pause */
int		sb_quiet;		/* Si 1, HP de la SB teint */
int		sb_userdefined;		/* User handler dfini oui/non */
void interrupt	(*sb_oldhandler)();	/* Ancien handler d'irq SB */
void		(*sb_userhandler)();	/* Handler SB dfini par l'user */

/************************************************************************
 *   Reset de la SB:                                                   *
 *      - Envoyer 1 au port de reset (2x6h)                             *
 *      - Attendre 3ms                                                  *
 *      - Envoyer 0 au port de reset (2x6h)                             *
 *      - Tenter une lecture un certain nombre de fois (ici 256)        *
 *      - Si l'octet lu est gal  0xaa, c'est gagn!                   *
 ************************************************************************/
int sb_reset ()
{
	WORD    i;	/* Compteur pour viter que le reset
			   ne boucle sans fin si le port n'est
			   pas bon */

	outportb(sb_port | DSP_RESET,1);
	delay(3);
	outportb(sb_port | DSP_RESET,0);

	for (i=0; i<256; i++)
		if ((inportb(sb_port | DSP_READY) & 0x80) == 0x80)
			return (inportb(sb_port | DSP_READ) == 0xaa);

	return 0;
}

/************************************************************************
 *   Lecture d'un octet sur la SB:                                     *
 *      - Attendre que le bit 7 du port 2xeh passe  1                  *
 *      - Lire l'octet au port 2xah                                     *
 ************************************************************************/
BYTE sb_read ()
{
	while ((inportb(sb_port | DSP_READY) & 0x80) != 0x80);
	return (inportb(sb_port | DSP_READ));
}
				
/************************************************************************
 *   Ecriture d'un octet sur la SB:                                    *
 *      - Attendre que le bit 7 du port 2xch passe  0                  *
 *      - Ecrire l'octet au port 2xch                                   *
 ************************************************************************/
void sb_write (BYTE data)
{
	while ((inportb(sb_port | DSP_WRITE) & 0x80) == 0x80);
	outportb(sb_port | DSP_WRITE,data);
}

/************************************************************************
 *   Allume le haut-parleur de la SB:                                  *
 *      - Si le HP est teint, alors:                                   *
 *      - Rgler flag sb_quiet  FALSE (HP non teint, donc allum)     *
 *          - Allumer le HP en envoyant la commande 0xd1  la SB        *
 ************************************************************************/
void sb_hpon ()
{
	if (sb_quiet)
	{
		sb_quiet = 0;
		sb_write(0xd1);
	}
}

/************************************************************************
 *   Eteint le haut-parleur de la SB:                                  *
 *      - Si le HP n'est pas dj teint, alors:                        *
 *      - Rgler flag sb_quiet  TRUE (HP teint)                       *
 *          - Eteindre le HP en envoyant la commande 0xd3  la SB       *
 ************************************************************************/
void sb_hpoff ()
{
	if (!sb_quiet)
	{
		sb_quiet = 1;
		sb_write(0xd3);
	}
}

/************************************************************************
 *   Lecture directe d'un sample sur la SB:                            *
 *      - Envoyer la commande 0x20 (direct sampling)                    *
 *      - Lire le sample par un petit sb_read tout simple               *
 ************************************************************************/
BYTE sb_directin ()
{
	sb_write(0x20);
	return (sb_read());
}

/************************************************************************
 *   Ecriture directe d'un sample sur la SB:                           *
 *      - Envoyer la commande 0x10 (direct out)                         *
 *      - Envoyer le sample par sb_write                                *
 ************************************************************************/
void sb_directout (BYTE data)
{
	sb_write(0x10);
	sb_write(data);
}

/************************************************************************
 *   Rgle la frquence de sampling pour les transferts DMA:           *
 *      - Envoyer la commande 0x40 (set sampling rate)                  *
 *      - Envoyer la 'timer constant', calcule suivant                 *
 *        la sacro-sainte formule :                                     *
 *                                                                      *
 *                                1,000,000                             *
 *                TC = 256 -                       *
 *                             Frquence (en Hz)                        *
 *                                                                      *
 ************************************************************************/
void sb_setfreq (WORD freq)
{
	sb_write(0x40);
	sb_write(256 - 1000000L / freq);
}

/************************************************************************
 *   Enregistrement d'un sample via DMA sur la SB:                     *
 *      - Si un transfert DMA est en cours, stopper le transfert        *
 *      - Eteindre le HP pour viter les effets secondaires...          *
 *      - Reprogrammer le contrleur DMA                                *
 *      - Envoyer la commande 0x24  la SB                              *
 *      - Envoyer la taille du transfert - 1                            *
 *  - Rgler flags sb_dmadone et sb_dmahold  FALSE                     *
 ************************************************************************/
void sb_dmain (WORD size, void *ptr)
{
	if (!sb_dmadone)
		sb_dmastop();

	sb_hpoff();	/* Eteint le HP sinon il se produit un effet
			   trange qui conduit la SB  renvoyer le son
			   qu'elle enregistre aux haut-parleurs, avec
			   cependant une qualit vraiment, heu...,
			   enfin vous voyez, quoi! */

	dma_start(sb_dma,SB_DMAREAD,size,ptr);
	size--;
	sb_write(0x24);
	sb_write((BYTE)(size));
	sb_write((BYTE)(size >> 8));

	sb_dmadone = 0;		/* On commence un transfert */
	sb_dmahold = 0;		/* Le transfert est en cours */
}

/************************************************************************
 *   Sortie d'un sample via DMA sur la SB:                             *
 *      - Si un transfert DMA est en cours, stopper le transfert        *
 *      - Allumer le HP pour que l'on entende quelque chose (?!)        *
 *      - Reprogrammer le contrleur DMA                                *
 *      - Envoyer la commande 0x14  la SB                              *
 *      - Envoyer la taille du transfert - 1                            *
 *  - Rgler flags sb_dmadone et sb_dmahold  FALSE                     *
 ************************************************************************/
void sb_dmaout (WORD size, void *ptr)
{
	if (!sb_dmadone)
		sb_dmastop();

	sb_hpon();      /* Pourquoi ce sb_hpon() ?? Reflchissez
			   un peu, je pense que vous finirez
			   par trouver... */

	dma_start(sb_dma,SB_DMAWRITE,size,ptr);
	size--;
	sb_write(0x14);
	sb_write((BYTE)(size));
	sb_write((BYTE)(size >> 8));

	sb_dmadone = 0;		/* On commence un transfert */
	sb_dmahold = 0;		/* Le transfert est en cours */
}

/************************************************************************
 *   Arrt d'un transfert DMA:                                         *
 *      - Si un transfert DMA n'est pas termin ou en pause, alors:     *
 *          - Stopper le contrleur DMA (dma_stop())                    *
 *          - Stopper la SB en envoyant la commande 0xd0                *
 *      - Rgler flag sb_dmahold  TRUE (transfert en pause)            *
 ************************************************************************/
void sb_dmastop ()
{
	if (!sb_dmadone && !sb_dmahold)
	{
		dma_stop(sb_dma);
		sb_write(0xd0);
		sb_dmahold = 1;
	}
}

/************************************************************************
 *   Continuation d'un transfert DMA:                                  *
 *      - Si un transfert DMA a t arrt, alors:                      *
 *          - Relancer le contrleur DMA (dma_continue())               *
 *          - Relancer la SB en envoyant la commande 0xd4               *
 *      - Rgler flag sb_dmahold  FALSE (transfert non paus)          *
 ************************************************************************/
void sb_dmacontinue ()
{
	if (sb_dmahold)
	{
		dma_continue(sb_dma);
		sb_write(0xd4);
		sb_dmahold = 0;
	}
}

/************************************************************************
 *   Nouveau handler d'IRQ pour la SB:                                 *
 *      - Lire un octet au port 2xeh (pourquoi? mystre...)             *
 *  - Si l'IRQ est >= 8 (second contrleur d'IRQ), alors:               *
 *      - Envoyer un End_Of_Interrupt (0x20) au port 0xa0               *
 *  - Envoyer une commande End_Of_Interrupt sur le port 0x20            *
 *  - Rgler le flag sb_dmadone  TRUE                                  *
 *  - Si l'utilisateur a dfini un handler  lui, alors:                *
 *      - Appeler sb_userhandler()                                      *
 ************************************************************************/
void interrupt sb_handler ()
{
	/* La ligne qui suit ne sert absolument  rien (du moins
	   sur une SB 2.0), mais elle figure dans les docs et dans
	   tous les sources que j'ai pu voir, alors... */

	inportb(sb_port | DSP_READY);

	if (sb_irq >= 8)
		outportb(0xa0,0x20);	/* End_Of_Interrupt au PIC2 */
	outportb(0x20,0x20);		/* End_Of_Interrupt au PIC1 */

	sb_dmadone = 1;		/* Le transfert est termin */

	/* Appel du handler utilisateur */
	if (sb_userdefined)
		sb_userhandler();
}

/************************************************************************
 *   Installe le handler d'irq de la SB:                               *
 *  - Si l'IRQ est < 8, alors:                                          *
 *      - Ranger l'ancienne valeur du vecteur sb_irq + 8                *
 *      - Dtourne l'int sb_irq + 8 sur notre sb_handler()              *
 *      - Dmasque l'irq sur le registre de masquage du PIC             *
 *  - Sinon, l'irq concerne le second contrleur:                       *
 *      - Ranger l'ancienne valeur du vecteur sb_irq + 0x70             *
 *      - Dtourne l'int sb_irq + 0x70 sur notre sb_handler()           *
 *      - Dmasque l'irq 2 sur le registre de masquage du PIC 1         *
 *        pour permettre la cascade des IRQs                            *
 *      - Dmasque l'irq sur le registre de masquage du PIC 2           *
 ************************************************************************/
void sb_inithandler ()
{
	if (sb_irq < 8)
	{
		/* Premier PIC */
		sb_oldhandler = getvect(sb_irq+0x8);
		setvect(sb_irq+0x8,sb_handler);
		outportb(0x21,inportb(0x21) & !(1 << sb_irq));
	}
	else
	{
		/* Second PIC */
		sb_oldhandler = getvect(sb_irq+0x70);
		setvect(sb_irq+0x70,sb_handler);
		outportb(0x21,inportb(0x21) & !(1 << 2));
		outportb(0xa1,inportb(0xa1) & !(1 << (sb_irq-8)));
	}
}

/************************************************************************
 *   Dsinstalle le handler d'irq de la SB:                            *
 *  - Si l'IRQ est < 8, alors:                                          *
 *      - Rinstalle l'ancien vecteur en sb_irq + 8                     *
 *  - Sinon, l'IRQ est >= 8, alors:                                     *
 *      - Rinstalle l'ancien vecteur en sb_irq + 0x70                  *
 ************************************************************************/
void sb_donehandler ()
{
	if (sb_irq < 8)
		setvect(sb_irq+0x8,sb_oldhandler);
	else
		setvect(sb_irq+0x70,sb_oldhandler);
}

/************************************************************************
 *   Installe un "handler utilisateur" appel  chaque IRQ de la SB:	*
 *  - Sauve l'adresse du handler dans sb_userhandler                    *
 *  - Positionne le flag sb_userdefined  TRUE                          *
 ************************************************************************/
void sb_setuserhandler (void (*handler)())
{
	sb_userhandler = handler;
	sb_userdefined = 1;
}

/************************************************************************
 *   Dtection automatique de la SB:                                   *
 *      - Si une chane d'environnement "BLASTER" est prsente, alors:  *
 *          - Dgager le port, l'IRQ, et le canal DMA                   *
 *          - Tenter un p'tit reset                                     *
 *          - Tenter un p'tit transfert DMA                             *
 *              -> Si tout a bien march, on a bien une SB!             *
 *      - Si pas de chane d'environnement, alors on fait un test HARD: *
 *          - Tester un reset sur les ports 0x210  0x260               *
 *          - Chercher la ligne d'IRQ et le canal DMA avec une          *
 *            srie de transferts successifs:                           *
 *            (sur SB, les lignes d'IRQ autorises sont: 2,3,5,7,10     *
 *                  et les canaux DMA autoriss sont: 1,3,5)            *
 *              - Essayer chaque canal DMA                              *
 *              - Pour chaque canal DMA, essayer chaque IRQ             *
 *                  -> Si aucun couple IRQ & DMA ne convient, pas de SB *
 ************************************************************************/
int sb_detect ()
{
	char	*blaster;
	WORD	i;
	WORD	irq;
	WORD	dma;

	if ((blaster = getenv("BLASTER")) != NULL)
	{
		/* Lit la chane d'environnement */
		sscanf(blaster,"A%x I%u D%u",&sb_port,&sb_irq,&sb_dma);

		/* Si le reset marche, alors teste irq & dma */
		if (sb_reset())
		{
			sb_inithandler();
			sb_setfreq(22000);
			sb_dmaout(16,NULL);
			for (i=0; i<0xffff && !sb_dmadone; i++);
			sb_donehandler();
		}

		/* Si le transfert DMA a fonctionn, on a bien une SB */
		if (sb_dmadone)
			return 1;
	}
	else  /* Test HARD */
	{
		/* Teste tous les ports compris entre 0x210 et 0x260 */
		for (sb_port=0x210; sb_port<=0x260; sb_port+=0x10)
		{
			/* Si le reset marche, cherche l'irq et le dma */
			if (sb_reset())
				for (dma=0; dma<SB_NBRDMA; dma++)
					for (irq=0; irq<SB_NBRIRQ; irq++)
					{
						sb_irq = sb_irqavail[irq];
						sb_dma = sb_dmaavail[dma];

						/* Tente un p'tit transfert DMA */
						sb_inithandler();
						sb_setfreq(22000);
						sb_dmaout(16,NULL);
						for (i=0; i<0xffff && !sb_dmadone; i++);
						sb_donehandler();

						/* Si l'irq a t appele, on a bien une SB */
						if (sb_dmadone)
							return 1;
					}
		}
	}
	return 0;
}

/************************************************************************
 *   Initialisation du module SB:                                      *
 *  - Appelle sb_detect() pour voir si une SB est bien l               *
 *  - Installe le handler d'IRQ                                         *
 *  - Positionne les flags sb_quiet, sb_dmadone et sb_dmahold           *
 *  - Allume le HP (petite ruse, voyez plus bas...)                     *
 ************************************************************************/
int sb_init ()
{
	/* Si pas de SB, alors pas la peine de continuer... */
	if (!sb_detect())
		return 0;

	sb_inithandler();

	sb_quiet = 1;		/* SB muette */
	sb_dmadone = 1;		/* Transfert DMA termin */
	sb_dmahold = 0;		/* Transfert DMA non paus */
	sb_userdefined = 0;	/* Pas de handler dfini par l'user */

	/* Petite ruse: comme sb_quiet est  1, sb_hpon() nous
	   allume bien gentiment le haut-parleur et positionne le
	   flag sb_quiet  FALSE, ce qui nous permet d'tre sr de
	   l'tat du HP sans passer par la commande 0xd8, qui
	   n'existe pas sur les SB < SB Pro */

	sb_hpon();

	return 1;
}

/************************************************************************
 *   Dsinitialise(??) le module SB:                                   *
 *  - Stoppe le transfert DMA en cours (si il y a en un)                *
 *  - Eteint le HP                                                      *
 *  - Dsinstalle le handler d'IRQ                                      *
 ************************************************************************/
void sb_done ()
{
	sb_dmastop();
	sb_hpoff();
	sb_donehandler();
}
