/***************************************************************************
				  TIMER.C


	Ensemble de routines de gestion du 8254 et du RTC

****************************************************************************

Auteur          : Pierre Larbier
Dvelopp le    : 13.2.1995
Dernire MAJ    : 13.2.1995


Ce fichier accompagne l'article "Mesure du Temps" du Reporter n5.

Toutes les routines de ce fichier sont librement utilisables (si vous les
employez dans un programme commercial, envoyez-moi un petit mot ... a fait
plaisir). Elles ont t testes mais c'est sans aucune garantie ...

***************************************************************************/
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <time.h>
#include <bios.h>

/************************************************************************/
/*                                                                      */
/* Ne pas compiler ce fichier avec l'option Stack Checking car on       */
/* travaille sous interruption                                          */
/*                                                                      */
/*                                                                      */
/* ATTENTION !! L'utilisation de l'mulateur 8087 pose des problmes de */
/* prcision. Je souponne que cel est d  son chargement dans le     */
/* cache du processeur (je n'en suis absolument pas certain ...). En    */
/* tat de cause il introduit une erreur de 20s (sur un P5/90) lors    */
/* certains appels de fonctions de mesure du temps. C'est pourquoi on   */
/* compile en forant l'usage d'un 80x87 (navr pour ceux qui ne        */
/* disposent pas de coprocesseur ...).				        */
/*                                                                      */
/************************************************************************/




#ifdef __SC__
/************************************************************************/
/*                                                                      */
/* compilation avec Symantec C++ 6.1 :  				*/
/*        SC -e -Jm -ml -cod -ff -4 -o+time -v2 -o%1.exe %1.c           */
/*  (les rpertoires o se trouvent les include et le linker sont       */
/*    correctement dfinis )						*/
/*                                                                      */
/************************************************************************/
	#define COMPILATEUR "Symantec C++ 6.1"
	/* Gestion du curseur avec Symantec C++ 6.1 */
	#include <disp.h>
	#define SHOWCUR() disp_showcursor()
	#define HIDECUR() disp_hidecursor()
	#ifdef DOS386
		Ce programme ne fonctionne pas avec DOSX
	#endif
	/* Symantec dispose de sa propre fonction de dtection de CPU */
	#define CPUTYPE() cputype()

#elif __TURBOC__
/************************************************************************/
/*                                                                      */
/* compilation avec Borland C++ 3.1 :  				        */
/*   BCC -3 -C -G -N- -a -B -f287 -k -ms -u -tDe -e%1.exe -o%1.obj %1.c */
/*  (les rpertoires o se trouvent les include et les lib sont         */
/*    correctement dfinis )						*/
/*                                                                      */
/* Vous pouvez aussi utiliser le fichier timerbc.prj.                   */
/************************************************************************/
	#define COMPILATEUR "Borland C++ 3.1"
	/* Gestion du curseur avec Borland C++ 3.1 */
	#define HIDECUR() _setcursortype(_NOCURSOR)
	#define SHOWCUR() _setcursortype(_NORMALCURSOR)
	/* BC++3.1 ne connait pas _dosdate_t ni _dostime_t */
	#define _dosdate_t dosdate_t
	#define _dostime_t dostime_t
	/* Notre fonction de dtection pour Borland C++ 3.1 */
	#define CPUTYPE() CpuId()

#elif _MSC_VER
/************************************************************************/
/*                                                                      */
/* compilation avec Microsoft C++ 7.0 :  				*/
/*   Utiliser le fichier timerc7.mak fourni                             */
/*                                                                      */
/************************************************************************/
	#define COMPILATEUR "Microsoft C++"
	/* Gestion du curseur avec Microsoft C++ 7.0*/
	/* On le place en bas de l'cran car sinon on le retrouve n'importe o */
	#include <graph.h>
	short oldcursor;
	#define SHOWCUR() _settextcursor(oldcursor);_displaycursor(_GCURSORON)
	#define HIDECUR() oldcursor=_gettextcursor(); _displaycursor(_GCURSOROFF);_settextposition(25,1)
	/* Et pour les versions antrieures  MSVC (Microsoft C++ 8.0) */
	#ifndef MK_FP
		#define MK_FP(seg,offs) (void _far *)((((ulong)seg)<<16) | offs)
	#endif
	/* Microsoft ne connait pas la pseudo instruction assembleur db */
	#define db _emit
	/* Notre fonction de dtection pour Microsoft C++ 7.0 */
	#define CPUTYPE() CpuId()

#endif


/* Fonction 386 pour un assembleur intgr 16 bits (Microsoft et Borland) */
#define OP32 __asm db 0x66

/* Gestion de validation des interruption en pile */
#define Pdisable() _disable();CliCpt++
#define Penable()  if(--CliCpt<=0){CliCpt=0;_enable();}

/* Gagne de la place et de la clart  la lecture du source */
#define detourne(it,fct,sav) Pdisable();sav=_dos_getvect(it);_dos_setvect(it,fct);Penable()
#define restore(it,sav) Pdisable();_dos_setvect(it,sav);Penable()


/* Attente de la synchro verticale */
#define SyncVert()  while((inp(0x3da)&8)==0);while((inp(0x3da)&8)!=0)

/* Attente de la synchro ligne */
#define SyncLigne()  while((inp(0x3da)&1)==0);while((inp(0x3da)&1)!=0)

/* Attente du l'expiration du timer 0 (interruptions disables) */
#define SyncTimer()  outp(0x20,0xa);while((inp(0x20)&1)==0)



#define AVANT   0
#define APRES   1

#define OK      0
#define ERREUR  1

/* Type des processeurs */
#define Proc8086        0
#define Proc80286       2
#define Proc80386       3
#define Proc80486       4
#define ProcP5          5
#define ProcInconnu 	6



/* Types utiliss */
typedef unsigned char uchar;
typedef unsigned short ushort;
typedef unsigned long ulong;


typedef struct {
	unsigned short lsb;
	unsigned long msb;
} CPT48B;


/* Prototypes des fonctions de ce fichier */
/******************************************/
/* Attente minimale en C */
void io_delay(void);

/* Lecture du compteur n0 */
ushort LitTimer0(void);

/* Fonction sous interruption 18.2 fois/s qui implmente un compteur 32 bits */
void __interrupt __far TickPerso(void);

/* Dtection du processeur */
char CpuId(void);

/* Lecture du timer du Pentium */
double LitTimerPentium(void);

/* Teste si on peut utiliser le timer du Pentium */
char TesteTimerPentium(void);

/* Lecture de notre timer tendu sur 48 bits */
void PTimer(CPT48B *cpt);

/* Dbut de mesure du temps */
void TimerDeb(void);

/* Fin de mesure du temps */
float TimerFin(char mode);

/* Produit un retard (exprim en seconde) */
void Pdelay(float delai);

/* Initialise les constantes utilises dans la mesure du temps */
void TimerInit(void);

/* Remet en place le timer systme */
void TimerRestore(void);

/* Lit un registre du MC146818A */
uchar LitRTC(uchar registre);

/* Ecrit un registre du MC146818A */
void EcritRTC(uchar registre, uchar val);

/* Remet le DOS  l'heure */
short remise_a_l_heure(void);



/* Prototypes des fonctions de dmo de ce fichier */
/**************************************************/
/* Analyse le mode de fonctionnement du 8254 */
void TimerAnalyseMode(void);

/* Fonction d'exemple excute  chaque VBL */
void __far fct_vbl(void);

/* Fonction d'it synchronise  la VBL */
void __interrupt __far ItVbl(void);

/* Routine sous it 18.2 fois/s utilise dans la nano-dmo */
void __interrupt __far ItUserTick(void);

/* Routine sous it actualisation de la RTC utilise dans la nano-dmo */
void __far ItUF(void);

/* Routine sous it priodique de la RTC utilise dans la nano-dmo */
void __far ItPF(void);

/* Routine sous it alarme la RTC utilise dans la nano-dmo */
void __far ItAF(void);

/* Routine de selection de la source d'it RTC utilise dans la nano-dmo */
void __interrupt __far ItRTC(void);



/* Liste des variables globales */
CPT48B TiDeb, TiFin;
float TiOffsetTimer,TiOffsetDiff,PDelayOffset1,PDelayOffset2;
volatile long Pcpt,PDelayFact,IncVBL,AccVBL;
volatile short CliCpt = 0;

/* Variables globales de la nano-dmo */
volatile ushort DureeVbl,CoulFond,DureeDemo,DemoFlag,CptSound;

void (__interrupt __far *OldTick)();
void (__interrupt __far *OldUserTick)();
void (__interrupt __far *OldRTC)();




/*************************************************************************

  Procdure : io_delay

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction effectue un petit dlai qu'on peut placer aprs
	      un accs I/O. Ce n'est utile que sur les vieilles cartes-mres.

	      Pour forcer Microsoft C++ 7.0  l'appeller malgr son
	      optimiseur, on rajoute la ligne __asm.


  Entre : aucune

  Sortie : aucune

*************************************************************************/
void io_delay(void)
{
__asm	nop
}


/*************************************************************************

  Procdure : LitTimer0

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction lit le compteur n0 du 8254.
	      Elle corrige les erreurs eventuelles de lecture.

  Entre : aucune

  Sortie : valeur lue 16 bits replace en ordre croissant ( la base les
	      compteurs du 8254 sont des dcompteurs).

*************************************************************************/
ushort LitTimer0(void)
{
unsigned lsb1,lsb2;
		outp(0x43,0x0);
		io_delay();
		lsb1 = inp(0x40);
		io_delay();
		lsb1 |= (inp(0x40) << 8);
                io_delay();
                outp(0x43,0x0);
		io_delay();
                lsb2 = inp(0x40);
		io_delay();
                lsb2 |= (inp(0x40) << 8);
		if ((lsb1 > lsb2) || (lsb1 < 6))
			return ~lsb1;
		else
			return ~lsb2 - 4;
}


/*************************************************************************

  Procdure : TickPerso

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction incrmente un compteur 32 bits sous interruption
	      n0x8 (ou 0x1C).

	      La variable globale incrmente est Pcpt.
  Entre : aucune

  Sortie : aucune

*************************************************************************/
void __interrupt __far TickPerso(void)
{
	Pcpt++;

	_chain_intr(OldTick);
}


/*************************************************************************

  Procdure : CpuId

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction dtermine le type de processeur utilis. Elle
	      est directement drive de la note d'application Intel AP-485
	      (Microprocessors: Vol III 1994).
	      Elle n'utilise que des instructions asm 80286.

	      ATTENTION : cette routine ne fonctionne pas sous OS/2

  Entre : aucune

  Sortie : char pouvant valoir :
		Proc8086 => dtection d'un 8086/8088/NEC V20/V30
		Proc80286 => dtection d'un 80286
		Proc80386 => dtection d'un 80386
		Proc80486 => dtection d'un 80486
		ProcP5 => dtection d'un Pentium
		ProcInconnu => on a pas russi  dtecter le processeur

*************************************************************************/
char CpuId(void)
{
ushort flags, val;

        /* Dtecte si on est sur un 8088/86 */
	/* Teste si les bits 12  15 des flags sont toujours  1 */
        Pdisable();
__asm   pushf           /* lit les flags */
__asm   pop ax          /* rcupre les flags dans ax */
__asm   mov flags,ax    /* sauve les flags pour la suite */
__asm   and ax,0fffh    /* tente de forcer les bits 12  15  0 */
__asm   push ax         /* place les flags modifis dans ax */
__asm   popf
__asm   pushf
__asm   pop ax
__asm   mov val,ax
        Penable();
        if ((val >> 12) == 0xf)
		return Proc8086;

	/* Dtecte si on est sur un 80286 */
        /* Teste si les bits 12  15 des flags sont toujours  0 */
                Pdisable();
__asm   mov ax,flags    /* Rcupre les flags */
__asm   or ax,0f000h    /* Tente de forcer les 4 bits de poids fort */
__asm   push ax         /* place les flags modifis dans ax */
__asm   popf
__asm   pushf
__asm   pop ax
__asm   mov val,ax
        Penable();
        if ((val >> 12) == 0x0)
		return Proc80286;

	/* Dtecte si on est sur un 80386 */
	/* Le bit AC des EFLAGS ne peut pas tre positionn sur un 386 */
	Pdisable();
__asm   mov bx,sp       /* sauve la pile car on ne veut pas dclencher */
__asm   and sp, 0fffch  /* d'erreur d'alignement en touchant  AC !! */
	OP32
__asm   pushf           /* empile les EFLAGS */
	OP32
__asm   pop ax          /* pour les rcuprer dans EAX */
	OP32
__asm   mov cx,ax       /* les sauvegarde dans ECX */
	OP32
__asm   db 35h          /* XOR avec 40000h pour inverser le bit AC */
__asm   db 0
__asm   db 0
__asm   db 4
__asm   db 0
	OP32            /* push les nouveaux EFLAGS */
__asm   push ax
	OP32
__asm   popf
	OP32
__asm   pushf
	OP32
__asm   pop ax          /* pour les rcuprer dans EAX */
	OP32            /* et comparer avec les anciens EFLAGS */
__asm   xor ax,cx
__asm   mov sp,bx       /* remet la pile en place */
__asm   mov ax,0
__asm   jz CpuIdSuite1  /* saute si on a pas russi  positionner le bit */
__asm   mov ax,1        /* On a mieux qu'un 386 */
CpuIdSuite1:
__asm   mov val,ax
	Penable();
	if (val == 0)
		return Proc80386;

	/* Dtecte si on est sur un 80486 */
	/* Le bit ID des EFLAGS ne peut pas tre modifi sur les 486DX actuels */
	Pdisable();
	OP32
__asm   pushf           /* empile les EFLAGS */
	OP32
__asm   pop ax          /* pour les rcuprer dans EAX */
	OP32
__asm   mov cx,ax           /* les sauvegarde dans ECX */
	OP32
__asm   db 35h          /* XOR avec 200000h pour inverser le bit ID */
__asm   db 0
__asm   db 0
__asm   db 20h
__asm   db 0
	OP32            /* push les nouveaux EFLAGS */
__asm   push ax
	OP32
__asm   popf
	OP32
__asm   pushf
	OP32
__asm   pop ax          /* pour les rcuprer dans EAX */
	OP32            /* et comparer avec les anciens EFLAGS */
__asm   xor ax,cx
__asm   mov ax,0
__asm   jz CpuIdSuite2  /* saute si on a pas russi  inverser le bit */
__asm   mov ax,1        /* On a mieux qu'un 486DX actuel */
CpuIdSuite2:
__asm   mov val,ax
	Penable();
	if (val == 0)
		return Proc80486;

	/* Dtecte si on est sur un Pentium ou un processeur plus rcent */
	/* Comme le bit ID est modifiable, on dispose de l'instruction CPUID */
	Pdisable();
	OP32             /* 1 dans EAX pour lire le type de processeur */
__asm   xor ax,ax        /* donc pas de test bizarre prconis par Intel !*/
	OP32
__asm   inc ax
__asm   db 0fh           /* Instruction CPUID */
__asm   db 0a2h
__asm   mov val,ax       /* rsultat dans AH */
	Penable();
	val = (val >> 8) & 0xf;
	/* Si on trouve 4 => 486, 5 => Pentium, autre chose : inconnu */
	switch(val) {
		case 4 :
			return Proc80486;
		case 5:
			return ProcP5;
	}

	return ProcInconnu;
}


/*************************************************************************

  Procdure : LitTimerPentium

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction lit le timer interne du Pentium.

	      Le timer du Pentium est remis  0 au reset du processeur.
	      Il s'incrmente  chaque coup d'horloge.

	      ATTENTION : cette fonction ne doit tre execute qu'en mode
	      rel ou en mode V86 avec privilge maximal.

  Entre : aucune

  Sortie : double refletant la valeur du timer.

*************************************************************************/
double LitTimerPentium(void)
{
ushort msb, mot3 , mot2 , lsb;

	OP32
__asm   db 0b9h  /* move ECX,10h (on lit le registre 10h) */
__asm   db 10h
 __asm   db 0
__asm   db 0
__asm   db 0
__asm   db 0fh   /* Instruction RDSMR registre 10h*/
__asm   db 32h
__asm   nop      /* poil d'attente (??) */
__asm   nop      /* le rsultat est dans edx:eax */
__asm   mov lsb,ax
__asm   mov mot3,dx
	OP32
__asm   shr ax,16
	OP32
__asm   shr dx,16
__asm   mov mot2,ax
__asm   mov msb,dx

	return (double)lsb + 65536.*
	       ((double)mot2 + 65536.*
	       ((double)mot3+65536.*(double)msb));
}


/*************************************************************************

  Procdure : TesteTimerPentium

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction teste si il est possible de lire le timer du
	      Pentium. Le teste n'est pas exhaustif et je ne garanti pas
	      qu'il suffise pour viter la violation de privilge.
	      En particulier, j'ai t incapable de vrifier qu'on est bien
	      en mode V86 et de dterminer le CPL. Le temps me manque pour
	      finaliser cette routine.

	      Nanmoins il m'a toujours suffit ....

  Entre : aucune

  Sortie : OK => on peut utiliser le timer du Pentium
	   ERREUR => on ne doit pas utiliser le timer du Pentium

*************************************************************************/
char TesteTimerPentium(void)
{
ushort val;
uchar status;

	/* DOSX de Symantec C++ 6.1 place le processeur en mode protg,
		la lecture du timer du P5 est donc impossible */
	#ifdef DOS386
		return ERREUR;
	#endif

	/* Si on est en mode rel, le timer est lisible */
__asm   smsw ax
__asm   mov val,ax
	if ((val & 1) == 0)
		return OK;

	/* On est dans un mode protg mais il n'est pas possible de faire la
	diffrence entre le mode V86 et le mode protg, il ne nous reste
	plus que la supposition suivante :
	si le processeur est en mode protg c'est pour une bonne raison sinon
	il est en mode V86. Cette bonne raison, je l'interprte comme la
	capacit de faire du multi-tches. Si c'est le cas, le systme doit
	interdire l'accs au timer 0 car c'est un moyen de "planter" toutes
	les tches DOS. On va donc tester la capacit de modifier le
	fonctionnement du timer 0 */

	outp(0x43,0xe2); /* Lit le mode du timer 0 */
	io_delay();
	status = inp(0x40);
        io_delay();
        outp(0x43,(status & 0xf) | 0x31); /* passe en BCD */
	io_delay();
        outp(0x40,0);
        io_delay();
	outp(0x40,0);
	io_delay();

	/* Puis on lit le mode programm */
	outp(0x43,0xe2);
	io_delay();
        val = (ushort)inp(0x40);
	/* On est pass en BCD ?? */
        if ((val & 1) != 1)
		return ERREUR;

	/* Ca marche, et on revient en mode normal */
        outp(0x43,(status & 0xf) | 0x30);
        io_delay();
        outp(0x40,0);
	io_delay();
        outp(0x40,0);

	/* Voil, on suppose que l'accs au timer du Pentium est possible */
	return OK;
}



/*************************************************************************

  Procdure : PTimer

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction constitue un compteur 48 bits  partir des
	      16 bits du compteur n0 du 8254 et des 32 bits de la variable
	      Pcpt.
	      Ces 48 bits correspondent  l'instant auquel on est avec une
	      rsolution de 838.106ns.

  Entre : pointeur sur une structure CPT48B qui contiendra le compteur

  Sortie : contenu de la structure.

*************************************************************************/
void PTimer(CPT48B *cpt)
{
	Pdisable();
	cpt->lsb = LitTimer0();
	cpt->msb =  Pcpt + (unsigned long)(inp(0x20) & (~cpt->lsb >> 15));
	Penable();
}


/*************************************************************************

  Procdure : TimerDeb

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction dmarre une mesure de temps en stockant les
	      48 bits de notre compteur tendu dans la variable globale
	      TiDeb.

  Entre : aucune

  Sortie : aucune

*************************************************************************/
void TimerDeb(void)
{
	PTimer(&TiDeb);
	return;
}


/*************************************************************************

  Procdure : TimerFin

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction renvoie le temps coul depuis l'appel prcdent
	       TimerDeb. Le rsultat est exprim en seconde dans un float.

	      L'instant de l'appel de cette routine est stock dans la
	      variable globale TiFin.

	      Lorsque la compensation est faite, on utilise la variable
	      globale TiOffsetTimer. Elle est initalise par la fonction
	      TimerInit.

  Entre : mode, AVANT : on ne fait pas de compensation de la dure de la
			 mesure.
		 APRES : on compense la dure de la mesure.

  Sortie : la dure dans un float.

*************************************************************************/
float TimerFin(char mode)
{
float lsb,msb;

	PTimer(&TiFin);
	lsb = (float)TiFin.lsb - (float)TiDeb.lsb;
	msb = ((float)TiFin.msb - (float)TiDeb.msb) * 65536.;
	if (mode == AVANT)
		return (msb + lsb) * 838.1058807e-9;
	else
		return (msb + lsb) * 838.1058807e-9  - TiOffsetTimer;

}


/*************************************************************************

  Procdure : Pdelay

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction effectue une attente dont la dure est spcifie
	     dans un float en secondes.
	     Les interruptions sont partiellement inhibes dans cette routine
	     au maximum pendant 1ms.

  Entre : delai : float qui contient l'attente en seconde  effectuer.

  Sortie : aucune

*************************************************************************/
void Pdelay(float delai)
{
float diff=0,fin,msb,lsb;
short i,lim;
CPT48B Deb, Fin;

	delai = delai - PDelayOffset1;
	/* Cas d'un retard d'emble infrieur  une milliseconde */
	if (delai <= 1e-3) {
		Pdisable();
		lim = (short)(delai*(float)PDelayFact);
		for(i=0 ; i<lim ; i++)
			io_delay();
		Penable();
		return;
	}

	/* Dbut sans inhiber les interruptions */
	PTimer(&Deb);
	fin = delai - 1e-3;
	while(diff < fin) {
		PTimer(&Fin);
		lsb = (float)Fin.lsb - (float)Deb.lsb;
		msb = ((float)Fin.msb - (float)Deb.msb) * 65536.;
		diff =  (msb + lsb) * 838.1058807e-9;
	}
	/* La milliseconde de la fin avec interruptions interdites */
	Pdisable();
	lim = (short)((delai-diff-TiOffsetTimer-PDelayOffset2)*(float)PDelayFact);
	for(i=0 ; i<lim ; i++)
		io_delay();
	Penable();
	return;
}


/*************************************************************************

  Procdure : TimerInit

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction initialise le compteur n0 du 8254, dtourne le
	     vecteur 0x1C pour y placer l'incrmentation de la variable
	     Pcpt et calibre les routines de mesure du temps.

  Entre : aucune

  Sortie : aucune

*************************************************************************/
void TimerInit(void)
{

long i;
float fin , msb , lsb;
CPT48B cptdeb,cptfin;

	/* Force la lecture des IRR */
	outp(0x20,0xa);

	/* Passe le timer 0 en mode 2 */
	outp(0x43,0x34);
	io_delay();
	outp(0x40,0x0);
	outp(0x40,0x0);

	/* Force le compteur d'inhibition des it  0 */
	CliCpt = 0;

	/* Installe notre propre routine dans l'it 0x1C */
	Pcpt = 0L;
	detourne(0x1C,TickPerso,OldTick);

	/* Puis on attend que a soit arriv  0 */
	Pdisable();
	SyncTimer();
	Penable();
	io_delay();


	/* Calibre la vitesse des fonctions timer */
	/* (comme l'it vient de tomber elle ne va pas nous gner) */

	/* Calibre les fonctions de mesure du temps */
	PTimer(&cptdeb);
	for(i=0; i<10; i++) {
		TimerDeb();
		TimerDeb();
                TimerDeb();
		TimerDeb();
		TimerDeb();
		TimerDeb();
                TimerDeb();
                TimerDeb();
                TimerDeb();
                TimerDeb();
        }
	PTimer(&cptfin);
	lsb = (float)cptfin.lsb - (float)cptdeb.lsb;
	msb = ((float)cptfin.msb - (float)cptdeb.msb) * 65536.;
	TiOffsetTimer = (msb + lsb) * 838.1058807e-9 / 100;


        /* Calibre la dure minimale du retard */
        PDelayFact = 1L;
	PDelayOffset1 = 0;
	Pdisable();
	TimerDeb();
	for(i=0 ; i<10; i++) {
		Pdelay(0);
		Pdelay(0);
		Pdelay(0);
		Pdelay(0);
		Pdelay(0);
		Pdelay(0);
		Pdelay(0);
		Pdelay(0);
		Pdelay(0);
		Pdelay(0);
	}
	PDelayOffset1 = TimerFin(APRES)/100;

	Penable();


        /* Calibre le facteur multiplicatif de Pdelay */
	for (i=0 ; i < 10 ; i++) {
		Pdisable();
		TimerDeb();
		Pdelay(900e-6);
		Pdelay(850e-6);
		Pdelay(900e-6);
		Pdelay(950e-6);
		Pdelay(900e-6);
		Pdelay(850e-6);
		Pdelay(900e-6);
		Pdelay(950e-6);
		Pdelay(850e-6);
		Pdelay(950e-6);
		fin = TimerFin(APRES);
		Penable();
		PDelayFact = (float)PDelayFact / fin * 900e-6 * 10;
	}

	/* On calibre l'offset des retards suprieurs  1ms
	   (pour Symantec C++ 6.1) */

	Pdisable();
	TimerDeb();
	Pdelay(1100e-6);
	Pdelay(1100e-6);
	Pdelay(1100e-6);
	Pdelay(1100e-6);
	Pdelay(1100e-6);
	Pdelay(1100e-6);
	Pdelay(1100e-6);
	Pdelay(1100e-6);
	Pdelay(1100e-6);
	Pdelay(1100e-6);
	fin = TimerFin(APRES);
	Penable();
	PDelayOffset2 = (fin-11000e-6)/10;

	return;
}

/*************************************************************************

  Procdure : TimerRestore

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction repasser le compteur n0 du 8254 en mode 3 (qui
	     est normalement son mode par dfaut) et remet en place
	     le vecteur d'interruption dtourn par TimerInit().

  Entre : aucune

  Sortie : aucune

*************************************************************************/
void TimerRestore(void)
{
	/* Repasse le timer 0 en mode 3 */
	outp(0x43,0x36);
	io_delay();
	outp(0x40,0x0);
	outp(0x40,0x0);

	/* Remet l'ancienne routine en place */
	restore(0x1C,OldTick);
	return;
}


/*************************************************************************

  Procdure : LitRTC

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction  lit un registre du MC146818A.
	     Elle ne lit que les 64 premiers registres car c'est la
	     configuration du MC146818A (les RTC actuels ont souvent
	     128 registres).
	     Elle effectue les vrifications ncessaires lorsqu'on
	     demande la lecture d'un registre horaire (les 10 premiers).

	     Les interruptions sont interdites le temps de la lecture.

  Entre : registre : numro du registre entre 0 et 63.

  Sortie : valeur lue dans un uchar. Si on demande la lecture d'un registre
	   suprieur  63 elle renvoie 0.

*************************************************************************/
uchar LitRTC(uchar registre)
{
uchar tmp;

	/* Si le numro de registre fourni est suprieur  63 on sort
	   sans rien faire */
	if (registre > 63)
		return 0;

	/* Si c'est un registre horaire on attend que le bit UIP soit  0 */
	if (registre < 10) {
		Pdisable();   /* interdit les interruptions */
		outp(0x70 , 0x8A);
		io_delay();
		while((inp(0x71) & 0x80) != 0) { /* Attend de pouvoir lire */
			  outp(0x70 , 0x8B);
			  io_delay();
		}
		outp(0x70 , registre | 0x80);  /* choisi le registre r       */
		io_delay();
		tmp = inp(0x71);          /* lit x dans le registre */
		outp(0x70 , 0);         /* Rautorise les NMI */
		Penable();              /* Remet en place les it */
	}
	else {
	/* Sinon on lit directement */
		Pdisable();                /* Disable toutes les it */
		outp(0x70 , registre | 0x80);
		io_delay();
		tmp = inp(0x71);
		outp(0x70 , 0);
		Penable();
	}
	/* Et on sort */
	return tmp;
}


/*************************************************************************

  Procdure : EcritRTC

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction  ecrit un registre du MC146818A.
	     Elle n'crit que les 64 premiers registres car c'est la
	     configuration du MC146818A (les RTC actuels ont souvent
	     128 registres).
	     Si on demande l'criture d'un registre horaire elle arrte
	     toute actualisation en cours.

	     Les interruptions sont interdites le temps de l'criture.

  Entre : registre : numro du registre entre 0 et 63.
	   val      : valeur  crire

  Sortie : aucune

*************************************************************************/
void EcritRTC(uchar registre, uchar val)
{
uchar tmp;

	/* Si le numro de registre fourni est suprieur  63 on sort
	   sans rien faire */
	if (registre > 63)
		return;


	/* Si c'est un registre horaire on arrte toute mise  jour
		en cours */
	if (registre < 10) {
		Pdisable();   /* interdit toutes les interruptions */
		outp(0x70 , 0x8B);
		io_delay();
		tmp = inp(0x71);
		outp(0x70 , 0x8B);
		io_delay();
		outp(0x71 , tmp | 0x80);/* positionne le bit SET du registre B */
		outp(0x70 , registre | 0x80);  /* choisi le registre */
		io_delay();
		outp(0x71 , val);         /* crit val dans le registre */
		outp(0x70 , 0x8B);        /* remet en place le registre B  */
		io_delay();
		outp(0x71 , tmp);         /* efface le bit SET du registre B */
		outp(0x70 , 0);           /* Rautorise les NMI */
		Penable();                /* Remet en place les it */
	}
	else {
	/* Sinon on crit directement */
		Pdisable();             /* interdit les interruptions */
		outp(0x70 , registre | 0x80);
		io_delay();
		outp(0x71 , val);         /* crit val dans le registre */
		outp(0x70 , 0);         /* Rautorise les NMI */
		Penable();              /* Remet en place les it */
	}
}


/*************************************************************************

  Procdure : remise_a_l_heure

  Auteur / Date : Pierre Larbier 13/2/1995

  Fonction : Cette fonction  remet le DOS  l'heure.
	     Elle est utilise si on a dtourne inconsidrment
	     l'interruption 0x8 en faisant perdre l'heure (et
	     accessoirement la date) au DOS.

	     Cette routine lit l'heure et la date dans le RTC pour remettre
	     le DOS  jour.

	     La prcision est de 1 seconde.

  Entre : aucune

  Sortie : aucune

*************************************************************************/
short remise_a_l_heure(void)
{
long timerdeb,timersys;
short lecture;
union REGS inreg, outreg;
struct _dosdate_t date;
struct _dostime_t heure;


	/* Commence par se synchroniser sur le timer systme, ce sera plus
	   facile pour grer le timeout (je suis fnant ici...) */
	Pdisable();
	SyncTimer();
	TimerDeb();
	Penable();
	io_delay();
	/* Voil, maintenant on a 54.9ms pour faire la mise  jour de
	   l'heure. Si on arrive pas  lire l'horloge sauvegarde dans cet
	   intervalle de temps, c'est qu'elle a un problme grave (horloge
	   arrte ou pile dcharge).
	   On va recommencer jusqu' obtenir un couple heure-date valide */
	_bios_timeofday(_TIME_GETCLOCK,&timerdeb);

        do {
		/* Lit la date dans l'horloge sauvegarde */
                do {
			/* Lit la date dans l'horloge sauvegarde */
                        inreg.h.ah = 4;
                        int86(0x1A,&inreg,&outreg);

			/* Teste le timeout */
                        _bios_timeofday(_TIME_GETCLOCK,&timersys);
			if (timersys != timerdeb)
                                return ERREUR;
		} while (outreg.x.cflag != 0);

                /* Maintenant on a dans outreg la date au format BCD, on la
                   met en forme pour le DOS */
                date.day = 10*(outreg.h.dl >> 4) + (outreg.h.dl & 0x0f);
		date.month = 10*(outreg.h.dh >> 4) + (outreg.h.dh & 0x0f);
		date.year = 10*(outreg.h.cl >> 4) + (outreg.h.cl & 0x0f);
                if (outreg.h.ch == 32)
			date.year += 2000;
		else
			date.year += 1900;

                /* Lit l'heure dans l'horloge sauvegarde */
                do {
                        /* Lit l'heure dans l'horloge sauvegarde */
			inreg.h.ah = 2;
                        int86(0x1A,&inreg,&outreg);

			/* Teste le timeout */
			_bios_timeofday(_TIME_GETCLOCK,&timersys);
                        if (timersys != timerdeb)
                                return ERREUR;
		} while (outreg.x.cflag != 0);

		/* Maintenant on a dans outreg l'heure au format BCD, on la
		   met en forme pour le DOS */
		heure.hour = 10*(outreg.h.ch >> 4) + (outreg.h.ch & 0x0f);
		heure.minute = 10*(outreg.h.cl >> 4) + (outreg.h.cl & 0x0f);
		heure.second = 10*(outreg.h.dh >> 4) + (outreg.h.dh & 0x0f);
		/* On se met  50 centimes pour diminuer l'erreur */
		heure.hsecond = 50;

                /* On vrifie que la date n'a pas chang entre temps (on
		   aurait vraiment pas de chance l ... lancer cette routine
                    minuit pile ...) */

                do {
			/* Lit la date dans l'horloge sauvegarde */
                        inreg.h.ah = 4;
                        int86(0x1A,&inreg,&outreg);

                        /* Teste le timeout */
                        _bios_timeofday(_TIME_GETCLOCK,&timersys);
			if (timersys != timerdeb)
                                return ERREUR;
		} while (outreg.x.cflag != 0);

                if (date.day != (10*(outreg.h.dl>>4) + (outreg.h.dl&0x0f)))
                        lecture = ERREUR;
                else
			lecture = OK;

        } while (lecture == ERREUR);

	/* Voil, on est au point maintenant, on demande au DOS de se mettre
	    l'heure. On ne teste pas qu'on se trouve dans les 2 secondes
	   qui prcdent un changement de jour pour 2 raisons :
	   - je suis fnant
           - a ne pose rellement pb qu'avec le MC146818A de Motorola et
             il y a beau temps que ce circuit n'est plus sur les cartes-mres
	     (sur mon PC c'est un DS12887 de Dallas par exemple).
        */
	_dos_setdate(&date);
	_dos_settime(&heure);

        /* Et c'est fini */
	return OK;
}



/*************************************************************************/
/*   Fonction de dmonstration de l'utilisation du 8254 et MC164818A     */
/*************************************************************************/


/*
     Analyse le mode de fonctionnement des 3 compteurs du 8254
*/

void TimerAnalyseMode(void)
{
unsigned char status, i, val;
unsigned short j;

        /* Lit les status des 3 compteurs */
	outp(0x43,0xee);
        io_delay();

        /* Lit et analyse le mode des trois compteurs */
        for (i=0 ; i<3 ; i++) {
                status = inp(0x40 + i);


                printf("Etat du compteur n%d\n",(short)i);
		printf("********************\n");

                if ((status & 1) == 0)
			printf("Le compteur est en mode binaire\n");
                else
			printf("Le compteur est en mode BCD\n");

                printf("Le compteur est programm en mode %d\n",
                                (status >> 1) & 0x7);

		printf("Le dernier mode d'accs programm est ");
                switch((status >>4) & 0x3) {
			case 0 : printf("une commande de latch\n");
				break;
			case 1 : printf("le poids faible seulement\n");
				break;
                        case 2 : printf("le poids fort seulement\n");
				break;
                        case 3 : printf("le poids faible puis fort\n");
                                break;
		}
		printf("\n");
	}

	/* Recherche la frquence du compteur n1 */
	/* Avec 10 000 lectures, j'ai toujours russi  trouver la valeur
           maximale */
	i = 0;
        for(j=0 ; j<10000; j++) {
                /* latche les donnes du compteur n1 */
		outp(0x43,0x40);
		io_delay();
		val = inp(0x41);
		if (val > i)
			i = val;
	}
	printf("la valeur maximale lue dans le compteur n1 est %d\n",
			(short)i);


	/* Est'il possible de changer le mode du compteur n0 ? */
	/* Lit le mode du compteur n0 */
	outp(0x43,0xe2);
	io_delay();
	status = inp(0x40);
	io_delay();
	/* Il n'y a que 2 modes possibles  l'allumage du PC : 2 ou 3
	   avec 65536 (0) comme valeur programme. On va passer en BCD
	   et regarder si a marche */
	outp(0x43,(status & 0xf) | 0x31);
	io_delay();
	outp(0x40,0);
	io_delay();
	outp(0x40,0);
	io_delay();

	/* Puis on lit le mode programm */
	outp(0x43,0xe2);
	io_delay();
	val = inp(0x40);
	/* On est pass en BCD ?? */
	if ((val & 1) == 1)
		printf("Il est possible de ");
	else
		printf("Il n'est pas possible de ");
	printf("changer le mode de fonctionnement du compteur n0\n");
	/* Et on revient en mode normal */
	outp(0x43,(status & 0xf) | 0x30);
	io_delay();
	outp(0x40,0);
	io_delay();
	outp(0x40,0);

	return;
}



/*
     Change la couleur du fond de l'cran  chaque VBL (dans le bleu)
*/
void __far fct_vbl(void)
{
	CoulFond = (CoulFond + 1) & 63;
	outp(0x3c8 , 0);
	outp(0x3c9 , 0x0);
	outp(0x3c9 , 0x0);
	outp(0x3c9 , CoulFond);
}


/*
     Synchronise une fonction avec la VBL
*/
void __interrupt __far ItVbl(void)
{
	/* Attend la synchro trame */
	while((inp(0x3da)&8)==0);

	/* Relance le compteur n0 en mode 0 */
	outp(0x43,0x30);
	outp(0x40,(uchar)(DureeVbl & 0xff));
	outp(0x40,(uchar)((DureeVbl >> 8) & 0xff));

	/* Appelle notre propre fonction  chaque VBL */
	fct_vbl();

	/* Appelle l'ancienne interruption 0x8 si c'est son heure */
	AccVBL = AccVBL + IncVBL;
	if (AccVBL >= 16777216L) {
		AccVBL -= 16777216L;
		_chain_intr(OldTick);
	}
	/* Et sort en acquitant l'interruption */
	outp(0x20,0x20);
	return;
}


/*
     Calcule la dure du BEEP avec l'interruption 0x1C
*/
void __interrupt __far ItUserTick(void)
{
      if (CptSound > 0)
		CptSound--;
	else
		outp(0x61,inp(0x61) & ~3);

	_chain_intr(OldUserTick);
}

/*
     Fonction appelle par l'interruption d'actualisation du RTC
*/
void __far ItUF(void)
{
	DureeDemo++;
	CptSound = 3;
	outp(0x43,0xb6);
	outp(0x42,0);
	outp(0x42,6);
	outp(0x61,inp(0x61) | 3);


}

/*
     Fonction appelle par l'interruption priodique du RTC
*/
ushort pos = 1;
void __far ItPF(void)
{
	/* Gnrateur pseudo-alatoire de Lehmer  rjection (cf Reporter n4)
	   tout l'cran sera couvert de X au bout de 2048 appels de cette
	   routine sous it */
	do {
		pos = ((pos<<2) + pos + 1) & 4095;
	} while(pos > 4000);
	/* Affiche le X dans un cran en mode texte (je ne vrifie pas que
	   l'cran est bien en 0xb8000 , la flemme ...) */
	*(char _far *)MK_FP(0xb800,pos & ~1) = 'X';
}


/*
     Fonction appelle par l'interruption d'alarme du RTC
*/
void __far ItAF(void)
{
	DemoFlag = 1;
}


/*
     Dtermine la source d'une interruption RTC (vecteur 0x70)
*/
void __interrupt __far ItRTC(void)
{
uchar src;
	/* Lit le registre 0xC pour appeller la source de l'it */
	src = LitRTC(0xc);
	do {
		if ((src & 0x10) != 0)
			ItUF();

		if ((src & 0x20) != 0)
			ItAF();

		if ((src & 0x40) != 0)
			ItPF();
		/* Une autre it RTC est tombe entre temps ?? Attention !! */
		src = LitRTC(0xc) & 0x70;
	} while (src != 0);

	/* Et on sort. Il n'est pas ncessaire d'appeller l'ancien
	gestionnaire car on a lu le registre 0xC et cel l'a remis
	 0, par consquent, l'ancien gestionnaire ne pourrait plus
	dterminer la source de l'it. */
	outp(0xA0,0x20);
	outp(0x20,0x20);
	return;
}



/*************************************************************************

	Petit programme de dmonstration de l'usage de quelques routines
	 de ce fichier.

*************************************************************************/

int main(void)
{
short i;
volatile float diff;
volatile uchar itmask, sec,proc;

#ifdef __SC__
	disp_open();
#endif


	/* O on crit la petite prsentation */
	printf("\n\n");
	printf("Petit exemple de l'utilisation des routines timer\n");
	printf("**************************************************\n\n");


	/* Examine le fonctionnement des 3 compteurs */
	TimerAnalyseMode();
	getch();
	/* Initialisation du systme de timer */
	TimerInit();

	/* Mesure de l'it timer 0 */
	itmask = inp(0x21); /* on ne va pas se laisser ennuyer par d'autres it */
	outp(0x21,0xfe);
	Pdisable();
	SyncTimer();
	TimerDeb();
	Penable();
	io_delay();
	diff= TimerFin(AVANT);
	outp(0x21,itmask);
	printf("\nDure de l'it timer (IRQ0) : %lds\n",(long)(diff*1e6));


	/* Mesure de la frquence vido sur 16 VBL */
	/* (l'interruption n0 venant de tomber, elle ne va pas nous gner au
		dbut) */
	SyncVert();
	TimerDeb();;
	for(i=0 ; i<15; i++) {
		SyncVert();
		io_delay();
	}
	Pdisable();
	SyncVert();
	diff = TimerFin(AVANT);
	Penable();
	printf("Frquence image de la carte vido : %.3f 0.001Hz\n",16/(diff));
	/* Calcule la dure pour se synchroniser tout  l'heure */
	DureeVbl = diff/(16*838.1058807e-9) - 119; /*on laisse 100s de marge*/
	IncVBL = (long)(16 * diff/838.1058807e-9);

	/* Dure de la ligne sur 10 lignes */
	SyncVert();
	Pdisable();
	SyncLigne();
	TimerDeb();
	for(i=0 ; i<10 ; i++) {
		SyncLigne();
		io_delay();
	}
	diff = TimerFin(AVANT);
	Penable();
	printf("Frquence ligne de la carte vido : %.1f kHz\n",0.01/diff);


	/* Gnration d'un retard de 500s */
	Pdisable();
	TimerDeb();
	Pdelay(500e-6);
	diff = TimerFin(APRES);
	Penable();
	printf("On a demand une pause de 500s, on a mesur : %lds\n",(long)(diff*1e6));


	/* Mesure du temps d'execution d'un seul printf */
	TimerDeb();
	printf("L'affichage de ce texte a pris ");
	diff = TimerFin(APRES);
	printf("%lds\n",(long)(diff*1e6));

	/* Remet tout en place */
	TimerRestore();


	/* Informations sur la configuration */
	proc = CPUTYPE();
	printf("\n");
	switch(proc) {
		case Proc8086 :
			puts("Ce programme est execut sur un 8086");
			break;
		case Proc80286 :
			puts("Ce programme est execut sur un 80286");
			break;
		case Proc80386 :
			puts("Ce programme est execut sur un 80386");
			break;
		case Proc80486 :
			puts("Ce programme est execut sur un 80486");
			break;
		case ProcP5 :
			puts("Ce programme est execut sur un Pentium");
			break;
		case ProcInconnu :
			puts("Ce programme est execut sur un processeur inconnu");
			break;

	}
	if (proc != ProcP5)
		puts("On ne pourra pas utiliser le timer du Pentium (bien sr !!)");
	else {
		if (TesteTimerPentium() == OK)
			puts("On peut utiliser le timer du Pentium");
		else
			puts("l'environnement ne permet pas d'utiliser le timer du Pentium");
	}

	getch();



	/****************************************/
	/*             NANO-DEMO                */
	/****************************************/

	/* On s'amuse a changer les couleurs sous interruption VBL pendant
	   20 secondes. Cette dure est programme par le RTC.
	   Dans le mme temps, on rempli l'cran de 'X' alatoirement sous
	   interruption priodique du RTC et on dclenche un beep sous
	   interruption d'actualisationdu RTC.
	   La dure du beep sonore est gre sous interruption 0x1C */

	puts("\nVous allez voir les 5 sources d'interruptions travailler en mme temps");
	puts("1- Alarme RTC : fixe la dure de la nano-dmo  20 secondes");
	puts("2- Actualisation RTC : dclenche les beep sonores et indique la dure");
	puts("3- Interruption priodique RTC : rempli l'cran de 'X'  128Hz");
	puts("4- Interruption VBL : change la couleur du fond de l'cran en bleu");
	puts("5- Interruption 18.2 fois par seconde : dtermine la dure des beeps");
	puts("En tche d'avant-plan on affiche la dure de la nano-dmo");
	puts("\nTapez sur une touche pour commencer");
	getch();

	/* Positionne le flag d'arrt */
	DemoFlag = 0;
	/* Positionne le compteur de dure du son */
	CptSound = 0;
	/* Positionne le l'accumulateur de dure de la VBL */
	AccVBL = 0;
	/* La dure de la dmo est nulle pour le moment */
	DureeDemo = 0;
	/* Commence avec un fond d'cran noir */
	CoulFond = 0;

	/* Fait croire qu'une attente est en cours */
	*(uchar _far *)MK_FP(0x40,0xa0) |= 0x1;
	*(uchar _far *)MK_FP(0x40,0xa0) &= ~0x80;
	/* Interdit les it du RTC (bit 4,5,7 du registre B) */
	EcritRTC(0xB,LitRTC(0xB) & ~0x70);
	/* Programme l'alarme pour 20 secondes + tard */
	sec = LitRTC(0x0); /* seconde actuelle en BCD */
	sec += 0x20;       /* calcule 20 secondes + tard en BCD */
	if (sec >= 0x60)
		sec -= 0x60;
	EcritRTC(0x1, sec);
	EcritRTC(0x3, 0xff);
	EcritRTC(0x5, 0xff);

	/* Programme l'interruption priodique  128Hz */
	EcritRTC(0xA,0x21);

	/* Dtourne le vecteur 0x70 */
	detourne(0x70,ItRTC,OldRTC);
	/* Autorise les it du RTC aprs avoir raz le registre C */
	LitRTC(0xC);
	EcritRTC(0xB,LitRTC(0xB) | 0x70);
	/* Dmasque les it sur les 8259 (pas utile normalement mais sait-on
		jamais ...) */
	outp(0xA1,inp(0xA1) & ~1);
	outp(0x21,inp(0x21) & ~4);

	/* Dtourne les interruptions 0x1C et 0x8 */
	detourne(0x1C,ItUserTick,OldUserTick);
	detourne(0x8,ItVbl,OldTick);

	/* Lance le compteur n0 en mode 0 (pour se synchroniser sur la VBL) */
	outp(0x43,0x30);
	outp(0x40,(uchar)(DureeVbl & 0xff));
	outp(0x40,(uchar)((DureeVbl >> 8) & 0xff));

	/* Efface le curseur (c'est plus joli */
	HIDECUR();
	/* C'est parti pour 20 secondes */
	while(DemoFlag == 0) {
	      printf("Il reste %d secondes ...\r",20-DureeDemo);
	}

	/* Remet le curseur */
	SHOWCUR();
	/* Attend la fin du beep pour finir proprement
		(au niveau des oreilles) */
	while (CptSound != 0);

	/* Remet en place les registres du RTC */
	/* Interdit l'it d'alarme (bit 5 du registre B) */
	EcritRTC(0xB,LitRTC(0xB) & ~0x70);
	LitRTC(0xC);
	/* Programme l'interruption priodique  1024Hz */
	EcritRTC(0xA,0x26);
	/* remet en place les 2 vecteurs d'interruption dtourns */
	restore(0x70,OldRTC);
	restore(0x1C,OldUserTick);
	restore(0x8,OldTick);
	/* remet l'indicateur du BIOS en place */
	*(uchar _far *)MK_FP(0x40,0xa0) &= ~0x1;
	*(uchar _far *)MK_FP(0x40,0xa0) |= 0x80;

	/* Repasse le compteur n0 en mode 3 car la routine d'interruption a
	   travaill en mode 0*/
	outp(0x43,0x36);
	io_delay();
	outp(0x40,0x0);
	outp(0x40,0x0);

	/* Remet les couleurs normales */
	outp(0x3c8 , 0);
	outp(0x3c9 , 0);
	outp(0x3c9 , 0);
	outp(0x3c9 , 0);

	/* Arrte le son */
	outp(0x61,inp(0x61) & ~3);


#ifdef __SC__
	disp_close();
#endif

	puts("\nEt c'est fini ....\n");

	return 1;
}
