/*      Ensemble de routines de gestion de nombres pseudo-alatoires

****************************************************************************

Auteur          : Pierre Larbier
Dvelopp le    : 3.4.1994
Dernire MAJ    : 3.4.1994


Toutes les routines de ce fichier sont librement utilisables (si vous les
employez dans un programme commercial, envoyez-moi un petit mot ... a fait
plaisir). Elles ont t testes mais c'est sans aucune garantie ...

***************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <time.h>
#include <dos.h>
#include <math.h>

/* Prototypes des fonctions de ce fichier */
/******************************************/

/* Implmentation du Standard Minimal de Park & Miller */
long alea_stdmin(void);
/* Gnrateur de Mitchell & Moore sur 32 bits */
unsigned long alea_Mitchell_Moore(void);
/* Gnrateur polynomial x^15+x^1+1 */
char alea_bit(void);

/* Gnrateur uniforme, sortie dans l'intervalle [0,1[ */
#define alea_uniforme()  (double)alea_Mitchell_Moore()/4294967296.
/* Gnrateur normal */
double alea_normal(double moyenne , double ecart_type);
/* Gnrateur exponentiel */
double alea_exp(double moyenne);

/* Fixe un nouveau point de dpart "alatoire" pour tous les gnrateurs */
void alea_randomize(void);
/* Choisi un nouveau point de dpart pour tous les gnrateurs */
void alea_srand(unsigned long depart);
/* Fixe la borne suprieure pour la fonction alea_random() */
void alea_fixe_borne(unsigned long max);
/* Gnre un nombre entier infrieur  une borne fixe */
unsigned long alea_random(void);




/* Variables globales utilises par les divers gnrateurs , elles sont
        initialises correctement */
long alea_var = 933757703L;
unsigned long alea_tabMM[64] =
        { 2002705692L , 1963366013L , 58860689L , 1429122403L , 1803119173L,
          1882197794L , 1664203448L , 1440332008L , 1224389472L , 1125550350L ,
          2088769674L , 1036733409L , 1843576952L , 1103773348L , 1154917050L ,
          1733657764L , 527917052L , 1446947207L , 736889421L , 362306498L ,
          1169172641L , 809207237L , 332095808L , 224246503L , 77175436L ,
          7430064L , 323034122L , 395828838L , 1938425507L , 1790571159L ,
          1441123902L , 1648850048L , 1093775848L , 630659016L , 1654283967L,
          79855660L , 2104281892L , 1905060048L , 1510533613L , 2134242504L ,
          794408887L , 724330410L , 1883889674L , 34859550L , 1768904866L ,
          220473794L , 1093764683L , 443008861L , 324122678L , 1511320354L ,
          324612962L , 1161588954L , 51715001L , 1590628419L , 1815400277L ,
          2132282610L , 66725134L , 462863404L , 1159461594L , 804397480L ,
          1098888495L , 659571265L , 103665041L , 689106370L };
short alea_ptMM=55, alea_pt55MM=0 , alea_pt24MM=31;
short alea_polyreg = 1;
double alea_sav;
short alea_etat = 0;
unsigned long alea_borne = 4294967295L;
short alea_dec = 0;
unsigned long alea_init = 0;

/*************************************************************************

  Procdure : alea_stdmin

  Auteur / Date : Pierre Larbier 3/4/1994

  Fonction : Produit un nombre pseudo-alatoire entier en utilisant le
                gnrateur de Park & Miller :
                        x(n+1) = x(n)*16807 mod (2^31-1)

                Les nombres produits sont des long dans l'intervalle
                [1,2^31-2].
                La priode du gnrateur est 2^31-2.

                L'tat du gnrateur est maintenu dans la variable globale
                alea_var (qui ne doit jamais valoir 0).

                L'algorithme de calcul ne ncessite ni division ni
                multiplication par un nombre de plus de 16 bits. La routine
                est donc facilement codable en asm sur la plupart des
                machines.

  Entres : aucune

  Sortie  : le nombre pseudo-alatoire produit.

*************************************************************************/
long alea_stdmin(void)
{
long haut , bas , inter;

        bas = 16807 * (alea_var & 0xffffL);
        haut = 16807 * (alea_var >> 16);
        inter = (bas >> 16) + haut;
        bas = ((bas & 0xffff) | ((inter & 0x7fff)<<16)) + (inter >> 15);
        if ((bas & 0x80000000L) != 0)
                bas = (bas + 1) & 0x7fffffffL;

        alea_var = bas;
        return bas;
}


/*************************************************************************

  Procdure : alea_Mitchell_Moore

  Auteur / Date : Pierre Larbier 3/4/1994

  Fonction : Produit un nombre pseudo-alatoire entier en utilisant le
                gnrateur de Mitchell & Moore :
                        x(n) = [x(n-24) +  x(n-55)] mod 2^32

                Les nombres produits sont des long dans l'intervalle
                [0,2^32-1].
                La priode du gnrateur est un multiple de 2^55-1.

                L'tat du gnrateur est maintenu dans le tableau
                alea_tabMM[64] ainsi que dans les variables alea_ptMM,
                alea_pt24MM et alea_pt55MM.

                Il ne m'a pas sembl utile de rajouter un mlange
                supplmentaire des donnes produites comme certains le
                conseillent (mais sans en donner de justification).

                Afin d'viter un test, on utilise un tableau de 64 entres
                au lieu des 55 qu'on trouve habituellement.

                ATTENTION : on utilise le fait que le cast implicite du
                rsultat de l'addition en unsigned long ralise le modulo
                2^32 (je n'ai jamais vu de compilateur C pour lequel ce ne
                soit pas vrai).

  Entres : aucune

  Sortie  : le nombre pseudo-alatoire produit.

*************************************************************************/
unsigned long alea_Mitchell_Moore(void)
{
unsigned long x;
        alea_tabMM[alea_ptMM] = alea_tabMM[alea_pt24MM] +
                                alea_tabMM[alea_pt55MM];
        x = alea_tabMM[alea_ptMM];
        alea_ptMM   = (alea_ptMM + 1) & 63;
        alea_pt24MM = (alea_pt24MM + 1) & 63;
        alea_pt55MM = (alea_pt55MM + 1) & 63;

        return x;
}


/*************************************************************************

  Procdure : alea_bit

  Auteur / Date : Pierre Larbier 3/4/1994

  Fonction : Produit un bit alatoire en utilisant le polynome gnrateur :
                x^15 + x + 1.

                La priode du gnrateur est de 2^15-1.

                L'tat du polynome est maintenu dans la variable
                alea_polyreg (qui ne doit jamais valoir 0).

  Entres : aucune

  Sortie  : un char valant 1 ou 0.

*************************************************************************/
char alea_bit(void)
{
short x;
        x = alea_polyreg & 0x4001;
        alea_polyreg = (alea_polyreg << 1) & 0x7fff;
        if ((x == 0) || (x == 0x4001)) {
                alea_polyreg |= 0;
                return 0;
        }
        else {
                alea_polyreg |= 1;
                return 1;
        }
}


/*************************************************************************

  Procdure : alea_normal

  Auteur / Date : Pierre Larbier 3/4/1994

  Fonction : Produit un nombre alatoire flottant suivant une distribution
                normale avec l'algorithme de Box-Muller-Marsaglia (voir
                D.Knuth, "Seminumerical Algorithms" p. 117)

                Le gnrateur uniforme servant de base est le Mitchell-Moore.

                Afin de limiter les calculs, on conserve le rsultat
                intermdiaire dans la variable globale alea_sav et la
                position du gnrateur dans alea_etat.

  Entres : moyenne : la moyenne des nombres dsire.
            ecart_type : l'cart_type des nombres dsir.

  Sortie  : le nombre pseudo-aleatoire.

*************************************************************************/
double alea_normal(double moyenne , double ecart_type)
{
double v1 , v2 , x;
        if (alea_etat == 0) {
                do {
                        v1 = 2.*alea_uniforme() - 1.;
                        v2 = 2.*alea_uniforme() - 1.;
                        x = v1*v1 + v2*v2;
                } while ((x >= 1.) || (x == 0.));
                x = sqrt(-2.*log(x)/x);
                alea_sav = x*v1;
                x *= v2;
        }
        else
                x = alea_sav;
        alea_etat ^= 1;
        return (x*ecart_type + moyenne);
}


/*************************************************************************

  Procdure : alea_exp

  Auteur / Date : Pierre Larbier 3/4/1994

  Fonction : Produit un nombre alatoire flottant suivant une distribution
                exponentielle.

                Le gnrateur uniforme servant de base est le Mitchell-Moore.


  Entres : moyenne : la moyenne des nombres dsire.

  Sortie  : le nombre pseudo-aleatoire.

*************************************************************************/
double alea_exp(double moyenne)
{
double x;
        do {
                x = alea_uniforme();
        } while (x == 0.);

        return (-moyenne*log(x));
}


/*************************************************************************

  Procdure : alea_randomize

  Auteur / Date : Pierre Larbier 3/4/1994

  Fonction : Fixe un nouveau point de dpart "alatoire" pour tous les
                gnrateurs de ce fichier.

                ATTENTION : cette procdure n'est pas portable. En l'tat
                elle ne fonctionne QUE sur un compatible PC. Pour choisir le
                point de dpart, on utilise le contenu du timer 0 du 8254.
                L'horloge qui l'anime est  1.193167MHz donc sa valeur 
                l'instant du premier appel de la routine est "alatoire".

                Aux appels suivants, on fait tourner un gnrateur Park &
                Miller pour produire les nouvelles graines des gnrateurs.

                On peut donc produire 2^31-2 squences diffrentes.

  Entres : aucune

  Sortie  : aucune

*************************************************************************/
void alea_randomize(void)
{
unsigned long lb , hb;

        while (alea_init == 0) {
                /* les 16 bits de poids faible sont donns par le contenu
                du timer 0, les 16 bits de poids fort par le temps en
                seconde depuis le 1/1/1970 (ou 1968 a dpend des
                compilateurs) */
                outp(0x43,0);
                asm pushf
                asm cli
                alea_init = (unsigned long)time(NULL);
                lb = (unsigned long)inp(0x40);
                hb = (unsigned long)inp(0x40);
                asm popf

                alea_init = ((alea_init << 16) | (hb << 8) | lb) & 0x7fffffffL;
        }

        alea_var = alea_init;
        alea_init = alea_stdmin();
        alea_srand(~alea_init);
}


/*************************************************************************

  Procdure : alea_srand

  Auteur / Date : Pierre Larbier 3/4/1994

  Fonction : Fixe un nouveau point de dpart pour tous les gnrateurs
                de ce fichier. La graine est fournie par l'utilisateur.

                Le gnrateur Mitchell & Moore est initialis par le Park &
                Miller.
                On effectue pas le test pour s'assurer que les 55 premiers
                nombres ne sont pas tous pairs parce que j'ai vrifi que le
                Standard Minimal ne peut pas produire 55 nombres pairs de
                rang.
                On peut donc produire 2^31-2 squences diffrentes.

                Si la graine fournie vaut 0, on prend 1  la place pour ne
                pas "coincer" les gnrateurs.

  Entres : depart : point de dpart de tous les gnrateurs :
                les 15 bits de poids faible sont pris pour le gnrateur
                        polynomial
                les 31 bits de poids faible sont pris pour le Park & Miller.
                le Mitchell & Moore est initialis par le Park & Miller.
                le Box-Muller-Marsaglia est remis  0.

  Sortie  : aucune

*************************************************************************/
void alea_srand(unsigned long depart)
{
unsigned long x;
short i;

        /* Gnrateur polynomial */
        x = depart & 0x7fff;
        if (x == 0)
                x = 1;
        alea_polyreg = (short)(x);

        /* Standard Minimal */
        x = depart & 0x7fffffffL;
        if (x == 0)
                x = 1;
        alea_var = (short)(x);

        /* Etat du Box-Muller-Marsaglia */
        alea_etat = 0;

        /* Etat du Mitchell-Moore */
        alea_ptMM   = 55;
        alea_pt55MM = 0;
        alea_pt24MM = 31;
        for (i=0 ; i<64 ; i++)
                alea_tabMM[i] = alea_stdmin();
        /* Je ne fait pas le test pour vrifier que les 55 premiers nombres
        ne sont pas tous pairs car dans le cas du Standard Minimal cela ne
        peut pas arriver (j'ai fait le test systmatique). */

        alea_var = (short)(x);
}


/*************************************************************************

  Procdure : alea_fixe_borne

  Auteur / Date : Pierre Larbier 3/4/1994

  Fonction : Fixe la borne suprieure des nombres fournis par alea_random().
                Si la borne vaut 0, on prend 2^32-1  la place.

                C'est galement la valeur par dfaut avant le premier appel
                de cette procdure.

  Entres : max : valeur maximale que pourront prendre les nombres fournis
                 par alea_random().

  Sortie  : aucune

*************************************************************************/
void alea_fixe_borne(unsigned long max)
{

        if (max == 0)
                max = 4294967295L;

        alea_borne = max;
        alea_dec = 0;

        while ((max & 0x80000000L) == 0) {
                alea_dec++;
                max = max << 1;
        }
}


/*************************************************************************

  Procdure : alea_random

  Auteur / Date : Pierre Larbier 3/4/1994

  Fonction : dlivre un nombre entier compris entre 0 et la borne suprieure
                initialise dans alea_fixe_borne() inclus.

                Cette procdure utilise la mthode de rjection au lieu de
                faire une division.

                Si la borne suprieure n'est pas initialise avec
                alea_fixe_borne(), la valeur max par dfaut est 2^32-1.

  Entres : aucune

  Sortie  : le nombre pseudo-alatoire.

*************************************************************************/
unsigned long alea_random(void)
{
unsigned long x;

        do {
                x = alea_Mitchell_Moore() >> alea_dec;
        } while (x > alea_borne);

        return x;
}


/*************************************************************************

        Petit programme de dmonstration de l'usage de quelques routines
         de ce fichier.

*************************************************************************/

void main(void)
{
long i , j , k , d1 , d2 , d3 ,max , min , des[216];
double x , y;


        /* O on crit la petite prsentation */
        printf("\n\n");
        printf("Exemple d'utilisation des routines de gnration de nombres pseudo-alatoires\n");
        printf("*****************************************************************************\n");
        printf(" Attention : les tests effectus ne correspondent en rien  ceux qui sont\n");
        printf("             faits pour valider un gnrateur pseudo-alatoire\n\n");
        /* L'aiguille de Buffon */
        printf("On va tester le gnrateur uniforme en simulant 100 000 lancs d'aiguilles\n");
        printf(" de Buffon. Le rsultat obtenu doit s'approcher de PI. \n");

        for (j=0 , i=0 ; i<100000L ; i++) {
                x = sin(alea_uniforme()*M_PI_2);
                y = 2*alea_uniforme();
                if ( ((x+y) >= 2) || ((y-x) <= 0) )
                        j++;
        }
        printf("Rsultat de la 'simulation' : %.5f\n\n",2*100000./(double)j);
        printf("Tapez sur une touche pour continuer\n");
        getch();
        printf("\n\n");

        /* Les ds sont-ils pips ?? */
        printf("A prsent, on simule 33 333 lancs de 3 ds.\n");
        for (i=0 ; i<216 ; i++)
                des[i] = 0;

        alea_randomize();
        alea_fixe_borne(5);
        for (i=0 , j=0 ; i<33333L ; i++) {
                j = alea_random();
                j += 6*alea_random();
                j += 36*alea_random();
                des[j]++;
        }
        j = 33334L;
        k = -1;
        for (i=0 ; i<216 ; i++) {
                if (des[i] > k) {
                        max = i;
                        k = des[i];
                }
                if (des[i] < j) {
                        min = i;
                        j = des[i];
                }
        }
        d1 = min/36;
        d2 = (min-d1*36)/6;
        d3 = min%6;
        printf("On a obtenu l'occurence minimale de %ld lancs avec la combinaison %ld-%ld-%ld\n",j, 1+d1 , 1+d2 , 1+d3);
        d1 = max/36;
        d2 = (max-d1*36)/6;
        d3 = max%6;
        printf("On a obtenu l'occurence maximale de %ld lancs avec la combinaison %ld-%ld-%ld\n",k, 1+d1 , 1+d2 , 1+d3);
        printf("La moyenne tant de 154.3, un tel cart vous inquite sur autant de lancs ?\n");
        printf("On se fait un petit Chi2 pour voir sur les 33 333 tirages prcdents?\n");

        for(i=0 , x=0 ; i<216 ; i++)
                x += (double)des[i]*(double)des[i];
        x = 216.*x/33333. - 33333.;

        printf("Le gnrateur est suspect si le Chi2 obtenu : %.2f n'est pas\n",x);
        printf("dans l'intervalle 182.1 - 250.1 (intervalle de confiance 5%% - 95%%).\n\n");
        printf("P.S : comme on a fait un randomize() avant les lancs , je ne peux pas\n");
        printf(" prvoir le Chi2 calcul 4 lignes plus haut. Si jamais il sort ou tangente\n");
        printf(" les 2 bornes de l'intervalle de confiance, pas de panique : a peut\n");
        printf(" arriver (1 fois sur 10 en moyenne).\n\n");
        printf("Tapez sur une touche pour continuer\n");
        getch();
        printf("\n\n");

        /* Ecart-type du gnrateur uniforme sur 100 000 tirages */
        for (x = 0, i = 0 ; i<100000L; i++) {
                y = alea_uniforme() - 0.5;
                x += y*y;
        }
        printf("Sur cent mille tirages, on a obtenu une variance\n");
        printf(" de 1/%.5f avec le gnrateur uniforme (idalement 1/12)\n\n",100000./x);

        /* Moyenne du gnrateur exponentiel sur 100 000 tirages */
        for (x = 0, i = 0 ; i<100000L; i++)
                x += alea_exp(5.);
        printf("Sur cent mille tirages, on a demand une moyenne de 5. au\n");
        printf(" gnrateur  distribution exponentielle. On a mesur : %.5f\n\n",x/100000.);

        /* Ecart-type du gnrateur normal sur 100 000 tirages */
        for (x = 0, i = 0 ; i<100000L; i++) {
                y = alea_normal(0 , 5.);
                x += y*y;
        }
        printf("Sur cent mille tirages, on a demand un cart-type de 5. au\n");
        printf(" gnrateur  distribution normale. On a mesur : %.5f\n\n",sqrt(x/100000.));

        printf("\nEt c'est fini\n");
}
