/***************************************************************************
				  CALENDE.C


	Ensemble de routines de gestion de date et de calendrier.

****************************************************************************

Auteur          : Pierre Larbier
Dvelopp le    : 7.12.1993
Dernire MAJ    : 21.11.1994

Toutes les routines de ce fichier sont librement utilisables (si vous les
employez dans un programme commercial, envoyez-moi un petit mot ... a fait
plaisir). Elles ont t testes mais c'est sans aucune garantie ...

***************************************************************************/
#include <conio.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

/* Le header dos.h n'est inclus QUE pour le main() de dmonstration des
   routines de ce fichier. Si vous n'tes pas sur PC, pour faire tourner
   le programme de dmonstration, il faut trouver la fonction quivalente
    getdate() qui renvoie la date du jour */
#include <dos.h>

#define OK      0
#define ERREUR  1

const char *nom_jour[] = {
	"Lundi",
	"Mardi",
	"Mercredi",
	"Jeudi",
	"Vendredi",
	"Samedi",
	"Dimanche"
	};

const char *nom_mois[] = {
	"janvier",
	"fevrier",
	"mars",
	"avril",
	"mai",
	"juin",
	"juillet",
	"aout",
	"septembre",
	"octobre",
	"novembre",
	"dcembre"
	};

const char *rom_chiffre[]  = { "I","V","X","L","C","D","M","|))","((|))"};

const char deb_rep[] = {22,22,22,23,22,22,22,23,23,23,23,24,23,23,23};

	/* Prototypes des fonctions du fichier CALENDE.C */
	/*************************************************/

/* Calcule le jour julien d'une date spcifie en entre */
long jour2jul(char jour , char mois , short annee);

/* Vrifie l'existence d'une date spcifie en entre */
char existj(char jour , char mois , short annee);

/* Calcule la date correspondant  un jour julien spcifi en entre */
char jul2jour(char *jour , char *mois , short *annee , long jule);

/* Calcule l'cart entre deux dates en nombre de jours */
long ecart_date(char j1 , char m1 , short a1 , char j2 , char m2 , short a2);

/* Calcule la nouvelle date obtenue en ajoutant un nombre de jours  une date
   donne */
char ajoute_jour(char *j, char *m, short *a, long n);

/* Calcule le nom et le numro du jour de la semaine d'un jour julien donn */
char *j_txt(long n , char *num);

/*  Renvoie le n de la semaine dans l'anne d'une date spcifie */
char jour2semaine(char jour , char mois , short annee);

/* Renvoie la date du lundi d'une semaine spcifie en entre */
char semaine2jour(char *jour , char *mois , short *annee , char semaine);

/* Place une date dans une chaine de carctre fournie */
char date2asc(char jour , char mois , short annee , char *txt);

/* Convertit une date grgorienne en rpublicaine */
char gregor2rep(char *jour , char *mois , short *annee);

/* Convertit une date rpublicaine en grgorienne */
char rep2gregor(char *jour , char *mois , short *annee);

/* Convertit un nombre en chiffre romains */
char chiffre2rom(short chiffre , char *romain);

/*************************************************************************

  Procdure : jour2jul

  Auteur / Date : Pierre Larbier 7/11/1994

  Fonction : Calcule le n du jour julien d'une date donne.
	     Le jour n0 correspond au Lundi 1er Janvier 4713 av. JC.
	     Nous sommes aujourd'hui le 7/12/1993 au jour n 2 449 329.
	     Cette fonction prend en compte :
		- le fait que l'anne 0 n'existe pas
		- le changement de calendrier le 4 octobre 1582 (le lendemain
		  du Jeudi 4 octobre 1582 fut le vendredi 15 octobre 1582)
		- les annes bissextiles sont conformes au calendrier julien
		  avant le 4 octobre 1582 et conformes au calendrier grgorien
		  aprs le 15 octobre 1582.


	     Cette fonction est suceptible de renvoyer n'importe quoi si la
	     date spcifie n'existe pas. Cependant, elle a la bonne ide
	     de considrer que le mois n13 est le mois de janvier de l'anne
	     suivante (ce qui est utilis dans la routine existj().
	     Les calculs sont effectus entirement en entier.

  Entres : jour, mois , annee : date dont on cherche le jour julien.

  Sortie  : jour julien correspondant  la date.

*************************************************************************/
long jour2jul(char jour , char mois , short annee)
{
long jule, gregorien , a , j , m;

	/* Attention, cette routine ne doit etre appelle que si on est sur
	   que le jour fourni en paramtre existe !!!!!! */
	a = (long)annee;
	m = (long)mois;
	j = (long)jour;

	/* On commence par faire un ajustement pour les annes ngatives */
	if (annee < 0)
		a++;

	/* Cette 1re formule tient compte des annes bissextiles "normales"
	   c'est  dire des annes divisibles par 4 (calendrier Julien) */
	jule = ((489*m-481)>>4) + j + 1721423L - (((a&3)+7)>>2)*((m+13)>>4)
				+ ((1461*(a-1))>>2);

	/* Avant le 04/10/1582 on est en calendrier julien, et en calendrier
	   grgorien aprs le 15/10/1582 */
	if (jule <= 2299160L)
		return jule;

	/* En calendrier grgorien les ajustements  faire sont :
	   - les annes divisibles par 100 ne sont pas bissextiles
	   - les annes divisibles par 400 sont bissextiles.
	   J'ai laiss l'ajustement en clair pour que vous puissiez intgrer
	   facilement la modification des 4000 ans */
	 gregorien = jule - (a/100L - a/400L - 2L);

	 if ( ( (a%100) == 0 ) && (m < 3) && ( (a%400) != 0) )
		gregorien++;

	return gregorien;
}

/*************************************************************************

  Procdure : existj

  Auteur / Date : Pierre Larbier 7/12/1993

  Fonction : Teste si une date spcifie est valide (par exemple le 53/21/1993
	     n'existe pas)

  Entres : jour, mois , annee : date dont on vrifie l'existence.

  Sortie  : OK si la date existe
	    ERREUR si :
		- le jour n'est pas dans l'intervalle [1,31]
		- le mois n'est pas dans l'intervalle [1,12]
		- l'anne est nulle
		- la date est entre le 5/10/1582 et le 14/10/1582 inclus
		- le jour est plus grand que le nombre de jour du mois en
		  question.

*************************************************************************/
char existj(char jour , char mois , short annee)
{
short nbj;
	/* On commence par liminer le plus gros */
	if ((jour < 1) || (jour > 31) || (mois < 1) || (mois > 12) || (annee == 0))
		return ERREUR;

	/* Puis le cas (trs rare normalement) des 10 jours  du 4 au 15 octobre 1582
	qui ont disparu quand on est pass du calendrier Julien au Grgorien */
	if ( ( annee == 1582) && (mois == 10) ) {
		if ( (jour > 4) && (jour < 15) )
			return ERREUR;
		else
			return OK;
	  }
	/* On regarde ensuite combien il y a de jour dans le mois en question grace
	    la fonction jour2jul(). A noter que jour2jul() marche pour notre
	   application si on lui passe 13 comme n de mois) */

	nbj = (short)jour2jul( 1 , mois + 1 , annee) - (short)jour2jul(1,mois,annee);
	/* Et si le jour propos est suprieur au nbre de jours de ce mois l,
	   on dit qu'il n'existe pas */
	if (jour > nbj)
		return ERREUR;

	return OK;
}

/*************************************************************************

  Procdure : jul2jour

  Auteur / Date : Pierre Larbier 7/12/1993

  Fonction : Renvoie la date correspondant  un jour julien spcif en entre.
	     C'est la fonction inverse de jour2jul().

	     L'algorithme utilis ici n'est pas trs brillant (c'est le moins
	     qu'on puisse dire) mais on ne travaille qu'en entier et pour
	     l'usage qu'on peut en faire, sa lenteur ne doit pas tre un
	     handicap trop grave.

	     J'ai vrifi que dans la gamme de dates possibles (du 1/1/-32768
	     au 31/12/32767) cette fonction concorde bien avec jour2jul().

  Entres : jour, mois , annee : pointeurs sur les lments de la date
				 recherche
	    jule : jour julien dont on veut la date correspondante.

  Sortie  : OK si la date produite est valide.
	    ERREUR si la date produite sort de l'intervalle
			 [1/1/-32768,31/12/32767]

*************************************************************************/
char jul2jour(char *jour , char *mois , short *annee , long jule)
{
long an , deb , fin;
char m , j;

	/* Si l'anne que l'on va trouver n'est pas reprsentable dans un
	   short on sort en erreur */
	if ((jule < -10247089L) || (jule > 13689325L))
		return ERREUR;

	/* On commence par chercher une premire estimation de l'anne */

	/* Cas de cette fichue anne 1582 */
	if ((jule >= 2298884L) && (jule <= 2299238L)) {
		an = 1582L;
		deb = 2298884L;
	}
	else {
		if (jule > 2299160L) {
			/* Cas des jours grgoriens */
			an = (4L*(jule - 2299239L))/1461L;
			/* Ajustement tenant compte des annes de sicle
			   non bissextiles (inutile quand on se limite
			    + ou - 32767 annes) */
			an = an + (3L*an)/146100L + 1583L;
		}
		else
			/* Cas des jours juliens */
			an = (4L*jule)/1461L - 4713L;

		if ((an >= 0L) && (an < 1583L))
			an++;
		deb = jour2jul(1,1,(short)an);
		fin = jour2jul(1,1,(short)an+1);
		if (deb == fin)
			fin = jour2jul(1,1,(short)an+2);
		else {
			if (jule >= fin) {
				an++;
				deb = fin;
			}
			else {
				if (jule < deb) {
					an--;
					deb = jour2jul(1,1,(short)an);
				}
			}
		}
	}
	*annee = (short)an;

	/* On dispose  prsent de l'anne et de son jour de dbut, on va
	   s'en servir pour trouver le mois et le jour */
	m = (short)((jule - deb)/31L) + 1;
	deb = jour2jul(1 , m+1 , (short)an);
	if (jule >= deb)
		m++;
	else
		deb = jour2jul(1 ,m , (short)an);

	j = (short)(jule - deb + 1L);

	/* Il ne nous reste plus que le cas vicieux d'octobre 1582 */
	if ((an == 1582L) && (m == 10) && (j > 4))
		j = j + 10;

	*mois = m;
	*jour = j;

	return OK;
}

/*************************************************************************

  Procdure : ecart_date

  Auteur / Date : Pierre Larbier 7/12/1993

  Fonction : Calcule le nombre de jours entre deux dates.
	     C'est une utilisation directe du jour julien.

	     Les deux dates doivent tre valides, sans quoi la fonction risque
	     de renvoyer n'importe quoi.

  Entres : j1 , m1 , a1 : date de dpart
	    j2 , m2 , a2 : date d'arrive

  Sortie  : nombre de jours entre les deux dates. Si j1/m1/a1 est le 7/12/1993
	    et j2/m2/a2 est le 8/12/1993 la fonction renvoie -1L.

*************************************************************************/
long ecart_date(char j1 , char m1 , short a1 , char j2 , char m2 , short a2)
{
	return jour2jul(j1 , m1 , a1) -  jour2jul(j2 , m2 , a2);
}

/*************************************************************************

  Procdure : ajoute_jour

  Auteur / Date : Pierre Larbier 7/12/1993

  Fonction : Calcule la date obtenue en ajoutant un nombre de jour spcifi
	      une date donne.

  Entres : jour , mois , annee : pointeurs sur les lments de la date de
				  dpart et d'arrive.
	    n : nombre de jours  ajouter (n peut tre ngatif ou nul).

  Sortie  : OK ou ERREUR si :
		- la date de dpart n'est pas valide.
		- la date d'arrive sort de l'intervalle
		  [1/1/-32768,31/12/32767].

*************************************************************************/
char ajoute_jour(char *j, char *m, short *a, long n)
{
	/* Teste la date de dpart */
	if (existj(*j , *m , *a) != OK)
		return ERREUR;

	/* Calcule la date finale obtenue */
	return  jul2jour(j, m, a, n + jour2jul(*j , *m , *a));;
}

/*************************************************************************

  Procdure : j_txt

  Auteur / Date : Pierre Larbier 7/12/1993

  Fonction : Calcule le numro du jour de la semaine d'un jour julien donn.
	     Les jours sont cods :
		0 : lundi
		1 : mardi
		2 : mercredi
		3 : jeudi
		4 : vendredi
		5 : samedi
		6 : dimanche
	     La fonction renvoie en plus un pointeur sur une chaine de
	     caractre contenant le nom du jour.

  Entres : n : jour julien dont on cherche le rang dans la semaine
	    num : pointeur sur un char o la routine place le n du jour de
		  la semaine calcul.

  Sortie  : pointeur sur une chaine de caractre contenant le nom du jour de
	    la semaine.

*************************************************************************/
char *j_txt(long n , char *num)
{
char c;
	if (n < 0L)
		c = 6 - (char)((-n-1)%7);
	else
		c = (char)(n%7);
	*num = c;
	return (char *)nom_jour[c];
}

/*************************************************************************

  Procdure : jour2semaine

  Auteur / Date : Pierre Larbier 7/12/1993

  Fonction : Renvoie le n de la semaine dans l'anne d'une date spcifie.
	     Le numro de semaine est compris entre 1 et 53.
	     Dans le cas rarissime o la date spcifie est avant le 1er lundi
	     de l'anne, on est en fait dans la dernire semaine de l'anne
	     prcdente.

  Entres : jour , mois , annee : lments de la date dont on cherche le
		numro de la semaine.

  Sortie  : numro de la semaine calcul.

*************************************************************************/
char jour2semaine(char jour , char mois , short annee)
{
long debut, courant , n;

	/* Commence par regarder si le jour en question existe. Sinon, on
	sort en renvoyant 0 (ce qui n'est pas un n de semaine possible) */
	if (existj(jour , mois , annee) != OK)
		return 0;

	/* Puis le n du jour spcifi */
	courant = jour2jul(jour , mois , annee);

	/* On recherche ensuite le jour julien correspondant au lundi de
	   la semaine n1 de l'anne spcifie */
	debut = jour2jul(1 , 1 , annee);
	if (debut < 0L)
		n = 6L - (-debut-1)%7;
	else
		n = debut%7;
	if (n != 0)
		debut += 7 - n;

	/* Au cas extremement improbable o le jour spcifi se trouverait
	   avant le premier lundi de l'anne, on est  la semaine 52 ou 53*/
	if (courant < debut) {
		if (annee == 1)
			n = -1;
		else
			n = annee - 1;
		debut = jour2jul(1 , 1 , (short)n);
		if (debut < 0L)
			n = 6L - (-debut-1)%7;
		else
			n = debut%7;
		if (n != 0)
			debut += 7 - n;
	}

	/* Il ne reste plus qu' faire la division entire (on bnficie
	   de la bonne ide de Grgoire XIII de passer d'un jeudi au vendredi
	   donc on a pas  tester le cas d'octobre 1582) */
	return (courant - debut)/7 + 1;
}

/*************************************************************************

  Procdure : semaine2jour

  Auteur / Date : Pierre Larbier 7/12/1993

  Fonction : Renvoie la date du lundi d'une semaine spcifie en entre.
	     Le numro de semaine est compris entre 1 et 53.
	     Cette fonction est quasiement l'inverse de jour2semaine().

  Entres : jour , mois , annee : pointeurs lments de la date correspondant
		 la semaine spcifie. A l'appel de la fonction seul *annee
		est utilis. La routine renseigne jour et mois.
	    semaine : numro de la semaine spcifi

  Sortie  : OK ou ERREUR si :
		- la semaine spcifie n'existe pas
		- l'anne spcifie n'existe pas

*************************************************************************/
char semaine2jour(char *jour , char *mois , short *annee , char semaine)
{
long debut, n , suivant;
short an;

	/* Premier test d'erreur */
	if ((semaine < 1) || (semaine > 53) || (*annee == 0))
		return ERREUR;

	/* On le jour julien correspondant au lundi de la semaine n1 de
	  l'anne spcifie */
	debut = jour2jul(1 , 1 , *annee);
	if (debut < 0L)
		n = 6L - (-debut-1)%7;
	else
		n = debut%7;
	if (n != 0)
		debut += 7 - n;

	/* Afin de faire la dernire vrification, recherche le jour julien
	   correspondant au premier lundi de l'anne suivante */
	an = *annee + 1;
	if (an == 0)
		an = 1;
	suivant = jour2jul(1 , 1 , an);
	if (debut < 0L)
		n = 6L - (-suivant-1)%7;
	else
		n = suivant%7;
	if (n != 0)
		suivant += 7 - n;
	if (((suivant - debut)/7) < semaine)
		return ERREUR;

	/* Et on sort en renvoyant la date du lundi de la semaine spcifie */
	jul2jour(jour, mois , annee , debut);
	ajoute_jour(jour, mois, annee, 7*(semaine-1));

	return OK;
}

/*************************************************************************

  Procdure : date2asc

  Auteur / Date : Pierre Larbier 7/12/1993

  Fonction : Remplit une chaine de caractres avec la date fournie en
		entre. Le format est le suivant :
		nom du jour (en lettres), jour (1 ou 2 chiffres) , nom du
		mois (en lettres) , annee (nombre positif , 5 chiffres au
		maximum) , "av. JC" si l'anne est ngative.
		Par exemple, la routine produit la chaine suivante pour le
		7/12/1993 : "Mardi 7 dcembre 1993".

		La plus longue chaine produite fait 34 caractres, donc il
		faut que le programme appelant rserve 35 octets pour la
		contenir (35  cause du \0 terminal).

  Entres : jour , mois , annee : date  convertir en ascii
	    txt : pointeur sur la chaine de caractres  renseigner.

  Sortie  : OK ou ERREUR si :
		- la date spcifie n'existe pas.

*************************************************************************/
char date2asc(char jour , char mois , short annee , char *txt)
{
char msg[35] , len;
char c;
	/* Si la date spcifie n'existe pas, on sort en erreur */
	if (existj(jour, mois, annee) != OK)
		return ERREUR;

	/* Constitue la chaine ascii reprsentant la date */
	msg[0] = 0;
	strcat(msg , j_txt(jour2jul(jour , mois , annee) , &c));
	len = strlen(msg);
	if (annee < 0)
		sprintf(&msg[len]," %d %s %d av. JC",jour,nom_mois[mois-1],-annee);
	else
		sprintf(&msg[len]," %d %s %d",jour,nom_mois[mois-1],annee);

	/* recopie la chaine obtenue */
	strcpy(txt, msg);

	return OK;
}

/*************************************************************************

  Procdure : gregor2rep

  Auteur / Date : Pierre Larbier 21/11/1994

  Fonction : Convertit une date grgorienne en date rpublicaine.
		Elle fonctionne pour les dates comprises entre le 22/9/1792
		et le 31/12/1805 inclus.
		Les numros des mois rpublicains sont les suivants :
		vendmiaire : 1
		brumaire    : 2
		frimaire    : 3
		nivse      : 4
		pluvise    : 5
		ventse     : 6
		germinal    : 7
		floral     : 8
		prairail    : 9
		messidor    : 10
		thermidor   : 11
		fructidor   : 12
		sans-culottides : 13 (ce n'est pas un mois  proprement
			     parler mais il faut bien les mettre
			     quelque-part).


  Entres : jour , mois , annee : pointeurs sur la date pour les convertir.
			en cas d'erreur ils ne sont pas modifis.

  Sortie  : OK ou ERREUR si :
		- la date spcifie n'existe pas dans le calendrier
		  rpublicain.

*************************************************************************/
char gregor2rep(char *jour , char *mois , short *annee)
{
long n;
short i;
	/* Commence par regarder si la date existe dans le calendrier
	   rpublicain */
	if (existj(*jour, *mois, *annee) != OK)
		return ERREUR;
	n = jour2jul(*jour, *mois, *annee);
	if ((n < 2375840L) || (n > 2380687L))
		return ERREUR;

	/* Cherche l'anne ... sans aucun effort d'optimisation... dsol */
	i = 1;
	while ((jour2jul(deb_rep[i-1], 9, i+1791) <= n) && (i != 15))
		i++;
	*annee = i-1;

	/* On en dduit le mois et le jour */
	n = n - jour2jul(deb_rep[i-2], 9, i+1790);
	*mois = 1 + n/30;
	*jour = 1 + n - (*mois - 1)*30;

	/* Et c'est fini */
	return OK;
}

/*************************************************************************

  Procdure : rep2gregor

  Auteur / Date : Pierre Larbier 21/11/1994

  Fonction : Convertit une date rpublicaine en date grgorienne
		Elle fonctionne pour les dates comprises entre le
		1er vendmiaire an I et le 10 nivse an XIV inclus.
		Les numros des mois rpublicains sont les suivants :
		vendmiaire : 1
		brumaire    : 2
		frimaire    : 3
		nivse      : 4
		pluvise    : 5
		ventse     : 6
		germinal    : 7
		floral     : 8
		prairail    : 9
		messidor    : 10
		thermidor   : 11
		fructidor   : 12
		sans-culottides : 13 (ce n'est pas un mois  proprement
			     parler mais il faut bien les mettre
			     quelque-part).


  Entres : jour , mois , annee : pointeurs sur la date pour les convertir.
			en cas d'erreur ils ne sont pas modifis.

  Sortie  : OK ou ERREUR si :
		- la date spcifie n'existe pas dans le calendrier
		  rpublicain.

*************************************************************************/
char rep2gregor(char *jour , char *mois , short *annee)
{
long n;
	/* Commence par regarder si la date existe */
	if ((*annee < 1) || (*annee > 14))
		return ERREUR;
	if (*jour > 30)
		return ERREUR;
	if (*annee == 14) {
		if (*mois > 4)
			return ERREUR;
		if ((*mois == 4) && (*jour > 10))
			return ERREUR;
	}
	if (*mois == 13) {
		if (*jour > 6)
			return ERREUR;
		if ((*jour == 6)&&(*annee!=3)&&(*annee!=7)&&(*annee!=11))
			return ERREUR;
	}
	/* Et on converti la date */
	n = jour2jul(deb_rep[*annee-1], 9, *annee+1791);
	n = n + (*mois-1)*30 + *jour - 1;
	jul2jour(jour , mois , annee , n);

	/* Et voil le travail */
	return OK;
}

/*************************************************************************

  Procdure : chiffre2rom

  Auteur / Date : Pierre Larbier 7/12/1993

  Fonction : Convertit un short en chiffres romains. Le nombre  convertir
		doit tre strictement positif.
		La syntaxe utilise pour le chiffre romain est celle employe
		courament aujourd'hui :
		- un chiffre dcimal correspond  un ensemble de symboles
		  romains (4 symboles au maximum pour le 8)
		- l'emploi de la synthse soustractive est limit au 4 et au
		   9.
		- On utilise les symboles "|))" et "((|))" pour reprsenter
		  respectivement 5000 et 10 000 (les symboles suprieurs ne
		  sont pas ncessaires puisqu'on est limit  32767).

		De ce fait, 84 sera cod "LXXXIV" et non pas "XXCIIII".
		De mme, 1999 sera cod "MCMXCIX" et pas "MIM" par exemple.

		La plus longue chaine produite fait 28 caractres (pour le
		nombre 28 888), le programme appelant devra donc rserver 29
		octets (29  cause du \0 terminal).


  Entres : jour , mois , annee : date  convertir en ascii
	    txt : pointeur sur la chaine de caractres  renseigner.

  Sortie  : OK ou ERREUR si :
		- le nombre est infrieur  1.

*************************************************************************/
char chiffre2rom(short chiffre , char *romain)
{
char msg[29], dec[6] , i;

	/* Si le chiffre spcifi est ngatif ou nul, on sort en erreur */
	if (chiffre <= 0)
		return ERREUR;

	/* Dcompose le chiffre en base 10 */
	i = 0;
	do {
		dec[i++] = chiffre % 10;
		chiffre = chiffre / 10;
	} while (chiffre != 0);

	/* Construit le chiffre romain */
	msg[0] = 0;
	do {
		switch(dec[--i]) {
			case 0: break;
			case 3: strcat(msg , rom_chiffre[2*i]);
			case 2: strcat(msg , rom_chiffre[2*i]);
			case 1: strcat(msg , rom_chiffre[2*i]); break;
			case 5: strcat(msg , rom_chiffre[2*i+1]); break;
			case 6: strcat(msg , rom_chiffre[2*i+1]);
				strcat(msg , rom_chiffre[2*i]); break;
			case 7: strcat(msg , rom_chiffre[2*i+1]);
				strcat(msg , rom_chiffre[2*i]);
				strcat(msg , rom_chiffre[2*i]); break;
			case 8: strcat(msg , rom_chiffre[2*i+1]);
				strcat(msg , rom_chiffre[2*i]);
				strcat(msg , rom_chiffre[2*i]);
				strcat(msg , rom_chiffre[2*i]); break;
			case 9: strcat(msg , rom_chiffre[2*i]);
				strcat(msg , rom_chiffre[2*i+2]); break;
			case 4: strcat(msg , rom_chiffre[2*i]);
				strcat(msg , rom_chiffre[2*i+1]); break;
		}
	} while (i > 0);

	/* recopie la chaine obtenue */
	strcpy(romain, msg);

	return OK;
}




/*************************************************************************

	Petit programme de dmonstration de l'usage de quelques routines
	 de ce fichier.

*************************************************************************/


void main(void)
{
char jour , mois;
short annee , i , j, m,test, res,succes=0;
long n;
char txte[80];
struct date d;
clock_t deb,fin;

	/* O on crit la petite prsentation */
	printf("\n\n");
	printf("Petit exemple de l'utilisation des routines de dates\n");
	printf("****************************************************\n\n");

	/* Entre la date de naissance */
	printf("Entrez votre jour de naissance : ");
	scanf("%d", &j);
	fflush(stdin);
	printf("Entrez votre mois de naissance : ");
	scanf("%d", &m);
	fflush(stdin);
	printf("Entrez votre anne de naissance (complte) : ");
	scanf("%d", &annee);
	fflush(stdin);
	jour = (char)j;
	mois = (char)m;

	/* Vrifie que la date rentre existe bien */
	if (existj(jour, mois, annee) != OK) {
		printf("\nCette date n'existe pas ...\n");
		printf("Alors je boude ...\n");
		return;
	}

	/* Affiche la date en toutes lettres */
	date2asc(jour , mois , annee ,txte);
	printf("\nAlors comme a, vous tes n le %s\n",txte);
	printf("au jour julien n%ld\n",jour2jul(jour,mois,annee));

	/* calcule l'age en jours */
	getdate(&d);
	n = ecart_date((char)d.da_day , (char)d.da_mon , d.da_year ,
			jour, mois , annee);
	if (n == 0L)
		printf("\nVous etes n aujourd'hui ????\n");
	else {
		if (n < 0L)
			printf("\nVous allez naitre dans %ld jours ??\n",-n);
		else
			printf("\nVous etes ag aujourd'hui de %ld jours\n",n);
	}

	/* Cherche la liste des Vendredi 13 cette anne l */
	for (i = 0 , mois = 1 ; mois < 13 ; mois++) {
		j_txt(jour2jul(13,mois,annee), &jour);
		if (jour == 4)
			i++;
	}

	if (annee < 0) {
		annee = -annee;
		chiffre2rom(annee , txte);
		printf("En l'an de grce %s av. JC, il y a eu %d Vendredi 13\n\n",txte , i);
	}
	else {
		chiffre2rom(annee , txte);
		printf("En l'an de grce %s apr. JC, il y a eu %d Vendredi 13\n\n",txte , i);
	}

	printf("\nVoulez vous vous entrainer  trouver un jour de la semaine de tte (o/n) ?");

boucle:
	if (toupper(getch()) != 'O') {
		printf("\nAlors, au revoir ...\n");
		return;
	}

	printf("\nC'est parti pour une srie de 10...\n");
	printf("Je vous rappelle que 1 correspond au lundi, 2 au mardi .. 0 au dimanche\n");
	deb = clock();
	randomize();
	for (i=0 ;i <10 ; i++) {
		printf("\nEssai n%d\n",i);
		do {
			annee = random(100) + 1900;
			j = random(32);
			m = random(13);
		} while (existj((char)j , (char)m , (short) annee) != OK);
		do{
			fflush(stdin);
			printf("A quel jour de la semaine correspond le %d/%d/%d ? ",(short)j,(short)m,(short)annee);
			scanf("%d", &test);
		} while ((test <0) || (test >= 8));
		date2asc((char)j , (char)m , annee ,txte);
		res =(jour2jul((char)j,(char)m,annee)+1) % 7;
		if (res == test) {
			printf("Bravo c'est le %s\n",txte);
			succes++;
		}
		else
			printf("Erreur c'est le %s\n",txte);
	}
	fin = clock();
	printf("\nNbre de succs : %d sur %d tentatives en %fs par rponse\n\n",succes,i,(fin - deb) / CLK_TCK/(float)i);
	printf("Voulez vous recommencer (o/n) ?");
	/* Moi les goto a ne me fait pas peur !!! Les mollah de
	   l'informatique peuvent bien me menacer de leurs foudres, je mets
	   mon goto */
	goto boucle;
}
