/*
 * $Header: /home/vikas/src/xtacacsd/RCS/xtacacsd.c,v 4.3 1998/03/08 21:35:11 vikas Exp $
 */

/*
 * TACACS daemon suitable for using on Un*x systems.
 *
 * AUTHOR:
 *
 *	Vikas Aggarwal (vikas@navya.com)  from the original version
 *	by Greg Satz (satz@cisco.com), January 1989
 *
 *	Copyright (c) 1995, 1996, 1997 by Vikas Aggarwal (vikas@navya.com)
 *
 *	Portions of the code are  Copyright (c) 1989 by cisco Systems, Inc.
 *
 */

/*
 * KNOWN CAVEATS:
 *	- uwtmp_entry() can only compare the number of characters stored in the
 *	 tacutmp structure (while comparing hostnames). Long hostnames which 
 *	 have the same first X characters will be treated similarly.
 *
 *	- cannot process NIS style entries in the alternate password files or
 *	  do 'ignore-case' matches in the default system password file.
 */

/*
 * $Log: xtacacsd.c,v $
 * Revision 4.3  1998/03/08 21:35:11  vikas
 * Added hooks for BSDI in check_expiration()
 *
 * Revision 4.2  1998/01/05 05:57:50  vikas
 * Now does not exit on getting an unknown error in recvfrom().
 * Bug was posted on BUGTRAQ@NETSPACE.ORG  by  c_karma@hotmail.com.
 *
 * Revision 4.1  1997/06/05 18:23:25  vikas
 * - Now puts GID into the comment field.
 * - added new expiredays variable for shadow password expiry days
 * - uwtmp_entry() now takes a login/logout boolean parameter.
 *   We let uwtmp_entry() decide how to flag the entry for logout/login.
 *
 *
 * Revision 4.0.1  1997/04/14 06:26:47  vikas
 * - Deleted all NOMALLOC definitions. Now stores the GID of the
 *   user in the comment field of wtmp (the TID was pretty much
 *   useless for any purpose).
 * - Now specify whether entry is login via separate arg to uwtmp_entry
 *   instead of making insertions here.
 * - Added expireddays variable and logic so that we can try extracting
 *   the expire value from the shadow passwd file, then sysV passwd file
 *   'age' field and finally the shell field.
 *
 * Revision 4.0  1996/05/19  17:58:27  vikas
 * Added PIDFILE, line ranges in config file, -Q option to not
 * respond if the user does not exist. ENABLE_LEVELS for cisco
 * release 10.3 and higher, support for OSF SIA, dbm support in
 * Getpw, graceful exit on getting HUP, support for QI.
 * Controllable 'abort_timeout' for exiting if the processing
 * takes too long (let inetd restart again).
 *
 * Revision 3.5  1995/11/10  22:09:15  vikas
 * Bug fixes for fseek() in BSDI
 *
 * Revision 3.4  1995/07/14  16:29:38  vikas
 * Bug fixes, extracted into smaller modules, our own UTMP structure,
 * added LOGGING and QUIET keywords.
 * See RELEASE file for complete list of changes.
 *
 * Revision 3.3  1995/01/03  02:42:51  vikas
 * 1. Added CHAP and ARAP support (brisco@rutgers.edu)
 * 2. Fixed bug in updating entries in the utmp file.
 * 3. Now converting all integers into network->host format for
 *    little/big endian machines.
 * 4. Added back support for creating individual host wtmp.<host> files
 *    Need 'HostWTMP' in the config file.
 * 5. Fixed bug in xslipon- was working on the tacacs packet directly
 *    instead of copying the username + password over.
 * 6. Wrote Getpw.c routines - got sick of the vagaries of the systems
 *    getpwent/setpwfile routines. Added a 'PASSWD DEFAULT' flag for
 *    searching shadow password files or NIS maps (though these are not
 *    searched in a case-insensitive manner).
 *
 *
 * Revision 3.2  1994/10/31  14:30:45  vikas
 * Fixed bug in Getpwnam() where I was doing a strncasecmp() instead
 * of a 'strcasecmp()'. Was causing matches with shorter usernames
 * than other existing ones to match and fail.
 *
 * Revision 3.1  1994/10/25  23:46:12  vikas
 * Release version.
 * 1. XSLIPON permissions were added in 3.0.1.1
 * 2. Support for SLIP ACL in/out lists (merged changes from Cisco's new
 *    release). Not incorporated CHAP and ARAP authentication yet.
 * 3. Support for Solaris shadow password files (rozycki@oeto.pk.edu.pl)
 * 4. More command line options moved into the config file.
 * 5. Can now specify LINE numbers as part of the config lines. Submitted
 *    by Robert.Kiessling@rrze.uni-erlangen.de)
 *
 * Revision 3.0.1.1  1994/10/21  14:12:28  vikas
 * Added support for the new packet type from Cisco for SLIP
 * access lists (revision numbers). Also added authentication for SLIPON
 * and small change to uwtmp()
 *
 * Revision 3.0  1994/09/07  18:18:44  vikas
 * More or less a renumbering of v2.4 to v3.0 since there are a large
 * number of new features. Also:
 *      - Fixed problem with gcc by using ANSI variable length functions
 *      - Added 'transaction-id' to the STAT lines (jposner@nih.gov)
 *      - Retry reopening the wtmp and utmp files.
 *      - Deleted newline in all report() messages.
 *
 * Revision 2.4  1994/08/29  14:25:41  vikas
 * New features include:
 *   - support for external password verification programs like
 *     SDI or Enigma Logic cards.
 *   - case insensitive username matches (Getpwnam)
 *   - get an OK from an external program after password verification
 *     (GETOK config file keyword)
 *   - Formatted stat logging at the NOTICE level (dlinegar@cisco.com
 *     and pfortin@cisco.com)
 *   - Ported to Solaris 2.x (gcc v2.4.5 had problems dealing with
 *     variable args lists. Was giving bogus report() output. Okay
 *     with the Sun C compiler).
 *   - Bug fixes in doing string compares in uwtmp() - should use sizeof
 *     instead of strlen.
 *
 * Revision 2.3  1994/06/15  13:58:08  vikas
 * Now uses proper 'varargs' calls in report().
 *
 * Revision 2.2  1994/06/13  14:36:10  vikas
 * Added Strdup() since some OS's don't have a strdup() function.
 *
 * Revision 2.1  1994/05/31  13:49:00  vikas
 * Deleted password cacheing. Now checks the config commands in
 * the order that they appear and not 'user-list' first, then group-list,
 * etc.
 *
 * Revision 2.0  1994/05/17  14:24:13  vikas
 * Major rewrite. New changes and features include:
 * 	- can now read config file on startup.
 * 	- customizable control over the responses based on the username,
 * 	  group gid or geco field.
 * 	- inactvity timer when running under inetd
 * 	- define any syslog facility (LOG_FACILITY)
 * 	- now updates and maintains the 'utmp' file also.
 * 	- can execute another Unix program in response to a query
 * 	  (useful for initiating dialback, etc.)
 * 	- *much* cleaner code, logging etc.
 *
 *
 * Revision 1.22  1994/05/06  20:49:47  vikas
 * Took and merged all the changes from the cisco version which
 * has the following header:
 * 	9-Apr-94: Add support for TACACS_LOGINREQ
 *
 * Revision 1.19  1993/08/20  21:58:07  aggarwal
 * Fixed bug in main, where I wasn't checking to see if 'gethostbyaddr'
 * was returning NULL value or not, and it would coredump if request
 * was from a host who was not in the nameserver.
 * Also put sequence numbers of '1', '2', '3' in the LOG_INFO syslog
 * messages to give some idea of what is following what.
 *
 * Revision 1.17  1993/01/12  20:07:26  aggarwal
 * Fixed alignment problem on the in_addr argument to inet_ntoa (thanks
 * to hooper@bitsy.ccs.queensu.ca (Andy Hooper). Was causing core dumps
 * periodically
 *
 * Revision 1.16  1992/08/28  23:11:39  aggarwal
 * Added tp fix (copy over since gethostaddr overwrites the old values).
 *
 * Revision 1.14  1992/08/13  19:56:21  aggarwal
 * Fixed the section with '(char *)(tp + 1)' - should be
 * '((char *)tp) + XTACASSIZE'. Thanks Evan Wetstone@rice.edu
 *
 * Revision 1.13  1992/08/07  15:44:56  aggarwal
 * Moved the 'hp' stuff out of the 'if(logging)' loop.
 *
 * Revision 1.11  1992/06/04  21:30:26  aggarwal
 * Now logs the slip entries to the same wtmp file as regular login entries.
 *
 * Revision 1.9  1992/05/18  20:59:29  aggarwal
 * Put 'ifdef' around NUMLOGIN defs so that can define while compiling.
 *
 * Revision 1.8  1992/05/18  17:39:48  aggarwal
 * Added MAXACCESSLIST after which the access list number wraps.
 *
 * Revision 1.7  1992/05/11  21:34:12  aggarwal
 * Was checking for tp->type instead of tp->response when 'quiet' was
 * ON. Fixed.
 *
 * Revision 1.5  1992/05/04  22:28:40  aggarwal
 * Added 'umask' for creation of new 'wtmp' files with world having
 * no write permission.
 *
 * Revision 1.4  1992/05/04  17:18:52  aggarwal
 * Fixed small bug where 'xlogin' was filling in 'oresponse' instead of
 * 'response'.
 *
 * Revision 1.3  1992/04/30  19:31:01  aggarwal
 * Latest merger of great stuff (via vikas@jvnc.net)
 * 	o GID controls for number of logins and deny access
 * 	o generic wtmp file format - auto creates wtmp filename from the
 * 	  name of the terminal server.
 * 	o Enhanced debugging and logging.
 * 	o Customizable 'tty' names (SLIP_TTY and TS_TTY)
 * 	o Multiple password files (upto 5)
 * 	o Quiet option (useful for a sequence of terminal servers)
 * 	o Names to lowercase (good old Unix :-)
 * 	o Command line 'utmp_file' option
 *
 *
 * Revision 1.2  1992/04/29  10:54:57  aggarwal
 * Version from CERFnet (contact pushp@cerf.net). Their version had:
 *	o 'n' flag to prevent the use of name servers. This is useful,
 * 	   if the nameservers are down, this speeds up the process.
 *	   Also, the accounting programs are supposed to do the conversion.
 *	o ensured that 'xslipon'/'xslipoff' don't use address->name translation
 *	  and uses only IP addr. Needed for sanity of the 'wtmp_entry()'
 *
 * Revision 1.1  1992/04/29  10:50:37  aggarwal
 * Initial revision
 *
 */

#ifndef lint
 static char rcsid[] = "$RCSfile: xtacacsd.c,v $ $Revision: 4.3 $ $Date: 1998/03/08 21:35:11 $" ;
 static char sccsid[] = "$RCSfile: xtacacsd.c,v $ $Revision: 4.3 $ $Date: 1998/03/08 21:35:11 $" ;
#endif



#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/file.h>

#include <netinet/in.h>

#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <netdb.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>		/* for timeval struct */
#ifdef AIX
# include <sys/select.h>
#endif

#include "version.h"
#include "tacacs.h"
#define MAIN
# include "common.h"			/* real definitions here */
#undef MAIN

char *pwfile[MAX_PASSWD_FILES + 1];	/* array of 'passwd' filenames */
char from_hname[SOME_ARBITRARILY_LARGE_NUMBER + 1];	/* to store hostname */
static char tacbuf[BUFSIZ];		/* packet recieved or sent */
static int  exitSignal ;		/* flag set if got signal */
static char comment[32];		/* for extended tacutmp struct */
void  HUPhandler();
void  DLYhandler();


/*
 * main
 * We can be called from inetd or via the rc scripts directly
 * Parse arguments and act appropiately.
 */

main (argc, argv)
    int argc;
    char **argv;

{		 /* main */
    int c, on = 1, s, i;
    struct timeval actualtimeout, *timeout;
    struct hostent *hp1;		/* Temp. while copying addr */
    struct servent *sp;
    tacacstype *tp;
    extern char *optarg;
    extern int optind ;

    stand = 0;				/* under inetd */
    nonameserver = 0;			/* name server used */

    abort_timeout = 0L;
    actualtimeout.tv_usec = 0L;
    actualtimeout.tv_sec  = TIMEOUT;
    timeout = &actualtimeout;

    for (i = 0 ; i <= MAX_PASSWD_FILES ; pwfile[i++] = NULL)
      ;

    umask(002) ;			/* default 'o-w' for new files */

    openlog("xtacacsd", LOG_PID, LOG_FACILITY);

    while ((c = getopt(argc, argv, "bc:dhilnpqQsu:w:")) != EOF)
      switch (c) 
      {
       case 'b':
	  blankpassword = 1;            /* allow blank passwords */
          break; 
       case 'c':			/* config file */
	  configfile = optarg;
	  break;
       case 'd':			/* debug (give debug messages) */
	  debug++;
	  break;
       case 'h':
	  update_host_wtmp++;
	  break;
       case 'i':			/* ignore case in usernames */
	  ignorecase++;			/* in SunOS4.x only ? */
	  break;
       case 'l':			/* logging (give INFO messages) */
	  logging = 1;
	  break;
       case 'n':			/* use no name server */
	  nonameserver = 1;
	  break;
       case 'p':			/* search in default password file */
	  do_system_pw = 1;
	  break;
       case 'q':			/*  no DENY type responses at all */
	  quiet = 1;
	  break;
       case 'Q':			/* no DENY if user doesnt exist */
	  Quiet = 1;
	  break;
       case 's':			/* stand-alone, default is inetd */
	  stand = 1;
	  break;
       case 'u':			/* utmp file for # of logins */
	  utmp_file = optarg ;
	  break ;
       case 'w':			/* wtmp file for login history */
	  wtmp_file = optarg ;
	  break;
       case '?':
       default:
	  report(LOG_ERR, "%s: illegal switch %s, exiting", argv[0], argv[optind - 1]);
	  report(LOG_ERR, "  Valid switches are:");
	  report(LOG_ERR, "\t -b\t Allow blank passwords");
	  report(LOG_ERR, "\t -c <conf file>\t Define config file to use");
	  report(LOG_ERR, "\t -d\t\t Enable copious amount of debugging");
	  report(LOG_ERR, "\t -h\t\t Update individual host wtmp files");
	  report(LOG_ERR, "\t -i\t\t Ignore case in username matches");
	  report(LOG_ERR, "\t -l\t\t Enable logging");
	  report(LOG_ERR, "\t -n\t\t Disable DNS, NIS and/or hosts lookups");
	  report(LOG_ERR, "\t -p\t\t Search in default password file");
	  report(LOG_ERR, "\t -q\t\t Do not respond for XTACACS FAILURES");
	  report(LOG_ERR, "\t -Q\t\t Do not respond if user does not exist");
	  report(LOG_ERR, "\t -s\t\t Run as a standalone daemon");
	  report(LOG_ERR, "\t -u <utmp file>\t Define UTMP file to use");
	  report(LOG_ERR, "\t -w <wtmp file>\t Define WTMP file to use");
	  exit(1);
      }		/* end switch() */

    if (debug)
	syslog(LOG_DEBUG, "server starting");

#ifndef DEBUG
    if (debug > 1)
      report(LOG_NOTICE, "%s", "Warning, program not compiled with DEBUG");
#endif

    if (debug > 1)
      report(LOG_DEBUG, "Version %s (%s)", XVERSION, (char *)rcsid);

#ifndef ENABLE_USER	/* warn any user can enable if enable use tacacs */
    if (debug)
      report(LOG_INFO,"Warning, ENABLE_USER not defined, anyone could enable");
#endif

    /* Alternate password files on command line */
    for (pwfilect=0 ; optind < argc && pwfilect < MAX_PASSWD_FILES; optind++)
      pwfile[pwfilect++] = argv[optind];

    if (optind < argc)		/* # files exceeded max */
      pwfile[pwfilect] = (char *)1 ;		/* so set a flag for later */

    umask(002) ;		/* reset umask, ensure 'o-w' for new files */

    if (stand) 
    {
	if (debug)
	  fprintf (stderr, "%s: server starting in standalone mode\n",
		   "xtacacsd");
	/* Background ourselves and let go of controlling tty */
	if (!debug) 
	{ /* go ahead and deamonize */
	    if (fork())
              exit(0);
	    closelog();		/* needed even though using close() */
	    for (c = 0; c < getdtablesize(); c++)
		(void) close(c);
	    (void) open("/", O_RDONLY);
	    (void) dup2(0, 1);
	    (void) dup2(0, 2);
#ifdef SYSV
	    setpgrp();			/* proper daemonizing in sysV */
	    signal (SIGHUP, SIG_IGN);	/* ignore HUP sent by exit below */
	    if (fork() != 0)
	      exit(0);			/* sends HUP to all its children */
#else	/* BSD */
	    c = open("/dev/tty", O_RDWR);
	    if (c >= 0)
	    {
#ifdef TIOCNOTTY
		ioctl(c, TIOCNOTTY, (char *)0);
#endif
		(void) close(c);
	    }
#endif	/* SYSV */
	    openlog("xtacacsd", LOG_PID, LOG_FACILITY);

	} /* !debug, go ahead and deamonize */

    }	/* end: if (stand) */

    if (stand)
    {
	/* Pick up a socket */
	
	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) 
	{
	    report(LOG_ERR, "socket: %s, exiting", sys_errlist[errno]);
	    exit(1);
	}
	
	/* Get port we need to pay attention to */
	
	bzero((caddr_t)&from, sizeof(from));
#ifdef sun
	from.sin_family = AF_INET;
	from.sin_addr.s_addr = INADDR_ANY;
#endif
	sp = getservbyname("tacacs", "udp");
	if (sp == NULL)
	    from.sin_port = htons(TACACS_PORT);
	else
	    from.sin_port = sp->s_port;

	if (bind(s, (struct sockaddr *)&from, sizeof(from)) < 0) 
	{
	    report(LOG_ERR, "bind: %s, exiting", sys_errlist[errno]);
	    exit(1);
	}

	timeout = NULL;		/* nuke any timeout value in standalone mode */
    }
    else		/* runnning under inetd */
    {
	s = 0;					/* inetd socket is stdin */
#ifdef FIONBIO
	if (ioctl(s, FIONBIO, &on) < 0) 
	{
	    syslog(LOG_ERR, "ioctl(FIONBIO): %m");
	    exit(1);
	}
#endif
    }

#ifdef PIDFILE
    if (1) {
	int pidfd;
	char pids[16];

	if ((pidfd =  open(PIDFILE, O_WRONLY | O_CREAT | O_TRUNC, 0644)))
	{
	    sprintf(pids, "%d\n", getpid());
	    write(pidfd, (char *)pids, strlen(pids)) ;
	    close(pidfd);
	}
    }
#endif /* PIDFILE */

    /* Note that we read config file at the very end. This allows cmd line
     * args override or precede config file.
     */
    if (configfile)
      if (read_config(configfile) == 0)
	exit(1);

    if (pwfile[MAX_PASSWD_FILES] != NULL) 
    {
	report(LOG_ERR, "# of passwd files exceeded max (%d)",
	       MAX_PASSWD_FILES);
	pwfile[MAX_PASSWD_FILES] = NULL;  /* reset since we are not exiting */
    }

    if (pwfile[0] == NULL)
      do_system_pw = 1;			/* use system call, ignorecase kaput */

#ifdef UTMP				/* record number of current logins */
    if (utmp_file == NULL)		/* not on command line */
      utmp_file = UTMP ;			/* .. so use default */
#endif
#ifdef WTMP				/* record of last logins */
    if (wtmp_file == NULL)		/* not on command line */
      wtmp_file = WTMP ;			/* .. so use defualt */
#endif

    if (debug)
      report(LOG_DEBUG, "wtmp: %s, utmp: %s", 
	     wtmp_file ? wtmp_file : "", utmp_file ? utmp_file : "");

    /* Initialize the file descriptors */
    if (wtmp_file != NULL)
      if ((wtmpfd = open(wtmp_file, O_WRONLY | O_CREAT | O_APPEND, 0664)) < 0)
	report(LOG_ERR, "Can't open wtmp file '%s': %s",
	       wtmp_file, sys_errlist[errno]);

    if (utmp_file != NULL)
      if ((utmpfd = open(utmp_file, O_RDWR | O_CREAT, 0664)) < 0)
	report(LOG_ERR, "Can't open utmp file '%s': %s",
	       utmp_file, sys_errlist[errno]);

    
    report (LOG_NOTICE, "STAT:%s:%s:%s:%s @ %s:line %s:%s:%s:%s",
	    "Service", "Username", "UID", "GID",
	    "From-host", "TTY", "TransID", 
	    "action-specific", "service-specific");

    /*
     * For 4.3BSD machines, this routine sets the file the pw routines use
     * to the given argument. We emulate it for others.
     */
    setpwent() ;			/* open/rewind default /etc/passwd */

    signal(SIGHUP,  HUPhandler);		/* exit gracefully */
    signal(SIGTERM, HUPhandler);		/* exit gracefully */
    signal(SIGALRM, DLYhandler);

    for ( ; ; )		/* forever */
    {
	int nfound ;
	fd_set readfds;

	expiredays = -1;			/* do not initialize to 0 */

	if (exitSignal) {
	    report (LOG_INFO, "main: Exiting on signal");
	    break;
	}

	if (abort_timeout > 0) {	/* reset while waiting for pkts */
	    interruptible = 0;
	    alarm(0);
	}

	FD_ZERO (&readfds); FD_SET(s, &readfds);
	nfound = select(s + 1, &readfds, NULL, NULL, timeout);
	if (nfound < 0) {
	    if (errno != EINTR) {
		report(LOG_ERR, "select: %s", sys_errlist[errno]);
	    }
	    continue;
	}
	if (!(FD_ISSET (s, &readfds))) {
	    if (stand)
	      continue ;
	    else
	    {
		syslog(LOG_INFO, "exiting after %ld minutes of inactivity\n",
		       actualtimeout.tv_sec / 60);
		exit(0);
	    }
	}
	
	if (abort_timeout > 0) {	/* simply exit if stuck somewhere */
	    interruptible = 1;
	    alarm(abort_timeout);
	}

	fromlen = sizeof(from);
	c = recvfrom(s, tacbuf, sizeof tacbuf,0, (struct sockaddr *)&from, &fromlen);
	
	if (c <= 0)
	{
	    if (errno == EINTR && stand)
	      continue ;
	    if (errno == EINTR && c == 0)	/* under inetd */
	      continue ;

	    report(LOG_ERR, "recvfrom(%s): %s",
		   (char *)inet_ntoa(from.sin_addr), sys_errlist[errno]);
	    if (errno == ENOTSOCK)	/* should be in inetd,,, */
	      exit(1);
	    
	    continue;
	}

	tp = (tacacstype *)tacbuf;
	ntoh_fixup (tp);
	
	/*
	 * save the hostname or the IP address,,, used thruout the program
	 */
      	if (nonameserver == 0)		/* gethostaddr returns a ptr */
	  hp1 = gethostbyaddr((char *)&from.sin_addr, 
			      sizeof (struct in_addr), AF_INET);
	if (hp1 == NULL || nonameserver)
	  strncpy(from_hname,(char *)inet_ntoa(from.sin_addr), sizeof from_hname);
	else
	  strncpy(from_hname, hp1->h_name, sizeof from_hname);
	
	report(LOG_INFO, "(1) main: request from %s [%s]",
	       from_hname, (char *)inet_ntoa(from.sin_addr));

	/* for lack of anything else, we are putting the TID in the comment
	 * field. Notice the extra space in the beginning since this is
	 * filled in by uwtmp_entry() with a '?' if it is a logout entry
	 * If we have a valid GID, we will put this into this field instead.
	 */
	sprintf (comment, " TID%d", tp->trans);	/* for uwtmp_entry() */
	if (tp->version == TA_VERSION)
	  old_process(s, &from, tp);
	else if (tp->version == XTA_VERSION)
	  new_process(s, &from, tp);
	else if (((tp->version & XTA_MAJOR_VERSION_MASK) == XTA_VERSION) &&
		 ((tp->version & XTA_MINOR_VERSION_MASK) <= XTA_MINOR_VERSION))
	{
#ifdef DEBUG
	    report(LOG_DEBUG,"main: recvd XTACACS with  minor version %d", 
		   (int)(tp->version & XTA_MINOR_VERSION_MASK));
#endif
	    newstyle_query = 1;
	    new_process(s, &from, tp);
	}
	else if (logging)
	{
	    report(LOG_INFO,"main: illegal version specified: %d (0x%x)", 
		   tp->version, tp->version);
	    report(LOG_INFO, "main: ignoring illegal version request") ;
	}

    }	/* end:  forever() */

    /*
     * here only on recieving signal
     */
#ifdef PIDFILE
    unlink (PIDFILE);
#endif
    exit (1);

} 	/* end: main */


#ifdef NOGETDTABLE
getdtablesize()
{
  return(_NFILE);
}
#endif



/*+ old_process:
 *
 *    	Perform necessary stuff to do a query. Return ANSWER
 */

old_process(s, client, tp)
    int s;
    struct sockaddr_in *client;
    tacacstype *tp;
{
    char *name, *passwd;
    struct passwd *pw;
    int expired;

    querytime = time(NULL);

#ifdef DEBUG
    report(LOG_DEBUG, "old_process: starting");
#endif

    if ( (name = (char *)malloc(tp->namelen+1)) == NULL)
      report(LOG_ERR, "malloc failed for %d bytes of name", tp->namelen+1 );
    else if ( (passwd = (char *)malloc(tp->pwlen+1)) == NULL)
      report(LOG_ERR, "malloc failed for %d bytes of passwd", tp->pwlen+1 );
    if (name == NULL || passwd == NULL)
      return (1);


    strncpy(name, ((char *)tp) + TACACS_SIZE, tp->namelen);
    name[tp->namelen] = '\0';
    strncpy(passwd, ((char *)tp) + TACACS_SIZE + tp->namelen, tp->pwlen);
    /*    if (tp->pwlen > PASSWD_LENGTH)	/* */
    /*	  tp->pwlen = PASSWD_LENGTH;		/* */
    passwd[tp->pwlen] = '\0';

#ifdef DEBUG	/* this prints out the password, commented out */
/*    if (debug && stand)
 *   {
 *	fprintf(stderr, "packet: %s %s\n", ((char *)tp) + TACACS_SIZE,
 *		((char *)tp) + TACACS_SIZE + tp->namelen);
 *	fprintf(stderr, "local:  %s %s\n", name, passwd);
 *    }
 */
#endif


    report (LOG_INFO,"old_process: user %s, host %s", name, from_hname);

    /* Assume failure */

    tp->oresponse = TA_A_REJECTED;
    tp->oreason = TA_A_DENIED;			/* perhaps TA_A_PASSWORD ? */
    if (authent_pw(name, passwd, &pw, 1) == A_PASS) /* verify the password */
    {
	tp->oresponse = TA_A_ACCEPTED;
	tp->oreason = TA_A_NONE;
	
	/*
	 * Now check the expiration time.
	 */
	if (expiredays >= 0)			/* from shadow passwd file */
	  expired = check_expiration(expiredays);	/* directly in days */
	else
	{
#if defined(SYSV) || defined(SVR4) || defined(__svr4__)
	  if (pw->pw_age && pw->pw_age[0])
	    expired = check_expiration(strtodate(pw->pw_age));
	  else
#else
# if defined(BSDI) || defined(FREEBSD)
	  if (pw->pw_expire)
	    expired = check_expiration(pw->pw_expire);
	  else
# endif
#endif /* SYSV */
	    expired = check_expiration(strtodate(pw->pw_shell));
	}

	if (expired == 2)
	{
	    tp->oresponse = TA_A_REJECTED;
	    tp->oreason = TA_A_EXPIRING;
	    goto done_checks ;
	} 
	else if (expired == 1)
	  tp->oreason = TA_A_EXPIRING;

	/*
	 * Now check the permissions for this request.
	 */
	if (pw != NULL) {
	  tp->type = OLD_REQUEST;
	  if (!check_perm(pw, tp)) {
	    tp->oresponse = TA_A_REJECTED;
	    tp->oreason = TA_A_DENIED;
	  }
	}
    }
 done_checks:


    report(LOG_INFO, "old_process: login query for %s (%.10s ..) %s",
	   name, pw == NULL ? "" : pw->pw_gecos, 
	   tp->oresponse == TA_A_ACCEPTED ? "accepted" : "rejected");

    /*
     * Now send off the response if fit...
     */

    tp->type = TA_ANSWER;
    if (quiet && tp->oresponse != TA_A_ACCEPTED)	/* don't respond */
      report (LOG_INFO, "(2)old_process: not sending DENIED since quiet");
    else
    {
	hton_fixup(tp);
	if (sendto(s, tacbuf, sizeof(tacacstype), 0, (struct sockaddr *)client,
		   sizeof(struct sockaddr_in)) != sizeof(tacacstype))
	  report(LOG_ERR, "write: %s", sys_errlist[errno]);
	else
	  report(LOG_INFO, "(2)old_process: Sent response to client");
    }

    free(name);
    free(passwd);

#ifdef DEBUG
    report (LOG_DEBUG, "old_process: done");
#endif

} 		/* end:  old_process */




/*+ new_process:
 *
 * 	Perform necessary stuff to do a query operation. Return ANSWER.
 */

new_process (s, client, tp)
    int s;
    struct sockaddr_in *client;
    xtacacstype *tp;
{
    report(LOG_DEBUG, "new_process: Querytype %d for '%.*s'", tp->type,
	   tp->namelen, ((char *)tp) + XTACACSSIZE ) ;

  querytime = time(NULL);

  switch (tp->type) 
  {
     case XTA_ARAP_AUTH: 
     case XTA_CHAP_AUTH: 
     case XTA_SLIPADDR:
     case XTA_ENABLE: 
     case XTA_LOGIN:
                	xauth(s, client, tp);
                	break;
      case XTA_CONNECT: 
			xconnect(s, client, tp);
                	break;

      case XTA_LOGOUT:
			xlogout(s, client, tp);
			break;
      case XTA_RELOAD:
			xreload(s, client, tp);
			break;
      case XTA_SLIPON:
			xslipon(s, client, tp);
			break;

      case XTA_SLIPOFF:
			xslipoff(s, client, tp);
			break;

      default:
			report(LOG_WARNING, 
			       "(2/3)new_process: illegal type- %d",tp->type);

   }	/* end switch */

}	/* end new_process */




/*+ xauth:
 *
 * 	Called by new_process(). Checks password of user against all the
 * password files. Routine very similar to the 'old_process()' function.
 * This same routine is called for LOGIN, SLIPADDR & ENABLE requests.
 * Calls uwtmp_entry() to make an entry in the wtmp file for *login*
 * requests only.
 */

xauth (s, client, tp)
    int s;
    struct sockaddr_in *client;
    xtacacstype *tp;
{
    int routing = (tp->flags & XTA_F_ROUTING);
    char *name, *passwd;
    struct passwd *pw;		/* set to NULL by other routines if denied */
    int authentok = A_FAIL;
    int expired, resplen;
    char linename[20];
    char *reqname ;		/* string for function name,,, */
    char *verifyname;		/* temp var for username to be verified */

    switch (tp->type)
    {
     case XTA_SLIPADDR:
	reqname = "xslipaddr"; break;
     case XTA_ENABLE:
	reqname = "xenable"; break;
     case XTA_LOGIN:
	reqname = "xlogin"; break;
     case XTA_ARAP_AUTH:
	reqname = "xarap"; break;
     case XTA_CHAP_AUTH:
	reqname = "xchap"; break;
     default:
	reqname = "xauth"; break;
    }

#ifdef DEBUG
    report(LOG_DEBUG, "%s: starting", reqname);
#endif

    if ( (name = (char *)malloc(tp->namelen+1)) == NULL)
      report(LOG_ERR, "malloc failed for %d bytes of name", tp->namelen+1 );
    else if ( (passwd = (char *)malloc(tp->pwlen+1)) == NULL)
      report(LOG_ERR, "malloc failed for %d bytes of passwd", tp->pwlen+1 );
    if (name == NULL || passwd == NULL)
      return;

    strncpy(name, ((char *)tp) + XTACACSSIZE, tp->namelen);
    name[tp->namelen] = '\0';
    strncpy(passwd, ((char *)tp) + XTACACSSIZE + tp->namelen, tp->pwlen);
    /*    if (tp->pwlen > PASSWD_LENGTH)	/* */
    /*	  tp->pwlen = PASSWD_LENGTH;		/* */
    passwd[tp->pwlen] = '\0';
    sprintf(linename, "%s%d", TS_TTY, tp->lport);


#ifdef DEBUG	/* this prints out the password, commented out */
/*    if (debug && stand)
 *   {
 *	fprintf(stderr, "packet: %s %s\n", ((char *)tp) + XTACACSSIZE,
 *		((char *)tp) + XTACACSSIZE + tp->namelen);
 *	fprintf(stderr, "local:  %s %s\n", name, passwd);
 *    }
 */
#endif

    report (LOG_INFO,"%s: user '%s' @ %s (line %s)", reqname,
	    name, from_hname, linename);

    /*
     * If IP address is default on the terminal servers, then the 'username'
     * will be 'default' for SLIP mode. Set to default for  XTA_SLIP and 
     * reject for XTA_LOGIN.
     */
    if (strncmp(name,"default",7) == 0)
      switch (tp->type)
      {
       case XTA_SLIPADDR:
	  strcpy(passwd, DEFAULTS_IP_PASSWORD); break;
       default:
	  strcpy(passwd, "*");	break;		/* Junk password */
      }

    verifyname = name;

#ifdef ENABLE_USER
    if (tp->type == XTA_ENABLE)
      verifyname = ENABLE_USER ;
#endif

    /* Assume failure */
    tp->response = XTA_A_REJECTED;
    tp->reason =   XTA_A_DENIED;		/* perhaps TA_A_PASSWORD ? */

    /* 
     * authent_pw fills &pw without checking if checkpw = 0.
     * Returns -1 if user not found.
     */
    if (tp->type == XTA_ARAP_AUTH || tp->type == XTA_CHAP_AUTH)
      authentok = authent_pw(verifyname, /*passwd*/ "", &pw, /* checkpw */ 0);
    else 
      authentok = authent_pw(verifyname, passwd, &pw, /* checkpw */ 1);
    if (authentok == A_PASS)
    {
	tp->response = XTA_A_ACCEPTED;
	tp->reason = XTA_A_NONE;

	tp->uuid = pw->pw_uid;
	tp->flags = xta_getflags(pw->pw_gecos);	/* always zero ? */

	/* Now check the expiration time.  */
	if (expiredays >= 0)			/* from shadow passwd file */
	{
	  expired = check_expiration(expiredays);
#ifdef DEBUG
	  report(LOG_DEBUG, " Checking expiration from shadow passwd file");
#endif
	}
	else
	{
#if defined(SYSV) || defined(SVR4) || defined(__svr4__)
	  if (pw->pw_age && pw->pw_age[0])
	  {
	    expired = check_expiration(strtodate(pw->pw_age));
# ifdef DEBUG
	    report(LOG_DEBUG, " Checking expiration from pw age field");
# endif
	  }
	  else
#else
# if defined(BSDI) || defined(FREEBSD)
	  if (pw->pw_expire)
	    expired = check_expiration(pw->pw_expire);
	  else
# endif
#endif /* SYSV */
	  {
	    expired = check_expiration(strtodate(pw->pw_shell));
#ifdef DEBUG
	    report(LOG_DEBUG, " Checking expiration from pw shell field");
#endif
	  }
	}	/* end if-else expiredays > 0 */

	if (expired == 2) 
          {
	    tp->response = XTA_A_REJECTED;
	    tp->reason = XTA_A_EXPIRING;
	    goto done_checks ;
          }
         else
           if (expired == 1)
	     tp->reason = XTA_A_EXPIRING;

	/*
	 * Now check the permissions for this request.
	 */
	if (pw != NULL)
	  check_perm(pw, tp);		/* fills in the tp response */
    }

 done_checks:

    if (pw)
      sprintf(comment, " GID%d", pw->pw_gid);	/* notice initial blank */
    /*
     * update utmp file for LOGIN entries only (not enable or slipaddr).
     * SLIP entries are made by 'xslipon'
     */
    if (tp->response == TA_A_ACCEPTED && tp->type == XTA_LOGIN)
      uwtmp_entry(linename, name, from_hname, querytime, comment, /*login*/1);

    /* Log */
    report(LOG_INFO,"%s:%s query from %s %s for %s (%.10s..) %s (ACL %d)", 
	   reqname,(tp->type == XTA_SLIPADDR && routing) ? " (routing)" : "",
	   from_hname, linename, name, pw == NULL ? "" : pw->pw_gecos,
	   tp->response == TA_A_ACCEPTED ? "accepted" : "rejected",
	   tp->accesslist);

    report(LOG_NOTICE, "STAT:%s:%s:%d:%d @ %s:line %s:%d:%s:%d",
	   reqname, name, 
	   pw == NULL ? -1 : pw->pw_uid, pw == NULL ? -1 : pw->pw_gid,
	   from_hname, linename, tp->trans,
	   tp->response == TA_A_ACCEPTED ? "accepted" : "rejected",
	   tp->accesslist);
	    
    /*
     * Now send off the response if fit...
     */

    resplen = sizeof(xtacacstype) ;	/* response length */

    if (tp->response != TA_A_ACCEPTED && (quiet || (Quiet && authentok == A_NOUSER) ))
      report(LOG_INFO,"(2)%s: not sending DENIED since quiet mode",reqname);

    else if (tp->response != TA_A_ACCEPTED && authentok == A_ERROR)
      report (LOG_INFO, "(2)%s: not sending DENIED since authent system error",
	      reqname);

    else		/* send a response */
    {
#ifdef ARAP_SUPPORT
	if (tp->type == XTA_ARAP_AUTH)
	  if ( (resplen = create_arap_resp(tp, pw)) <= 0)
	    report(LOG_ERR, "xauth: create_arap_resp returned %d, FATAL",
		   resplen);
#endif
#ifdef CHAP_SUPPORT
	if (tp->type == XTA_CHAP_AUTH)
	  if ( (resplen = create_chap_resp(tp, pw)) <= 0)
	    report(LOG_ERR, "xauth: create_chap_resp returned %d, FATAL",
		   resplen);
#endif
	tp->type = TA_ANSWER;

	if (debug > 3 && stand) 
	  xdump_pkt(tp);

	hton_fixup(tp);
	if (sendto(s, tacbuf, resplen, 0, (struct sockaddr *)client,
		   sizeof(struct sockaddr_in)) != resplen)
	  report(LOG_ERR, "%s sendto(): %s", reqname, sys_errlist[errno]);
	else	/* All okay */ 
	  report(LOG_INFO, "(2)%s: Sent response to client", reqname);

    }	/* endif !quiet */

    free(name);
    free(passwd);

#ifdef DEBUG
    report(LOG_DEBUG, "%s: done", reqname);
#endif

}		/* end: xuth()  */



/*+ xconnect:
 *
 *	Called when user connects to a specific host & port. Notice that
 * we are not calling check_perm() here since we can achieve all this
 * using ACL's. Perhaps I should ?
 */

xconnect(s, client, tp)
    int s;
    struct sockaddr_in *client;
    xtacacstype *tp;
{
    char linename[20];
    char name[128];
    char dhostname[SOME_ARBITRARILY_LARGE_NUMBER + 1];
    struct hostent *hp1;
    struct in_addr dhost_inaddr;	/* tmp struct for inet_ntoa */

    strncpy(name, ((char *)tp) + XTACACSSIZE, tp->namelen);
    name[tp->namelen] = '\0';
    sprintf(linename, "%s%d", TS_TTY, tp->lport);
    dhost_inaddr.s_addr = tp->dhost;

    if ( nonameserver == 0 )
      hp1= gethostbyaddr((char *)&tp->dhost, sizeof(struct in_addr),AF_INET);

    if (hp1 == NULL || nonameserver)
      strncpy(dhostname,(char *)inet_ntoa(dhost_inaddr), sizeof dhostname);
    else
      strncpy(dhostname, hp1->h_name, sizeof dhostname);	/* save name */
	
    report(LOG_INFO, "(2)xconnect: from %s [%s] %s for %s (%lu) to %s:%d",
	   from_hname, (char *)inet_ntoa(from.sin_addr),
	   linename, name, tp->uuid, dhostname, tp->dport);

    report(LOG_NOTICE, "STAT:%s:%s:%lu:%d @ %s:line %s:%d:%s:%d",
	   "xconnect", name, tp->uuid, -1, from_hname, linename, tp->trans,
	   dhostname, tp->dport);
	    
    replyok (s, client, tp);

} 	/* end: xconnect() */



/* xlogout:
 *
 * Called when the user logs out. Updates utmp/wtmp files.
 */

xlogout (s, client, tp)
    int s;
    struct sockaddr_in *client;
    xtacacstype *tp;
{
    char linename[20];
    char name[128];
    char *reason;

    strncpy(name, ((char *)tp) + XTACACSSIZE, tp->namelen);
    name[tp->namelen] = '\0';
    sprintf(linename, "%s%d", TS_TTY, tp->lport);

    switch(tp->reason) {
      case XTA_R_IDLE:
	reason = "Idle-timeout";
	break;
      case XTA_R_DROP:
	reason = "Carrier-Drop";
	break;
      case XTA_R_BAD:
	reason = "Bad-Passwords";
	break;
      case XTA_R_QUIT:
	reason = "Quit";
	break;
      default:
	reason = "Unknown-reason";
	break;
    }

    report (LOG_INFO, "(2)xlogout: user %s (uid %lu) @ %s (line %s), %s",
	    name,  tp->uuid, from_hname, linename, reason);

    report(LOG_NOTICE, "STAT:%s:%s:%lu:%d @ %s:line %s:%d:%s:%s",
	   "xlogout", name, tp->uuid, -1, from_hname, linename, tp->trans,
	   "", reason);

    comment[0] = ' '; strncpy(&(comment[1]), reason, sizeof(comment)-2);
    /* name[0] = UT_LOGOUT_CHAR ;	/* indicate logout entry */
    uwtmp_entry(linename, name, from_hname, querytime, comment, /*islogin*/0);
    replyok (s, client, tp);

}		/* end: xlogout() */


/*+ xreload:
 *
 *	Called when a terminal server is rebooted. It calls uwtmp_entry
 * to make an entry similar to the one made when a system reboots. Needed
 * for accounting 'ac'. Also flushes all utmp entries for that host.
 */

xreload(s, client, tp)
    int s;
    struct sockaddr_in *client;
    xtacacstype *tp;

{
    report (LOG_INFO, "(2/3)xreload: system reload from %s", from_hname);
    report(LOG_NOTICE, "STAT:%s:%s:%d:%d @ %s:line %s:%d:%s:%s",
	   "xreload", from_hname, -1,-1, from_hname, "all", tp->trans,
	   "", "");

    uwtmp_entry(/*line*/ "~", "xreload", from_hname, querytime, comment, /*login*/1);
    replyok (s, client, tp);

}	/* end: xreload */


/*+ xslipon:
 *
 *	Called when a user invokes 'slip'. Makes an entry in wtmp with
 * a tty name of SLIP_TTY and host is the selected SLIP address.
 *
 * Note that the user's slip address has already been authenticated by
 * slipaddr. This function should use the username (sent by the client in
 * the xtacacs query) to see if permit or deny, and check access lists.
 *
 * Only if the minor number of the request indicates a new type, do
 * we have space for an ACL.
 */

xslipon(s, client, tp)
    int s;
    struct sockaddr_in *client;
    xtacacstype *tp;
{
    int paklen = XTACACSSIZE ;
    int authentok = A_FAIL;
    struct passwd *pw;
    struct hostent *hp1;
    char dhostname[SOME_ARBITRARILY_LARGE_NUMBER + 1];
    char linename[20];
    char name[128];	 /* to copy over the name, since appending NULL */
    struct in_addr dhost_inaddr;	/* tmp structure for inet_ntoa */

    /* Assume accepted since check_perm() rejects if this is set to DENY */
    tp->response = XTA_A_ACCEPTED;
    tp->reason =   XTA_A_NONE;		/* perhaps TA_A_PASSWORD ? */

    strncpy(name, ((char *)tp) + XTACACSSIZE, tp->namelen);
    name[tp->namelen] = '\0';	/* terminating null */
    sprintf(linename, "%s%d", SLIP_TTY, tp->lport);	/* name of line */

    if (tp->namelen == 0) {		/* username required */
	tp->response = XTA_A_REJECTED;
	tp->reason = XTA_A_LOGINREQ;
	tp->type = TA_ANSWER;
	hton_fixup(tp);
	if (sendto(s, tacbuf, XTACACSSIZE, 0, (struct sockaddr *)client,
		   sizeof(struct sockaddr_in)) != XTACACSSIZE)
	    syslog(LOG_ERR, "write: %m\n");
	report(LOG_INFO,
	       "(2)xslipon: Rejected null username query from %s (line %s)",
	       from_hname, linename);
	return ;
    }

    /*
     * Find the password entry from the username and then call
     * check_perm() to see if this should be allowed.
     */
    authentok = authent_pw(name, /* passwd */ "", &pw, /* checkpw */ 0);
    if (authentok != A_PASS)
    {
	if (quiet || (Quiet && authentok == A_NOUSER) || authentok == A_ERROR)
	  report(LOG_INFO,"(2)xslipon: %s: not sending deny since quiet mode",
		 name);
	else if (authentok == A_ERROR)
	  report(LOG_INFO,"(2)xslipon: %s not sending deny since lookup error",
		 name);
	else
	{
	    tp->response = XTA_A_REJECTED;
	    tp->reason = XTA_A_LOGINREQ;
	    tp->type = TA_ANSWER;
	    hton_fixup(tp);
	    if (sendto(s, tacbuf, XTACACSSIZE, 0, (struct sockaddr *)client,
		       sizeof(struct sockaddr_in)) != XTACACSSIZE)
	      syslog(LOG_ERR, "write: %m\n");
	    report(LOG_INFO,
		   "(2)xslipon: user %s @  %s (line %s) rejected (no such user)",
		   name, from_hname, linename);
	}
	return (1) ;
    }

    if (newstyle_query)
      tp->pwlen = 0;		/* reset, used for SLIP ACL structure length */

    check_perm(pw, tp);		/* fills in the tp response */

    dhost_inaddr.s_addr = tp->dhost;

    if ( nonameserver == 0 )
      hp1= gethostbyaddr((char *)&tp->dhost, sizeof(struct in_addr),AF_INET);

    if (hp1 == NULL || nonameserver)
      strncpy(dhostname,(char *)inet_ntoa(dhost_inaddr), sizeof dhostname);
    else
      strncpy(dhostname, hp1->h_name, sizeof dhostname);	/* save name */

    report(LOG_INFO,
	   "(2)xslipon: user %s (uid %lu) @ %s (line %s) address %s %s %s",
	   name, tp->uuid, from_hname, linename, dhostname,
	   (tp->flags & XTA_F_ROUTING) ? "(routing)" : "",
	   tp->response == TA_A_ACCEPTED ? "accepted" : "rejected");

    report(LOG_NOTICE, "STAT:%s:%s:%lu:%d @ %s:line %s:%d:%s:%s%s",
	   "xslipon", name, tp->uuid, pw ? pw->pw_gid : -1, from_hname,
	   linename, tp->trans,
	   dhostname, (tp->flags & XTA_F_ROUTING) ? "(routing)-" : "",
	   tp->response == TA_A_ACCEPTED ? "accepted" : "rejected");

    if (pw)
      sprintf(comment, " GID%d", pw->pw_gid);	/* note initial space */

    if (tp->response == TA_A_ACCEPTED)
      uwtmp_entry(linename, name, from_hname, querytime, comment, /*login*/1);

    if (newstyle_query)
      paklen = XTACACSSIZE + (int)tp->pwlen;
    else
      paklen = XTACACSSIZE ;

    tp->type = TA_ANSWER;
    hton_fixup(tp);
    if (sendto(s, tacbuf, paklen, 0, (struct sockaddr *)client,
	       sizeof(struct sockaddr_in)) != paklen)
      syslog(LOG_ERR, "write: %m\n");
    else
      report(LOG_INFO, "(2)xslipon: Sent response (len=%d) to client", paklen);

}		/* end: xslipon */


/*+ xslipoff:
 *
 * 	Called when slip turned off by user. Entry logged in wtmp file
 * with host of the selected SLIP address.
 */

xslipoff(s, client, tp)
    int s;
    struct sockaddr_in *client;
    xtacacstype *tp;
{
    struct hostent *hp1;
    char dhostname[SOME_ARBITRARILY_LARGE_NUMBER + 1];
    char linename[20];
    char name[128];
    struct in_addr dhost_inaddr;	/* tmp structure for inet_ntoa */

    strncpy(name, ((char *)tp) + XTACACSSIZE, tp->namelen);
    name[tp->namelen] = '\0';
    sprintf(linename, "%s%d", SLIP_TTY, tp->lport);

    dhost_inaddr.s_addr = tp->dhost;

    if ( nonameserver == 0 )
      hp1= gethostbyaddr((char *)&tp->dhost, sizeof(struct in_addr),AF_INET);

    if (hp1 == NULL || nonameserver)
      strncpy(dhostname,(char *)inet_ntoa(dhost_inaddr), sizeof dhostname);
    else
      strncpy(dhostname, hp1->h_name, sizeof dhostname);	/* save name */

    report(LOG_INFO,"(2)xslipoff: user %s (uid %lu) @ %s (line %s) address %s",
	   name, tp->uuid, from_hname, linename, dhostname);

    report(LOG_NOTICE, "STAT:%s:%s:%lu:%d @ %s:line %s:%d:%s:%s",
	   "xslipoff", name, tp->uuid, -1, from_hname, linename, tp->trans,
	   dhostname, "");

    /* name[0] = UT_LOGOUT_CHAR ;	/* indicate logout entry */
    comment[0] = ' '; strncpy(&(comment[1]), dhostname, sizeof(comment)-2);
    uwtmp_entry(linename, name, from_hname, querytime, comment, /*islogin*/0);
    replyok (s, client, tp);

}		/* end: xslipoff() */


xta_getflags (string)
    char * string;
{
    return(0);
}

/*+ replyok:
 *
 *	This function sends an "ok" reply to client (for things like reload,
 * logout). Called by xslipoff()
 *
 */

replyok (s, client, tp)
    int s;
    struct sockaddr_in *client;
    xtacacstype *tp;

{
    tp->response = XTA_A_ACCEPTED;
    tp->reason = XTA_A_NONE;
    tp->type = TA_ANSWER;
    hton_fixup(tp);
    if (sendto(s, tacbuf, sizeof(xtacacstype), 0, (struct sockaddr *)client,
	       sizeof(struct sockaddr_in)) != sizeof(xtacacstype))
      report (LOG_ERR, "write: %s", sys_errlist[errno]);

}	/* end: replyok() */


xdump_pkt(tp)
     xtacacstype *tp;
{
    struct in_addr dhost_inaddr;	/* tmp struct for inet_ntoa */
    dhost_inaddr.s_addr = tp->dhost;

    fprintf(stderr, "XTacacs packet Dump:\n");
    fprintf(stderr, "\tversion: %d\n",tp->version);
    fprintf(stderr, "\ttype: %d\n",tp->type);
    fprintf(stderr, "\ttrans: %d\n",tp->trans);
    fprintf(stderr, "\tnamelen: %d\n",tp->namelen);
    fprintf(stderr, "\tpwlen: %d\n",tp->pwlen);
    fprintf(stderr, "\tresponse: %d\n",tp->response);
    fprintf(stderr, "\treason: %d\n",tp->reason);
#ifdef __alpha
    fprintf(stderr, "\tuuid: %u\n",tp->uuid);
#else
    fprintf(stderr, "\tuuid: %lu\n",tp->uuid);
#endif
    fprintf(stderr, "\tdhost: [%s]\n",(char *)inet_ntoa(dhost_inaddr));
    fprintf(stderr, "\tdport: %d\n",tp->dport);
    fprintf(stderr, "\tlport: %d\n",tp->lport);
#ifdef __alpha
    fprintf(stderr, "\tflags: 0x%x\n",tp->flags);
#else
    fprintf(stderr, "\tflags: 0x%lx\n",tp->flags);
#endif

    fprintf(stderr, "\talist: %d\n",tp->accesslist);

    if (tp->type == XTA_ARAP_AUTH) {
	int i;
	char *s = ((char *)tp) + XTACACSSIZE ;

	fprintf(stderr, "chal_1: ");
	for (i=0; i<8; i++)
	  fprintf(stderr, "0x%x ",s + tp->namelen + i);
	fprintf(stderr, "\nchal_2: ");
	for (i=0; i<8; i++)
	  fprintf(stderr, "0x%x ",s + tp->namelen + ARAP_CHAL_SIZE + i);
	fprintf(stderr, "\nresp_1: ");
	for (i=0; i<8; i++)
	  fprintf(stderr, "0x%x ",s + i);
	fprintf(stderr, "\nresp_2: ");
	for (i=0; i<8; i++)
	  fprintf(stderr, "0x%x ",s + ARAP_RESP_SIZE + i);
	fprintf(stderr, "\n");
    }	/* endif ARAP pkt */

    if (tp->type == XTA_CHAP_AUTH) {
	int i;
	char *s = ((char *)tp) + XTACACSSIZE ;

	fprintf(stderr, "CHAP MD5 resp: ");
	for (i=0; i<MD5_LEN; i++)
	  fprintf(stderr, "0x%x ",s + tp->namelen + i);
	fprintf(stderr, "\n");
    }	/* endif CHAP pkt */

}	/* end: xdump_pkt() */

ntoh_fixup(bp)
     xtacacstype *bp;
{
    bp->trans = ntohs(bp->trans);
    if (bp->version == TA_VERSION)	/* old packet type */
      return ;

    bp->uuid  = ntohl(bp->uuid);
/*    bp->dhost = ntohl(bp->dhost);	/* Do we need this ? */
    bp->dport = ntohs(bp->dport);
    bp->lport = ntohs(bp->lport);
    bp->flags = ntohl(bp->flags);
    bp->accesslist = ntohs(bp->accesslist);
}

hton_fixup(bp)
     xtacacstype *bp;
{
    bp->trans = htons(bp->trans);
    if (bp->version == TA_VERSION)	/* old packet type */
      return ;

    bp->uuid  = htonl(bp->uuid);
/*    bp->dhost = htonl(bp->dhost);	/* do we need this ? */
    bp->dport = htons(bp->dport);
    bp->lport = htons(bp->lport);
    bp->flags = htonl(bp->flags);
    bp->accesslist = htons(bp->accesslist);
}

/*+ HUPhandler
 * Set flag to exit gracefully in the main loop.
 */

void HUPhandler()
{
    exitSignal = 1;
}

/*+ DLYhandler
 * Exit process if request processing time is excessive.
 */

void DLYhandler()
{

    if (interruptible) {
	report(LOG_INFO,"main: processing exceeded %ld sec, exiting on alarm",
	       abort_timeout);
	exit(1);
    }
    else {
	report(LOG_INFO, "main: processing exceeded %ld sec, will exit",
	       abort_timeout);
	exitSignal = 1;
    }
}
