/* $Header: /home/vikas/src/xtacacsd/RCS/xpasswd.c,v 1.3 1997/05/28 20:11:01 vikas Exp $ */

/*
 *
 * xpasswd.c -- change a users' password in a file different than
 * /etc/passwd. Does NOT handle shadow password files.
 *
 * Root can change other users password. If dont want to check old
 * password, then set the main() variable checkold = 0.
 *
 * If no username is specified, it gets the username from the getuid().
 * This might not be valid for non /etc/passwd files. In such cases,
 * the "-l" flag must be used to specify user by name.
 *
 * In all cases, the correct old password must be supplied before
 * the change is permitted.
 *
 * Needs to link with Getpw.o and possibly -lcrypt
 *
 * Modified from the initial version by Tom Brisco, Jan 1997
 * by vikas@navya.com for the xtacacsd package.
 *
 */

/*
 * $Log: xpasswd.c,v $
 * Revision 1.3  1997/05/28 20:11:01  vikas
 * Added NO_FLOCK for good old Unixware (I think)
 *
 * Revision 1.2  1997/04/14 06:09:00  vikas
 * Changes to adapt the software into part of xtacacsd package.
 * - permit 'root' to change other user's password.
 * - added variable checkold to not check the previous password
 * - Replaced memset() with bzero() + macros.
 * - Removed all code that was in Getpw so that can link xpasswd with Getpw
 * - Now using file locking of password file
 * - While replacing password, just replace second field in the password
 *   line and dont try rebuilding the line with its components.
 * - Can specify an alternate password file on command line (-l option)
 *
 *
 */

/* #define DEBUG */

#include <sys/types.h>
#include <termios.h>
#include <unistd.h>
#include <stdio.h>
#include <pwd.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#ifndef NO_FLOCK
# include <sys/file.h>		/* for flock() */
#endif

#if defined(SYSV) || defined(SVR4) || defined(__svr4__)
#  define index strchr
#  define rindex strrchr
#  define bcopy(a, b, c) memmove(b, a, c)
#  define bzero(a, b)    memset(a, 0, b)
#endif

#define MAXPWSTR 32
#define MINPASSWDLEN 4		/* minimum password length */

#ifndef PWFLE
# define PWFILE "/etc/passwd"
#endif

extern struct passwd *Fgetpwent(), *Getpwnam();	/* in Getpw.c */

void
usage()
{
  fprintf(stderr,
	  "USAGE: xpasswd [-l username] [-f passwd file]\n");
  exit(EINVAL);
}

void
echo(onoff)
  int onoff;
{
  struct termios term;
  if (tcgetattr(0, &term)) {
    fprintf(stderr,"Can't get mode to set mode ~ECHO\n");
    exit(1);
  }
  if (onoff) term.c_lflag |= ECHO;
  else term.c_lflag &= ~ECHO;
  if (tcsetattr(0, TCSANOW, &term)) {
    fprintf(stderr,"Can't set mode ~ECHO\n");
    exit(1);
  }
}	/* echo */

/*
 * Given a username and password file, extract the user's passwd struct
 * If no username is specified, extract the UID from the getuid() call.
 */
struct passwd *
getUserent(uname, pwfile)
  char *uname;		/* can be null to get by UID */
  char *pwfile;
{
  uid_t uid;
  struct passwd *pwp;
  FILE * pwfp;

  if (reset_pwfile(pwfile) < 0)
    exit(1);

  if (uname && *uname)
    pwp = Getpwnam(uname, /*ignorecase*/0);    /* get username by name */
  else {
    /* get username by number */
    uid = getuid();
    while ((pwp = Fgetpwent(NULL)) != NULL)
      if (pwp->pw_uid == uid)
	break;
    if (pwp) strcpy(uname, pwp->pw_name);	/* hope its malloced */
  } 

  Endpwent();
  return(pwp);
}

static int
copyfile(from, to)
  char *from, *to;		/* file names */
{
  int fromfd, tofd;
  int nread, nwrite, left;
  char buf[8192];

  if ( (fromfd = open(from, O_RDONLY)) < 0) {
    perror(from);
    return (-1);
  }
  if ( (tofd = open(to, O_WRONLY | O_TRUNC)) < 0) {
    perror(to);
    return (-1);
  }

  while ( (nread = read(fromfd, buf, sizeof(buf))) > 0)
    for (left = 0; left < nread; nread-= nwrite, left += nwrite)
      if ((nwrite = write(tofd, buf + left, nread)) < 0)
	perror("copyfile() write error: ");

  if (nread < 0)
    perror("copyfile() read error: ");
  return(nread);
}

/*
 * To update a new password in the password file. NOTE that this
 * is NOT inserting a NEW entry, just updating an existing entry.
 * From Tom Brisco
 */
chgPasswd(username, nepasswd, passwdFile)
  char * username;
  char * nepasswd;
  char * passwdFile;
{
  int pfd, ret = 0;
  FILE *pwfp, *tmpfp;
  char temp[64];
  char pwline[256];

  strcpy(temp,"/tmp/pwp.XXXXXX");
  strcpy(temp, (char *)mktemp(temp));

  if ((pfd = open(passwdFile, O_RDONLY)) < 0) {  /* file descr for locking */
    fprintf(stderr,"can't open passwd file\n");
    return(-1);
  }
  pwfp = fdopen(pfd, "r");		/* we require the FILE ptr */

  /* Lock the passwd file to prevent two processes from updating... */
  /* Good old SCO Unixware v2.1.1 doesn't seem to have flock() */
#if defined(LOCK_NB) && !defined(NO_FLOCK)
  if (flock(pfd, LOCK_EX | LOCK_NB) < 0) {
    perror("flock() on password file failed");
    return(-1);
  }
#endif
  if ((tmpfp = fopen(temp, "w")) == (FILE *)NULL) {
    fprintf(stderr,"cannot open temp file %s for write\n", temp);
    ret = -1;
    goto done;
  }

  fseek(pwfp, 0L, SEEK_SET);
  while((fgets(pwline, 256, pwfp)) != (char *)NULL)
  {
    if(strncmp(pwline, username, strlen(username)) ||
       *(pwline + strlen(username)) != ':')	/* look for : delimiter */
    {
      fputs(pwline, tmpfp);
    }
    else
    {	  /* we've got the name to replace */
      /* to avoid handling different syntaxes, we just change the passwd
       * and leave the rest fo the line intact
       */
      register char *p;
      for (p = pwline; *p && *p != ':'; ++p)	/* : at end of username */
	;
      if (*p)
	for (++p; *p && *p != ':'; ++p)		/* : at end of passwd */
	  ;
      fprintf(tmpfp, "%s:%s%s", username, nepasswd, p);
#ifdef DEBUG
      fprintf(stderr,"New password file entry=\n  %s:%s%s",
	      username, nepasswd, p);
#endif
    }
  }	/* end while() */
  fclose(tmpfp);
  fclose(pwfp);

  /* now copy temp file to permanent file */
#ifdef DEBUG
  fprintf(stderr, "New temp password file in %s\n", temp);
#endif /* DEBUG */

  if (copyfile(temp, passwdFile) < 0)
    fprintf(stderr, "Warning: there were errors copying temp file %s to %s\n",
	    temp, passwdFile);

done:
  unlink(temp);		/* dont leave this lying around */
#if defined(LOCK_NB) && !defined(NO_FLOCK)
  flock(pfd, LOCK_UN);	/* unlock */
#endif
  close(pfd);
  return(ret);
}	/* chgPasswd() */


main (argc, argv)
 int argc;
 char *argv[];
{
  struct passwd * pwp = (struct passwd *)NULL;
  char opasswd[MAXPWSTR];   /* old password - clear text */
  char oepasswd[MAXPWSTR];  /* old password encrypted */
  char osalt[2];            /* old password salt value */
  char npasswd[MAXPWSTR];   /* new password - clear text */
  char npasswd2[MAXPWSTR];
  char nepasswd[MAXPWSTR];  /* new password encrypted */
  char nsalt[2];            /* new password salt value */

  char *passwdFile = PWFILE;

  char username[MAXPWSTR];
  int optchar;			/* keep integer, dont make character */
  extern char *optarg;
  extern int optind;

  int pid, nsalti1, nsalti2;
  int checkold = 0;			/* CHANGE TO 0 TO NOT CHECK OLD */

  bzero(opasswd,  sizeof(char) * MAXPWSTR);
  bzero(oepasswd, sizeof(char) * MAXPWSTR);
  bzero(npasswd,  sizeof(char) * MAXPWSTR);
  bzero(nepasswd, sizeof(char) * MAXPWSTR);
  bzero(username, sizeof(char) * MAXPWSTR);

  while ((optchar = getopt(argc, argv, "l:f:")) != EOF) {
    switch (optchar) {
	case ':': fprintf(stderr,"missing value for option\n");
		  usage();
        case 'l': strcpy(username, optarg);
		  break;
        case 'f': passwdFile = optarg;
		  break;
	case '?':
        default:  fprintf(stderr,"unknown option %c\n", optchar);
		  usage();
    }
  }
  pwp = getUserent(username, passwdFile);  /* username should be malloced */
  if(!strlen(username)) usage();

  printf("Changing password for\n\tUsername: %s, uid:%d, PWFile: %s\n",
	 username, pwp ? pwp->pw_uid : -1, passwdFile);

  if (!pwp) usage();

  if (getuid() == 0) checkold = 0;

  /* now, turn off echo, and request old passwd */
  echo(0);
  if (checkold) {
    printf("Old Password:"); fflush(stdout);
    fgets(opasswd, MAXPWSTR, stdin); opasswd[strlen(opasswd) - 1] = '\0';
  }
  printf("\nNew Password:"); fflush(stdout);
  fgets(npasswd, MAXPWSTR, stdin); npasswd[strlen(npasswd) - 1] = '\0';
  printf("\nNew Password (again):"); fflush(stdout);
  fgets(npasswd2, MAXPWSTR, stdin); npasswd2[strlen(npasswd2) - 1] = '\0';
  printf("\n");
  echo(1);
  if (strcmp(npasswd, npasswd2)) {
    fprintf(stderr,"new passwords don't match\n");
    exit(1);
  }
  if (!strcmp(npasswd, opasswd)) {
    fprintf(stderr,"old and new passwords are the same\n");
    fprintf(stderr,"(doesn't the pointlessness of it bother you?)\n");
    exit(1);
  }
#ifdef MINPASSWDLEN
  if (strlen(npasswd) < MINPASSWDLEN) {
    fprintf(stderr,"new password is too small (must be atleast %d char)\n",
	    MINPASSWDLEN);
    exit(1);
  }
#endif
    
#ifdef DEBUG
  /* printf("Old Password: %s\n",opasswd);	/* */
  /* printf("New Password: %s\n",npasswd);	/* */
#endif /* DEBUG */

  /* now we have old and new passwords, do the work */
  /* I intentionally collect the "new" password before validating
   * the old -- this should slow down hackers some ....
   */

  if (checkold) {
    osalt[0] = pwp->pw_passwd[0]; osalt[1] = pwp->pw_passwd[1];
    strcpy(oepasswd, (char *)crypt(opasswd, osalt));
    if (strcmp(oepasswd, pwp->pw_passwd))
    {
      fprintf(stderr,"Old password doesnt match\n");
#ifdef DEBUG
      printf("\tPrevious= %s\n\tEntered = %s\n", pwp->pw_passwd, oepasswd);
#endif
      exit(1);
    }
  }

  /* generate new salt values */
  pid = getpid();
  /* now permute this a bit */
  pid = pid * pid;
  nsalti1 = (pid&0xFF);
  nsalti1 %= 122;
  if (nsalti1 < 65) nsalti1 += 65;
  if ((nsalti1 > 90) && (nsalti1 < 97)) nsalti1 += 8;
  nsalti2 = (pid&0xFF00)>>16;
  nsalti2 %= 122;
  if (nsalti2 < 65) nsalti2 += 65;
  if ((nsalti2 > 90) && (nsalti2 < 97)) nsalti2 += 8;
#ifdef DEBUG2
  printf("new Salt: %d %d \"%c\" \"%c\"\n",nsalti1,nsalti2,nsalti1,nsalti2);
#endif /* DEBUG */

  nsalt[0] = nsalti1; nsalt[1] = nsalti2;
  strcpy(nepasswd, (char *)crypt(npasswd, nsalt));

#ifdef DEBUG
  printf("new Encrypted Passwd: %s\n", nepasswd);
#endif /* DEBUG */

  chgPasswd(username, nepasswd, passwdFile);
}


