/* $Header: /home/vikas/src/xtacacsd/RCS/uwtmp.c,v 1.4 1997/06/21 04:57:47 vikas Exp $ */

/*
 * Routines to manipulate the utmp and the wtmp files.
 * Used by xtacacsd as well as taclast/tacwho
 *
 *	-vikas@navya.com
 */

/*
 * $Log: uwtmp.c,v $
 * Revision 1.4  1997/06/21 04:57:47  vikas
 * Was setting the logout flag when the user was logging in instead of
 * the other way around.
 *
 * Revision 1.3  1997/05/28 20:09:42  vikas
 * small change so that the comment length can be any length in the
 * ascii wtmp file.
 *
 * Revision 1.2  1997/04/14 06:40:01  vikas
 * Now has a function setlogoout() to indicate that an entry is logout.
 * Changed lineEq from a macro into a function to handle ISDN lines on
 * cisco v10.x which were giving dummy line numbers (always 65535).
 * Now creates ASCII wtmp files (finally!!). Should slowly remove the
 * binary wtmp files altogether.
 * Added cur_login_count() so that it does not count the same user
 * who matches the supplied hostname and line (was counting a user
 * going into PPP mode as 1).
 *
 * Revision 1.1  1997/04/14 06:37:31  vikas
 * Initial revision
 *
 * :: OLDER LOGS FROM MISC.C ::
 * Revision 1.4  1996/05/19  17:26:24  vikas
 * Added routines to check the size of the utmp/wtmp files and
 * match with the currently compiled utmp structure while
 * updating the files.
 *
 * Revision 1.3  1996/04/07  19:03:09  vikas
 * Added support for LOGOUT_CHAR in the username instead of having
 * only a null string in the username for logouts.
 * Added macros to make things look nicer. Also checks to see
 * the size of the file when opening and warns if wrong size.
 *
 * Revision 1.2  1995/11/10  22:10:03  vikas
 * Fixed lseek() bug for BSDI machines ; subtle but created large
 * utmp files.
 *
 * Revision 1.1  1995/07/14  16:18:47  vikas
 * Initial revision
 *
 */

#ifndef lint
 static char rcsid[] = "$RCSfile: uwtmp.c,v $ $Revision: 1.4 $ $Date: 1997/06/21 04:57:47 $" ;
 static char sccsid[] = "$RCSfile: uwtmp.c,v $ $Revision: 1.4 $ $Date: 1997/06/21 04:57:47 $" ;
#endif

#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
/* #include <pwd.h>		/* */
#include <unistd.h>		/* for SEEK_ definitions */
#include <syslog.h>		/* for report() level definitions */
#include <fcntl.h>		/* for open() flags */
#include <sys/stat.h>		/* for fstat */
#include <netinet/in.h>		/* htonl() definitions */

#include "tacutmp.h"

#include "tacacs.h"	/* */
#include "common.h"	/* */

#ifndef UT_BLANKSTR_FILL
# define UT_BLANKSTR_FILL  "*"   /* filler for blank fields */
#endif

/*
 * Function to test the size of the supplied file and see if the tacutmp
 * record size is evenly divisible. Else, try and assume an old style file
 * and delete the size of the UT_COMMENT and try again.
 * Required for backward compatibility.
 *
 */
static int 
get_utmp_entrysize(fd)
     int fd;		/* opened file descriptor */
{
    struct stat stb;
    struct tacutmp entry;
    int entrysize;

    entrysize = sizeof entry;

    if (fd < 0)
      return (-1);

    if (fstat(fd, &stb) == -1) {
	report(LOG_ERR, "Error fstat-ing file %s\n", sys_errlist[errno]);
	return (-1);
    }

    if ( (stb.st_size % entrysize) != 0)
    {
#ifdef XTACUTMP		/* new tacupd, old format utmp file */
	entrysize = entrysize - UT_COMMENTSIZE ;
	if (stb.st_size % entrysize )
	  return (-1);	/* should be treated as fatal */
#else	/* old tacupd, new format utmp file, so just exit */
	return (-1);
#endif	/* XTACUTMP */

    }	/* if (stb.st_size(...) */

    return (entrysize);
}

/*+
 * Once upon a time ;-), a null username signified a logout on a line.
 * Since the xtacacs udp packets could get lost, the first character of
 * the username was changed to UT_LOGOUT_CHAR to indicate a logout.
 * However, this made it difficult to differentiate between users
 * 	 jsmith and asmith.
 * Soooo, in v4.1, here is another technique for identifying logout entries.
 * Since we have a 'comment' field which is fairly long, we set the FIRST
 * byte of this field to the logout character.
 */
setlogout(ut, loginflag)
  struct tacutmp *ut;
  int loginflag;		/* 1 if login */
{
#ifndef XTACUTMP
  if (!loginflag)
    *(ut->ut_name) = UT_LOGOUT_CHAR;	/* old style UTMP files */
#else

  if (loginflag)
  {
    if (*(ut->ut_comment) == UT_LOGOUT_CHAR)
      *(ut->ut_comment) = ' ';	/* ensure its not a logout character */
  }
  else
  {
    if (*(ut->ut_comment) == '\0')
      *(ut->ut_comment + 1) = '\0' ;	/* need to shift the null */
    *(ut->ut_comment) =  UT_LOGOUT_CHAR ;
  }
#endif /* XTACUTMP */

} 	/* setlogout() */

/*
 * tell if an entry is logout or login. This has to identify all OLD
 * methods of logouts also. Hence, check for:
 *	if (*username) == NULL
 *	if (*username) == UT_LOGOUT_CHAR
 *	if (*comment)  == UT_LOGOUT_CHAR
 */
islogout(ut)
  struct tacutmp *ut;
{
  if (*(ut->ut_name) == '\0' || *(ut->ut_name) == UT_LOGOUT_CHAR)
    return (1);	/* is logout */
#ifdef XTACUTMP
  if (*(ut->ut_comment) ==  UT_LOGOUT_CHAR)
    return (1);
#endif
  return (0);	/* not a logout */
}


/*+ uwtmp_entry:
 *
 * Updates the 'wtmp_file' and utmp files.
 *
 * Called by 'slipOn' and 'slipOff' with line types of SLIP_TTY.
 * If the 'name' is a null string or begins with a UT_LOGOUT_CHAR
 * (in logout's), the corresponding entry in the utmp file is flushed.
 *
 * The calling program MUST initialize the wtmpfd & utmpfd file descriptors
 * to -1 so that this function attempts to open them.
 */

/* #define lineEq(A, B)	(!strncmp(A.ut_line, B.ut_line, UT_LINESIZE)) /* */
#define nameEq(A, B)	(!strncmp(A.ut_name, B.ut_name, UT_NAMESIZE))
#define hostEq(A, B)	(!strncmp(A.ut_host, B.ut_host, UT_HOSTSIZE))

#define DUMMYLINE	"65535"		/* ISDN lines send this */

/* Since ISDN lines on cisco4000 were giving dummy line numbers (always
 * 65535), we have to match on the username instead
 */
lineEq(oentry, nentry)
  struct tacutmp oentry, nentry;
{
  int ignoreline = 0;
  int equal = 0;

  if(strncmp(oentry.ut_line, nentry.ut_line, UT_LINESIZE))
    return (0);		/* lines are not equal */

  /* since they are equal, check if this is a dummy ISDN line */
  if (strncmp(nentry.ut_line + 3, DUMMYLINE, UT_LINESIZE - 3))
    return (1);		/* okay since not a dummy ISDN line */

  /*
   * we have a dummy line number that we cannot rely on, so try matching the
   * username. Not exactly reliable if the person has 2 lines on the same
   * terminal server... If the usernames are same, assume same line.
   * Note that until Jan 1997, the terminal server was always returning
   * a blank username for TTY logouts only (slip logouts had a name)
   * so if the nentry.ut_name is blank, we should return a OK else we
   * get an accumulation of TTY logins.
   */
  if (*oentry.ut_name && *nentry.ut_name)
  {
    if (*oentry.ut_name == UT_LOGOUT_CHAR || *nentry.ut_name == UT_LOGOUT_CHAR)
      return(!strncmp(oentry.ut_name+1, nentry.ut_name+1, UT_NAMESIZE-1));
    else
      return(nameEq(oentry, nentry));
  }
  return(1);	/* return equal, since there is no other way of checking */

}	/* end lineEq() */


uwtmp_entry (line, name, host, timestamp, comment, lflag)
     char *line ;		/* Incoming line for user */
     char *name;		/* User's login name */
     char *host;		/* name/ip of host sending this info */
     unsigned long timestamp;	/* login time, can be zero */
     char *comment;		/* for extended xtacutmp structure */
     int lflag;	   		/* 1 for login, 0 for logout, 2 for unknown */
{
    struct tacutmp oentry, entry;
    register char *c ;
    int		loginflag;
    static int  ascopen = 0;		/* flag for ascii wtmp */
    static int  urecsize, wrecsize;	/* current size of the utmp entries */
    static int  dofsync = 0;		/* dont FSYNC every time */
    static int  retryopen = -1;

    interruptible = 0;			/* no alarms allowed here */
    loginflag = lflag;			/* make local variable */

#ifdef ASCII_WTMP
    wtmpfd = 1;		/* set so that we dont try reopening it at all */
#endif
    
    if (utmpfd < 0  || wtmpfd < 0)
      retryopen = ++retryopen % 10 ;	/* retry opening every 10th time */

    if (wtmpfd < 0 && wtmp_file != NULL && retryopen == 0 )
      if ((wtmpfd = open(wtmp_file, O_WRONLY | O_CREAT | O_APPEND, 0664)) < 0)
	report(LOG_ERR, "Can't open wtmp file '%s': %s",
	       wtmp_file, sys_errlist[errno]);

    if (utmpfd < 0 && utmp_file != NULL && retryopen == 0)
      if ((utmpfd = open(utmp_file, O_RDWR | O_CREAT, 0664)) < 0)
	report(LOG_ERR, "Can't open utmp file '%s': %s",
	       utmp_file, sys_errlist[errno]);

    if (utmpfd < 0 && wtmpfd < 0) {
      interruptible = 1;
      return ;
    }

#ifndef ASCII_WTMP	/* binary wtmp file */
    if (wtmpfd >= 0 && wrecsize <= 0) 	/* get record size */
      if ( (wrecsize = get_utmp_entrysize(wtmpfd)) <= 0 ) {
	  report(LOG_ALERT,
		 "wtmp file format not compatible with compiled tacutmp size, CORRECT IMMEDIATELY USING XTACUTMP DEFINE, NOT WRITING WTMP\n");
	  close(wtmpfd);
	  wtmpfd = -1;
      }
#endif
    if (utmpfd >= 0 && urecsize <= 0)
      if ( (urecsize = get_utmp_entrysize(utmpfd)) <= 0 ) {
	  report(LOG_ALERT,
		 "utmp file format not compatible with compiled tacutmp size, CORRECT IMMEDIATELY USING XTACUTMP DEFINE, NOT WRITING UTMP\n");
	  close(utmpfd);
	  utmpfd = -1;
      }

    bzero(&entry, sizeof entry);

    /* Convert the hostname to all lowercase (DNS responses are case
     * sensitive even though the query is not.
     */
    for (c = host ; *c != '\0' ; ++c)
      *c = tolower(*c) ;

      /* notice no terminating null in the following if no space for it */
#define SCOPY(A,B) \
  { \
      if (strlen(A) < sizeof B) \
	strcpy(B, A); \
      else bcopy(A, B, sizeof B); \
  }
    SCOPY(line, entry.ut_line);
    SCOPY(name, entry.ut_name);
    SCOPY(host, entry.ut_host);
#ifdef XTACUTMP
    SCOPY(comment, entry.ut_comment);
#endif
    
    if (timestamp)  entry.ut_time = timestamp;
    else  entry.ut_time = time(NULL);	/* fill with current time if 0 */

    if (loginflag == 2)
      loginflag = islogout(&entry);	/* if converting wtmp file */
    setlogout(&entry, loginflag);	/* do after copying comment -> entry */
      
#ifdef DEBUG
    report(LOG_DEBUG, "uwtmp_entry: %s '%s'@%s %s, time= %ld, comment= '%s' ",
	   loginflag ? "LOGIN" : "LOGOUT", name, line, host,
	   entry.ut_time, comment);
#endif

#ifndef ASCII_WTMP	/* ASCII wtmp file only */
    if (wtmpfd >= 0)	/* file was opened in append mode, so just write out */
    {
      if (write(wtmpfd, &entry, wrecsize) != wrecsize)
	report(LOG_ERR, "uwtmp_entry: couldn't write wtmp file '%s', %s",
	       wtmp_file, sys_errlist[errno]);
# ifndef NOFSYNC
      else
	if ((dofsync = ++dofsync % 20) == 0)	/* sync every 20 times */
	  fsync (wtmpfd);
# endif
    }	/* endif wtmpfd >= 0 */
# ifdef DEBUG
    else
      report(LOG_DEBUG, "uwtmp_entry: wtmp entry not written, unopened fd");
# endif
#endif	/* ASCII_WTMP */

    /*
     * Create ASCII WTMP file. We should actually stop using the
     * binary wtmp files altogether.
     */
    if (ascopen >= 0)
    {
      static int  syncfile = 0;
      static FILE *ascwtmpf = NULL;	/* ASCII wtmp file */

      if (ascwtmpf == NULL)		/* first time */
      {
	char awtmpfile[BUFSIZ];
	sprintf(awtmpfile, "%s.ascii", wtmp_file);
	if ((ascwtmpf = fopen(awtmpfile, "a")) == NULL)
	{
	  report(LOG_ERR, "uwtmp_entry: Couldn't open ASCII wtmp- %s",
		 awtmpfile);
	  ascopen = -1;		/* set flag so we dont retry file again */
	}
	else
	  ascopen = 1;
      }	/* end if (ascwtmpf == NULL) */

      if (ascwtmpf != NULL)
      {
	int commentlen ;
#ifdef XTACUTMP
	char *cmt = entry.ut_comment;	/* has logout char tacked on */
	commentlen = sizeof(entry.ut_comment);
#else
	char *cmt = comment;	/* not part of structure anyways */
	commentlen = strlen(comment);
#endif

	fprintf(ascwtmpf, "%ld %s %s %s %*s\n",
		entry.ut_time,		/* hope this is filled already */
		host[0] ? host:UT_BLANKSTR_FILL,
		line[0] ? line:UT_BLANKSTR_FILL,
		name[0] ? name:UT_BLANKSTR_FILL,
		commentlen,
		cmt[0] ? cmt:UT_BLANKSTR_FILL);

	if ((syncfile = (++syncfile) % 20) == 0)
	  fflush(ascwtmpf);
      }
    }	/* end if (ascopen >= 0) */


    /*
     * Update alternate wtmp file (with the hostname tacked on) so
     * that 'last' etc. can handle it. This is still in binary
     * file format. Dont know how many people are still using this.
     */
    if (update_host_wtmp)
    {
	char hwtmpfile[BUFSIZ] ;	/* alternate file to update */
	int  hwtmpfd ;

	strcpy(hwtmpfile, wtmp_file);	/* same parent wtmp name */
	strcat(hwtmpfile, ".");		/* add a dot */

	/* Create a sensible filename  and add entry */

	if (*from_hname < '0' || *from_hname > '9')  /* Not an IP address,,, */
	  strncat(hwtmpfile, from_hname, strcspn(from_hname, ".")) ;   /* only header */
	else
	  strncat(hwtmpfile, from_hname, strlen(from_hname)) ;
    
	hwtmpfile[sizeof(hwtmpfile) -1] = '\0';		/* terminating NULL */
	
	hwtmpfd = open(hwtmpfile, O_WRONLY | O_CREAT | O_APPEND, 0664);
	if (hwtmpfd < 0) {
	    report(LOG_ERR, "uwtmp_entry: couldn't open host wtmp- %s",
		   hwtmpfile);
	}
	else {
	    if (write(hwtmpfd, &entry, wrecsize) != wrecsize)
	      report(LOG_ERR, "uwtmp_entry: write host wtmp %s- %s",
		     hwtmpfile, sys_errlist[errno]);
	    else
	      close(hwtmpfd) ;	/* close the alternate wtmp file */
	}

    }	/* end update_host_wtmp */

    
    /*
     ******* Now update the utmp file *******
     */
    if (utmpfd < 0)			/* utmp file not open */
    {
	interruptible = 1;
	return;
    }

    lseek (utmpfd, 0, SEEK_SET);		/* rewind file */

    /*
     * new utmp entry. Make a attempt to avoid duplicates. However,
     * cannot really avoid unless we search thru entire file each
     * time to make sure that the entry does not exist further up.
     */
    if (loginflag)
    {	/* login entry */
      int n;

      while ((n = read(utmpfd, (char *)&oentry, urecsize)) == urecsize)
      {		/* write in erased entry */
	if (islogout(&oentry))
	{
	  /* back up one entry */
	  lseek(utmpfd, -(off_t)urecsize, SEEK_CUR);
	  if (write(utmpfd, (char *)&entry,urecsize) != urecsize)
	    report(LOG_ERR, "utmp_entry: couldn't write utmp file, %s",
		   sys_errlist[errno]);
#ifdef DEBUG
	  else
	    report (LOG_DEBUG, "utmp_entry: writing in erased utmp entry");
#endif
	  break;	/* out of while() */
	}	/* if (islogout) */
	else if (hostEq(oentry, entry) && lineEq(oentry, entry))
	{
	  report(LOG_INFO, "utmp_entry: overwriting existing entry for %.*s on %s@%.*s",
		 sizeof oentry.ut_name, oentry.ut_name, entry.ut_line,
		 sizeof entry.ut_host, entry.ut_host);
	  /* back up one entry */
	  lseek(utmpfd, -(off_t)urecsize, SEEK_CUR);
	  if (write(utmpfd, (char *)&entry, urecsize) != urecsize)
	    report(LOG_ERR, "utmp_entry: couldn't write utmp file, %s",
		   sys_errlist[errno]);
	  break;
	}
      }	/* end while() */


      if (n != urecsize)	/* end of file, so write new entry here */
      {
	if (write(utmpfd, (char *)&entry, urecsize) != urecsize)
	  report(LOG_ERR, "utmp_entry: couldn't write utmp file, %s",
		 sys_errlist[errno]);
#ifdef DEBUG
	else
	  report (LOG_DEBUG, "utmp_entry: writing utmp entry at end of file");
#endif
      }

    }	/* end: if (loginflag) */

    else	/* logout entry, so erase old entry. Line is ~ on reload */
    {
      while (read(utmpfd, (char *)&oentry, urecsize) == urecsize)
      {
	if (hostEq(oentry, entry))
	  if (*entry.ut_line == '~' || lineEq(oentry, entry))
	  {
	    /* back up one entry */
#ifdef DEBUG
	    report (LOG_DEBUG, "utmp_entry: deleting %.*s %.*s %.*s",
		    sizeof oentry.ut_host, oentry.ut_host,
		    sizeof oentry.ut_name, oentry.ut_name,
		    sizeof oentry.ut_line, oentry.ut_line);
#endif
	    lseek(utmpfd, -(off_t)urecsize, SEEK_CUR);
	    bzero(oentry.ut_line, sizeof oentry.ut_line);
	    bzero(oentry.ut_name, sizeof oentry.ut_name);	
	    bzero(oentry.ut_host, sizeof oentry.ut_host);

	    if (write(utmpfd, (char *)&oentry, urecsize) != urecsize)
	    {
	      report(LOG_ERR, "utmp_entry: write- %s", 
		     sys_errlist[errno]);
	      break;
	    }

	  }	/* end: if(same host + line) */

      }	/* end: while() */


    }	/* end: if-else new/erase entry */

#ifndef NOFSYNC
    fsync(utmpfd);
#endif

    interruptible = 1;

}	/* end: uwtmp_entry */

/*
 * Count how many times the given user is logged in. While counting,
 * skip the entry which matches the supplied host & line (i.e. in case
 * the person is going into PPP mode, dont deny because of the existing
 * login entry).
 */
cur_login_count(user, host, linenum)
     char *user;
     char *host;		/* which one is he coming in on */
     int  linenum;
{
    int count = 0;
    char sliline[20], ttyline[20];
    static int recsize ;
    struct tacutmp entry ;	/* to read from the utmp file */

    if (!recsize)		/* get it only first time */
      if ( (recsize = get_utmp_entrysize(utmpfd)) < 0 ) {
	  report(LOG_ALERT, "cur_login_count: get_utmp_entrysize() ret -1, FIX COMPILE TIME DEFINE XTACUTMP");
	  return (-1);
      }

    sprintf(ttyline, "%s%d", TS_TTY, linenum);
    sprintf(sliline, "%s%d", SLIP_TTY, linenum);

    lseek(utmpfd, 0, SEEK_SET);	/* rewind */
    while (read(utmpfd, (char *)&entry, recsize) == recsize) 
    {
      if (strncmp(entry.ut_name, user, sizeof entry.ut_name))
	continue;	/* no match in username */
      if (strncasecmp(entry.ut_host, host, sizeof entry.ut_host) == 0)
      {
	if (strncmp(entry.ut_line, ttyline, sizeof entry.ut_line) == 0 ||
	    strncmp(entry.ut_line, sliline, sizeof entry.ut_line) == 0)
	  continue;		/* dont count the current line */
      }
      ++count;
    }
    return(count);

}	/* cur_login_count() */

