/*
 * $Header: /home/vikas/src/xtacacsd/RCS/tacupd.c,v 1.9 1997/06/05 18:20:29 vikas Exp $
 */

/*
 * Manipulate the xtacacs created WTMP and UTMP files. 
 * DOES NOT DO ANY FILE LOCKING, meant to be used only in cases of
 * utmp & wtmp files getting out of sync with temrinal server in case
 * of lost tacacs packets.
 *
 * Use 'taclast' to print out and view the tacacs utmp or wtmp files.
 *
 * CAVEATS:
 *	Cannot specify the entry time in a more user friendly manner (have
 * to enter number of secs similar to output of the time() system call).
 * Note that too much manipulation might mess up the parsing algorithm of
 * taclast.
 *
 * Copyright (c) Vikas Aggarwal, vikas@navya.com	Nov 1995
 *
 * $Log: tacupd.c,v $
 * Revision 1.9  1997/06/05 18:20:29  vikas
 * Added line to prevent copying over reload entries in roll-overs.
 *
 * Revision 1.8  1997/05/30 11:11:06  vikas
 * Better help/usage message.
 *
 * Revision 1.7  1997/04/28 07:17:28  vikas
 * Now does NOT try to delete an entry from an ASCII wtmp file.
 *
 * Revision 1.6  1997/04/14 04:41:57  vikas
 * Added binary and ascii options (-A and -B).
 *
 * Revision 1.5  1997/01/09 11:02:53  vikas
 * Fixed bug in interactive deletion of entries (changed scanf to fgets)
 * Added support for ASCII wtmp files during rollover
 *
 * Revision 1.4  1996/05/21  17:43:13  vikas
 * Added Rollover of wtmp files (-R option).
 *
 * Revision 1.3  1996/05/19  17:48:22  vikas
 * Added options to convert the wtmp/utmp files to/from ascii.
 * Now uses the uwtmp_entry() routine from misc.c instead of
 * its own independent routine. Handles extended and pre-v4.0
 * utmp structures.
 *
 * Revision 1.2  1996/02/18  03:30:23  vikas
 * Can now specify the time in the entry also (in num of secs since 1970)
 * and also the 'change' option (instead of appending in the end of files)
 *
 * Revision 1.1  1995/11/10  22:16:42  vikas
 * Initial revision
 *
 */

#ifndef lint
static char rcsid[] = "$RCSfile: tacupd.c,v $ $Revision: 1.9 $ $Date: 1997/06/05 18:20:29 $" ;
#endif /* not lint */


#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "tacutmp.h"

#if defined(SYSV) || defined(SVR4) || defined(__svr4__)
#  define index strchr
#  define bcopy(a, b, c) memmove(b, a, c)
#  define bzero(a, b)    memset(a, 0, b)
#endif

#if !defined(HAVESYSERR) && !defined(__FreeBSD__) && !defined(__BSDI__)
extern char *sys_errlist[];
#endif

#ifdef tolower			/* because some OS's screwed up  */
# undef tolower
# define tolower(_c) (isupper(_c)?((_c)-'A'+'a'):(_c))
#endif

#define UT_BLANKSTR_FILL  "*"	/* filler for blank fields while dumping */

int cvttoascii, cvttoraw;
int rollover;			/* roll over wtmp files */
int clearentry;			/* delete entry in wtmp/utmp file */
int asciiwtmp;			/* ASCII wtmp file during rollover */
int utmpfd = -1, wtmpfd = -1;	/* file descriptors */
char *wtmp_file, *utmp_file;

/* Following are not really needed, but are external variables required
 * in the uwtmp.c module.
 */
int 	debug, logging, update_host_wtmp, stand, interruptible;
char	*from_hname;

main (ac, av)
     int ac;
     char *av[];
{
    int c;
    int islogin = 1;		/* by default, making a login entry */
    char *line, *name, *host;
    unsigned long tm = 0 ;
    extern char *optarg;
    extern int optind;

    line = name = host = "" ;
    if (!strcmp(av[0], "tacdump"))	/* if called as tacdump */
      cvttoascii++ ;

#ifdef ASCII_WTMP
    asciiwtmp = 1;		/* default */
#endif
    while ((c = getopt(ac, av, "acrABLRf:h:l:n:t:u:w:")) != EOF)
      switch (c) {
       case 'u':
	  utmp_file = optarg;
	  break;
       case 'f':
       case 'w':
	  wtmp_file = optarg;
	  break;
       case 'l':
	  line = optarg;
	  break;
       case 'R':
	  rollover = 1;
	  break;
       case 'n':
	  name = optarg;
	  break;
       case 'h':
	  host = optarg;
	  break;
       case 'A':		/* ASCII WTMP file in ROLLOVER */
	  asciiwtmp++ ;
	  break;
       case 'B':		/* Binary WTMP file in ROLLOVER */
	  asciiwtmp = 0 ;
	  break;
       case 'L':		/* LOGOUT entry */
	 islogin = 0;
	 break;
       case 'a':		/* convert TO ascii */
	  cvttoascii++ ;
	  break;
       case 'c':
	  clearentry++;
	  break;
       case 'r':		/* convert TO raw UTMP format */
	  cvttoraw++ ;
	  break;
       case 't':
	  sscanf(optarg, "%ld", &tm) ;
	  break;
       case '?':
       default:
	 fprintf(stderr, "Usage:\n");
	 fprintf(stderr, "  To ROLL OVER wtmp file:\n");
	 fprintf(stderr, "    %s -R [-A or -B] -w <old-wtmp-file>  -u <utmp-file>\n", av[0]);
	 fprintf(stderr, "\t\t -A for ascii wtmp file, -B for binary/raw wtmp file\n");

	 fprintf(stderr, "\n  To CONVERT files from/to ASCII to/from RAW format:\n");
	 fprintf(stderr, "    %s <-a or -r> -f <input file>\n", av[0]);
	 fprintf(stderr, "\t\t -a = convert to ASCII, -r = convert to raw\n");
	 fprintf(stderr, "\t\t (will write new output to stdout)\n");

	 fprintf(stderr, "\n  To CLEAR a utmp or wtmp file entry:\n");
	 fprintf(stderr, "    %s -c [-w wtmpfile OR -u utmpfile] -l lineno [-n name] -h host\n", av[0]);

	 fprintf(stderr, "\n  To ADD a new wtmp or utmp entry:\n");
	 fprintf(stderr, "    %s [-A or -B] [-L] -w wtmpfile  [-u utmpfile]", av[0]);
	 fprintf(stderr, " -l line_no  -h host  [-n name] [-t <timesecs>]\n");
	 fprintf(stderr, "\t\t (-A for ascii wtmp, -B for binary/raw wtmp)\n");
	 fprintf(stderr, "\t\t (-L to insert a LOGOUT entry)\n");
	 fprintf(stderr, "\t\t (Set name to ~ for host reboot)\n");
	 fprintf(stderr,"  CURRENT timesecs = %ld\n", time(NULL));
	 exit (1);
      }

#ifdef WTMP
    if (cvttoascii && wtmp_file == NULL)
	wtmp_file = WTMP;
#endif
    if (utmp_file == NULL && wtmp_file == NULL)  {
	fprintf (stderr, "ERROR: missing utmp & wtmp filenames\n");
	exit(1);
    }

    if (rollover) {
	if (utmp_file == NULL || wtmp_file == NULL) {
	    fprintf(stderr,"ERROR: missing wtmp or utmp files for rollover\n");
	    exit(1);
	}
	else
	  rollover_wtmp(utmp_file, wtmp_file);
    }

    if (cvttoascii && cvttoraw) {
	fprintf(stderr, "ERROR: only ONE of -a or -r should be specified\n");
	exit (1);
    }
    if (wtmp_file && cvttoascii)
      cvtto_ascii(wtmp_file);

    if (wtmp_file && cvttoraw)
      cvtto_raw(wtmp_file);

    if (*line == '\0' || *host == '\0') {
	fprintf (stderr, "ERROR: missing line or hostname\n");
	exit(1);
    }

    if (clearentry)
    {
      if (wtmp_file)
      {
	if (!asciiwtmp)
	  delete_wtmp_entry(wtmp_file, line, name, host);
	else
	{
	  printf("WTMP file is ASCII, delete the desired text line manually\n");
	  exit(1);
	}
      }
      if (utmp_file)
	delete_wtmp_entry(utmp_file, line, name, host);
      exit (0);
    }	/* if clearentry */

    /* this only to add an existing entry */
    uwtmp_entry(line, name, host, tm, /* comment */ "tacupd", islogin);
}	/* end main() */

/*
 * Function to test the size of the supplied file and see if the tacutmp
 * record size is evenly divisible. Else, try and assume an old style file
 * and delete the size of the UT_COMMENT and try again.
 * Required for backward compatibility.
 * Ditto as the similar function in uwtmp.c except that this uses fprintf()
 * instead of report().
 */
static int 
get_utmp_recsize(fd)
     int fd;		/* opened file descriptor */
{
    struct stat stb;
    struct tacutmp ut;
    int recsize;

    recsize = sizeof ut;

    if (fd < 0 || fstat(fd, &stb) == -1) {
	fprintf(stderr,"Error opening/stat-ing file %s\n", sys_errlist[errno]);
	return (-1);
    }

    if ( (stb.st_size % recsize) != 0)
    {
	fprintf(stderr,
		"WARNING: record size not compatible with compiled tacutmp (%d)\n", recsize);
	fprintf(stderr, "structure. This is controlled using the XTACUTMP compile time define.\n");

#ifdef XTACUTMP	/* new tacupd, old format utmp file */
	recsize = recsize - UT_COMMENTSIZE ;
	if (stb.st_size % recsize ) {
	  fprintf(stderr,"Cannot deal with it. Recompile WITHOUT XTACUTMP.\n");
	  return (-1);	/* should be treated as fatal */
	}
#else	/* old tacupd, new format utmp file, so just exit */
	fprintf(stderr, "Recompile WITH  XTACUTMP define and run again.\n");
	return (-1);
#endif	/* XTACUTMP */

    }	/* if (stb.st_size(...) */

    return (recsize);

}

/*+ Convert tacutmp structure to an ASCII string. Match this with the
 * printf statements in uwtmp.c  in uwtmp_entry()
 */
char *utmp2str(pentry, fmt)
  struct tacutmp *pentry;
  int fmt;			/* 1 if formatted string required */
{
  static char buf[256];		/* needs to be static */
  char *comment = UT_BLANKSTR_FILL;

#ifdef XTACUTMP
  if (pentry->ut_comment[0])    comment = pentry->ut_comment;
#endif

  /* print in a format that is easy to sort using Unix tools */
  if (pentry->ut_host[0] == '\0')
    strcpy(pentry->ut_host, UT_BLANKSTR_FILL);
  if (pentry->ut_line[0] == '\0')
    strcpy(pentry->ut_line, UT_BLANKSTR_FILL);
  if (pentry->ut_name[0] == '\0')
    strcpy(pentry->ut_name, UT_BLANKSTR_FILL);

  if (fmt)	/* constant width of fields for formatted */
    sprintf(buf, "%ld %-*.*s %-*.*s %-*.*s %s\n",
	    pentry->ut_time,
	    UT_HOSTSIZE, UT_HOSTSIZE, pentry->ut_host,
	    UT_LINESIZE, UT_LINESIZE, pentry->ut_line,
	    UT_NAMESIZE, UT_NAMESIZE, pentry->ut_name,
	    comment);
  else
    sprintf(buf, "%ld %.*s %.*s %.*s %s\n",
	    pentry->ut_time,
	    UT_HOSTSIZE, pentry->ut_host, UT_LINESIZE, pentry->ut_line,
	    UT_NAMESIZE, pentry->ut_name, comment);


  return (buf);
}	/* utmp2str() */

/*+
 * Graceful roll over of the wtmp file. For each user currently on the
 * system as per the UTMP file, it creates LOGOUT entries in the old
 * wtmp filename supplied and writes out corresponding new records to
 * the STDOUT (save to a file which will be the new wtmp file).
 */

rollover_wtmp()
{
    int recsize;
    int ufd, wfd;
    struct tacutmp  ut;
    time_t now;

    if ((ufd = open(utmp_file, O_RDONLY)) < 0) {
      perror(utmp_file);
      exit(1);
    }
    if ((wfd = open(wtmp_file, O_WRONLY | O_APPEND )) < 0) {
      perror(wtmp_file);
      exit(1);
    }

    if ( (recsize = get_utmp_recsize(ufd)) < 0)
      exit (1);

    bzero(&ut, sizeof ut);
    now = time(NULL);

    while (read(ufd, &ut, recsize) == recsize)
    {
      if (islogout(&ut))
	continue;

      if (ut.ut_line[0] == '~' && !ut.ut_line[1])	/* reload entry */
	continue;

      ut.ut_time = now;
      /* new entry to stdout */
      if (asciiwtmp)
      {
	char *s = utmp2str(&ut, /*fmt*/0);
	write(/*stdout*/ 1, s, strlen(s));
      }
      else
	write(/*stdout*/ 1, &ut, recsize);

      setlogout(&ut, /*login*/0);		/* create LOGOUT entry */
      /* write the logout entry to old file */
      if (asciiwtmp)		/* this is same as in uwtmp_entry.c */
      {
	char *s = utmp2str(&ut, /*fmt*/0);
	write(wfd, s, strlen(s));
      }
      else
	write(wfd, &ut, recsize);

      bzero(&ut, sizeof ut);
    }	/* while(read ufd) */

    close (ufd); close(wfd);

    exit(0);
}	/* end: rollover_wtmp() */


/*+ cvtto_ascii
 * Convert the wtmp file into ascii records. Helpful in manipulating the
 * entries. Check if the input file is of the old tacutmp structure.
 * If old format, tries to read those records. Dumps all records into
 * the new ASCII format.
 */

cvtto_ascii(file)
     char *file;
{
    int n, fd, recsize;
    struct tacutmp  ut;

    if ((fd = open(file, O_RDONLY)) < 0 )  {
	perror(file);
	exit(1);
    }

    if ( (recsize = get_utmp_recsize(fd)) < 0)
      exit (1);

    bzero(&ut, sizeof ut);

    /* print in a format that is easy to sort using Unix tools */
    while ((n = read(fd, (char *)&ut, recsize)) == recsize)
    {
      fputs(utmp2str(ut, /*fmt*/ 1), stdout);
      bzero(&ut, sizeof ut);
    }		/* end: while(read) */

    close(fd);
    exit(0);
}

/*
 * Convert ascii records to raw UTMP format
 */
#ifndef BUFSIZ
# define BUFSIZ 1024
#endif

cvtto_raw(file)
     char *file;
{
    long secs;
    char name[UT_NAMESIZE + 1], line[UT_LINESIZE + 1], host[UT_HOSTSIZE + 1];
    char comment[UT_COMMENTSIZE + 1];
    char buf[BUFSIZ];
    struct tacutmp entry;
    FILE *f;

    if ((f = fopen(file, "r")) == NULL) {
	perror(file);
	exit (1);
    }

    /*
     * set the utmp_file to NULL, so that uwtmp_entry() does not try
     * reopening it. Then set the wtmp_file to STDOUT and wtmpfd = 1
     * which is the stdout (terminal).
     */
    utmp_file = NULL;
    wtmp_file = "STDOUT";
    wtmpfd = 1;		/* stdout terminal */

    while (fgets(buf, BUFSIZ, f))
    {
	int n ;

	host[0] = line[0] = name[0] = comment[0] = '\0' ;
	if ((n = sscanf(buf, "%ld %s %s %s %s", &secs,host,line,name,comment)) < 4) {
	    fprintf(stderr,"No. of input fields < 4, ignoring '%s'",buf);
	    continue;
	}

	if (secs < 315550800l)	/* less than Jan 1, 1980 */
	{
	    fprintf (stderr, "Date older than 1980, ignoring '%s'", buf);
	    continue;
	}

	/* the '*' character was inserted by asciicvt for blank fields */
	if (!strncmp(line, UT_BLANKSTR_FILL, sizeof line))
	  line[0] = '\0';
	if (!strncmp(name, UT_BLANKSTR_FILL, sizeof name))
	  name[0] = '\0';
	if (!strncmp(host, UT_BLANKSTR_FILL, sizeof host))
	  host[0] = '\0';
#ifdef XTACUTMP
	if (!strncmp(comment, UT_BLANKSTR_FILL, sizeof comment))
	  comment[0] = '\0';
#else
	comment[0] = '\0';	/* null string */
#endif
	/* we set 'islogin' field to 2 so that uwtmp_entry() automatically
	 * figures out if this is a login or logout entry.
	 */
	uwtmp_entry(line, name, host, secs, comment, /*islogin*/2);

/*	write(1, &entry, sizeof entry);		/* write to stdout fd */

    }	/* end: while() */

    fclose(f);
    exit (0);
}

/*+ delete_wtmp_entry
 * Scan thru the UTMP/WTMP file and find entry matching the user supplied
 * arguments. since there can be multiple user+line+host matches, have
 * to ask the user if this is the correct one.
 */

delete_wtmp_entry(file, line, name, host)
  char *file;		/* file to be updated */
  char *line ;		/* Incoming line for user */
  char *name;		/* User's login name */
  char *host;		/* name/ip of host sending this info */
{
    int n = 0, fd, recsize ;
    char response[4];
    struct tacutmp entry;	/* tmp entry */

    printf("File %s\n", file);
    if ((fd = open(file, O_RDWR,  0664)) < 0) {
      perror(file);
      return (1);
    }

    if ((recsize = get_utmp_recsize(fd)) < 0)
      exit(1);

    while (read(fd, (char *)&entry, recsize) == recsize)
    {
      if (!strncmp(entry.ut_name, name, sizeof entry.ut_name))
	if (!strncmp(entry.ut_host, host, sizeof entry.ut_host))
	  if (!strncmp(entry.ut_line, line, sizeof entry.ut_line))
	  {
	    printf("%.*s@%.*s %.*s\nDelete this entry ? [y/n]: ",
		   sizeof entry.ut_name, entry.ut_name,
		   sizeof entry.ut_host, entry.ut_host,
		   sizeof entry.ut_line, entry.ut_line);

	    fgets(response, 4, stdin);
	    if (response[0] == 'Y' || response[0] == 'y')
	    {
	      lseek(fd, -(off_t)recsize, SEEK_CUR);
	      bzero(&entry, recsize);
	      if (write(fd, (char *)&entry, recsize) != recsize)
		fprintf(stderr, "utmp_entry: couldn't write file, %s\n",
			sys_errlist[errno]);
	      else
		fprintf(stdin, "Cleared entry successfully\n");
	    }
	  }
    }	/* end: while(read) */

    printf("Reached end of file\n");
    return (0);
}

