/* $Header: /home/vikas/src/xtacacsd/RCS/taclast.c,v 1.18 1998/01/05 05:26:24 vikas Exp $ */

/*
 * DERIVED FROM TACLAST v1.10 FOR TACACS ACCOUNTING
 *
 * This is similar to the 'last' program on Unix systems but for processing
 * my xtacacsd utmp/wtmp files (xtacacsd runs on Cisco routers).
 * Also acts like a simple 'who' program to read the data in the tacacs UTMP
 * file.
 *
 * Note that some login/logout records might be missing. In such a case:
 *	LOGIN - LOGOUT  = normal
 *	LOGIN - LOGIN   = LOGIN - (LOGOUT)LOGIN
 *	LOGOUT - LOGIN  = still logged on
 *	LOGOUT - LOGOUT = LOGOUT(LOGIN) - LOGOUT (missing first char)
 *
 * BUGS:
 *	While reading an ASCII wtmp file, it converts the read data into
 * the fixed length tacutmp structure. If the read data is larger then the
 * conversion might truncate the data.
 * Can avoid by setting the xx_NAMESIZE to be large from the beginning in
 * the tacutmp.h  header file.
 *
 * AUTHOR:
 *  	Vikas Aggarwal, vikas@navya.com, June 1995
 *
 * Partially derived from:
 *	$ NetBSD: last.c,v 1.6 1994/12/24 16:49:02 cgd Exp
 *
 * $Log: taclast.c,v $
 * Revision 1.18  1998/01/05 05:26:24  vikas
 * Changed all strncmp() to strncasecmp() for the usernames since the
 * total's were all getting messed up in SLIP and non-SLIP logins.
 *
 * Revision 1.17  1997/07/16 06:28:57  vikas
 * Compiler for AIX defines _AIX, just added that.
 *
 * Revision 1.16  1997/05/28 20:12:45  vikas
 * Cleaned up fetch_tacutmp_ascii() & added str2utmp(). Small bug
 * fixes here and there.
 *
 * Revision 1.15  1997/04/25 09:51:24  vikas
 * - added crashtm stucture, now keeping track of all crash times for
 *   each host.
 * - was not saving the hostname in the TTY structure. Was causing all
 *   entries to become inaccurate.
 * - Remaining BUG: not reading the first line of ascii wtmp file properly
 *
 *
 * Revision 1.10  1997/01/11 10:54:45  vikas
 * Now sees if hostname on logout is NULL (as on freeBSD)
 * Also prints out all users that dont have a login when the
 * end of the wtmp file is reached.
 * Tested version.
 *
 * Revision 1.9  1997/01/09 11:05:26  vikas
 * Now using the islogout() function. Also handling the case if
 * we get 2 consecutive logout's.
 *
 * Revision 1.8  1997/01/08 10:52:26  vikas
 * Did not need the & in the structure elements during sscandf()
 *
 * Revision 1.7  1997/01/07 10:36:45  vikas
 * (older logs are in the RCS directory)
 * -Added code to handle ASCII wtmp files fetch_tacutmp_ascii()
 * -Separated printf's in the wtmp code
 * -Now keeps and prints the incorrect times in the totals
 *
 */


#ifndef lint
static char rcsid[] = "$RCSfile: taclast.c,v $ $Revision: 1.18 $ $Date: 1998/01/05 05:26:24 $" ;
#endif /* not lint */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>		/* for MAXHOSTNAMELEN on SCO ? */
#include <sys/stat.h>

#include <fcntl.h>
#include <sys/file.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#if defined(AIX) || defined(_AIX)
# include <time.h>			/* struct tm defined here on AIX3 */
#endif
#include <unistd.h>			/* for SEEK_ definitions */

#include "tacutmp.h"			/* our very own utmp.h file */

#ifndef	SECSPERDAY
# define SECSPERDAY	((long) (60 * 60 * 24))
#endif
#ifndef DAYSPERWEEK
# define DAYSPERWEEK	7
#endif

#ifndef MAXHOSTNAMELEN
# define MAXHOSTNAMELEN 128
#endif

#if defined(SYSV) || defined(SVR4) || defined(__svr4__)
#  define index strchr
#  define bcopy(a, b, c) memmove(b, a, c)
#  define bzero(a, b)    memset(a, 0, b)
#endif

#define	NO	0				/* false/no */
#define	YES	1				/* true/yes */

#ifdef TACACCT
# include "tacacct.h"
#else
/* Following structure is used to gather time information */
typedef struct arg {
  char		*name;				/* argument */
  time_t	errsecs;			/* inaccurate seconds */
  time_t  	oksecs;				/* confirmed seconds  */
  time_t	maxsecs;			/* largest online time */
  time_t	count;				/* total number of entries */
#define	HOST_TYPE	-2
#define	TTY_TYPE	-3
#define	USER_TYPE	-4
  struct arg		*next;			/* linked list pointer */
} ARG;
#endif	/* TACACCT */

ARG	*HOSTarglist, *USERarglist;		/* head of linked list */

typedef struct ttytab {
  time_t	logout;				/* log out time */
  int	waslogin;				/* boolean for prev entry */
  char	tty[UT_LINESIZE + 1];			/* terminal name */
  char	host[UT_HOSTSIZE + 1];			/* host name */
  char	name[UT_NAMESIZE + 1];			/* user on this TTY */
  struct ttytab	*next;				/* linked list pointer */
} TTY;
TTY	*ttylist;				/* head of linked list */

struct _crashtm {
  time_t	crash;
  char		host[UT_HOSTSIZE + 1];
  struct _crashtm *next;
} *crashtm;


#ifndef MAX
# define MAX(A,B)      ((A) > (B) ? (A) : (B))
#endif

static int	asciiwtmp;			/* use ascii WTMP file */
static int	whomode;			/* do a 'who', not 'last' */
static int	ignorehost= 0;			/* dont compare hostnames */
static int	dototals;			/* want totaltimes */
static int	totalsonly;			/* want totaltimes ONLY */
static long	maxrec;				/* records to display */
static char	*file;				/* wtmp file */
static char	*weekdays[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", 
			       "Sat", "Sun"};

int	noargs;					/* no cmd line args */
time_t	tstzone = 0;				/* term server timezone */
time_t	loctzone =0;				/* local timezone */

/* Following are not really needed, but are external variables required
 * in the uwtmp.c module.
 */
int	wtmpfd, utmpfd;
int 	logging, update_host_wtmp, stand, interruptible;
char	*from_hname, *wtmp_file, *utmp_file;

void	 addarg (/* int, char * */);
TTY	*addtty (/* char * */);
void	 hostconv (/* char * */);
char	*ttyconv (/* char * */);
int	 want (/* struct tacutmp *, int */);
void	 wtmp (/* void */);
void	add_crash_entry();
void	str2utmp();
time_t	 pr_entry(/* struct tacutmp *, TTY * */);
int	pr_totaltimes();

char	*prognm;
int	debug;

main(argc, argv)
  int argc;
  char *argv[];
{
  extern int optind;
  extern char *optarg;
  int ch;
  char *p, *cfile = NULL;

  if ((prognm = (char *)strrchr(argv[0], '/')) == NULL)
    prognm = argv[0];
  else
    ++prognm;

  if (!strcmp(prognm, "tacwho"))		/* if called as tacwho */
    whomode = 1;

#ifdef ASCII_WTMP
  asciiwtmp = 1;		/* default files are ascii */
#endif
  maxrec = -1;
  while ((ch = getopt(argc, argv, "0123456789abc:df:h:itw")) != EOF)
    switch (ch) {
    case '0': case '1': case '2': case '3': case '4':
    case '5': case '6': case '7': case '8': case '9':
      /*
       * kludge: last was originally designed to take
       * a number after a dash.
       */
      if (maxrec == -1) {
	p = argv[optind - 1];
	if (p[0] == '-' && p[1] == ch && !p[2])
	  maxrec = atol(++p);
	else
	  maxrec = atol(argv[optind] + 1);
	if (!maxrec)
	  exit(0);
      }
      break;
    case 'a':
      asciiwtmp++;
      break;
    case 'b':		/* binary WTMP files */
      asciiwtmp = 0;
      break;
#ifdef TACACCT
    case 'c':
      cfile = optarg;
      break;
#endif
    case 'd':
      debug++;
      break;
    case 'f':
      file = optarg;
      break;
    case 'h':
      /*			hostconv(optarg);	/* not used */
      addarg(HOST_TYPE, optarg);
      break;
    case 'i':
      ignorehost++;		/* if separate wtmp files */
      break;
    case 't':
      dototals++;		/* if 2, then only print totals */
      if (dototals >= 2) totalsonly++;
      break;
    case 'w':
      whomode++;
      break;
    case '?':
    default:
      fprintf(stderr,
		    "usage: %s [-d (debug)] [-# (max num records)] [-f file] [-a ascii-wtmp-file] [-b binary-wtmp-file] [-h hostname] [-i (for ignore host)]", prognm);
      fprintf(stderr,
		    " [-w (for WHO mode)] [-t (totaltimes) [-t (only totals)]]");
#ifdef TACACCT
      fprintf(stderr, " [-c config-file] [user ...]\n");
#else
      fprintf(stderr, "[user ...]\n");
#endif
      exit(1);
    }

  if (argc) {
#ifndef POSIX
    setlinebuf(stdout);		/* */
#else
    setbuf(stdout, NULL);	/* */
#endif
    for (argv += optind; *argv; ++argv) {
      addarg(USER_TYPE, *argv);
    }
  }

#ifdef TACACCT
  loctzone = local_gmtoffset();
  timerange_init(cfile);	/* default config file used if NULL */
#endif

  if (!HOSTarglist && !USERarglist)
    noargs = 1;			/* no cmd line arguments */

  if (whomode) doutmp() ;
  else  wtmp();

  exit(0);
}	/* end main() */


/*+
 * Return one filled in tacutmp structure. Return NULL on end of file.
 * Work backwards in the file.
 * Works on ASCII wtmp files.
 * CAVEAT: has to truncate off the long names only because the tacutmp
 * structure has arrays of characters (fixed length), not pointers.
 */
struct tacutmp *fetch_tacutmp_ascii()
{
  static struct tacutmp entry;
  static char buf[4096];
  register char *p;
  char *line;
  static int curoffset, startoffset;
  static int nread, bufsz, numlines;
  static int wfd = -1;
  static char *curlineptr, *bufstart, *bufend;

  if (wfd < 0)	/* first time */
  {
    struct stat	stb;			/* stat of file for size */

    if (file == (char *)NULL || *file == '\0')
    {
      file = (char *)malloc(strlen(WTMP) + strlen(".ascii") + 2);
      sprintf(file, "%s%s", WTMP, ".ascii");
    }

    if ((wfd = open(file, O_RDONLY, 0)) < 0 || fstat(wfd, &stb) < 0) {
      fprintf(stderr, "%s: %s: ", prognm, file);
      perror ((char *)NULL);
      exit (1);
    }
    curoffset = stb.st_size;

    bufsz = sizeof(buf);
  }	/* if wfd < 0 */

nextline:
  if (curoffset <= 0 && curlineptr == bufstart)
  {
    if (debug)
      fprintf(stderr, "Processed %d lines\n", numlines);
    return (NULL);
  }

  if (curlineptr == bufstart)	/* finished curr buffer, need more data */
  {
    startoffset = (curoffset <= bufsz) ? 0 : (curoffset - bufsz);
    if (lseek(wfd, (off_t)startoffset, SEEK_SET) < 0 ||
	(nread = read(wfd, buf, curoffset - startoffset)) < 0)
    {
      fprintf(stderr, "(%s) %s: ", prognm, file);
      perror((char *)NULL);
      exit (1);
    }

    bufstart = buf;	/* set equal now, adjusted later */

    /* Now move forward until we hit a \n */
    if (startoffset > 0)	/* not at beginning of file */
    {
      for (p = buf ; *p != '\n' && nread > 0 ; ++p, --nread)
	;
      if (*p != '\n')	/* no newline in entire buffer, screwed up */
      {
	fprintf(stderr,"fetch_ascii() No newline in entire buffer, exiting\n");
	exit (1);
      }
      bufstart = p+1;	/* save new buffer start location (after a \n) */
      --nread;
    }

    curoffset -= nread;	/* point to next unread byte */

    /* Now move backwards from the end until we get a \n (make sure the
     * block ends on a \n). Only really required the first time.
     */
    for (p = bufstart + nread - 1; *p != '\n' && p > bufstart; --p)
      ;
    if (*p != '\n')   curlineptr = bufstart + nread - 1;
    else  curlineptr = p;

  }	/* end if curlineptr == bufstart */

  /* Here we are assured we have a bufstart beginning right after a \n
   * and curlineptr pointing to the end of the unread buffer at a \n.
   */
  for (*curlineptr = '\0'; *curlineptr != '\n' && bufstart < curlineptr; )
    --curlineptr ;

  if (*curlineptr == '\n')
    line = curlineptr + 1;
  else
    line = curlineptr;	/* curlineptr == bufstart */

  ++numlines;

  /* To test if this reversal is working properly, uncomment following line */
  /* fprintf(stderr, "%s\n", line); goto nextline;	/*  */

  /* Here, line points to a line that we need to parse */
  str2utmp(line, &entry);
  if (entry.ut_time == 0)
  {
    if (debug)
      fprintf(stderr, "  Current linenum from bottom of file= %d\n", numlines);
    goto nextline;
  }

  return (&entry);	/*  */
}	/* fetch_tacutmp_ascii() */


void str2utmp(str, pentry)
  char *str;
  struct tacutmp *pentry;
{
  register char *s, *t = str;

#define NextWord(s,t) \
    while (*t && (*t == ' ' || *t =='\t')) \
      ++t; /* skip spaces */ \
    for (s = t ; *t && *t != ' ' && *t != '\t' ; ++t) \
      ; \
    *t++ = '\0' ; \
    if (debug > 3) fprintf(stderr, ">> %s ", s);

    NextWord(s,t);
    if (*s < '0' || *s > '9')	/* not a number */
    {
      fprintf(stderr, "Bad timestamp in WTMP file '%s'\n", s);
      pentry->ut_time = 0;
      return ;
    }
    pentry->ut_time = atol(s);
    
    NextWord(s,t);
    strncpy(pentry->ut_host, s, sizeof(pentry->ut_host));
    
    NextWord(s,t);
    strncpy(pentry->ut_line, s, sizeof(pentry->ut_line));
    
    NextWord(s,t);
    strncpy(pentry->ut_name, s, sizeof(pentry->ut_name));

#ifdef XTACUTMP
    NextWord(s,t);	/* hope we dont exceed past end of string */
    strncpy(pentry->ut_comment, s, sizeof(pentry->ut_comment));
#endif
    if (debug > 3) fprintf(stderr,"\n");   /* a newline after debug msg */
#undef NextWord

}	/* str2utmp */

/*+
 * Return one filled in tacutmp structure. Return NULL on end of file.
 * Work backwards in the file.
 * This is for binary fixed record length wtmp files.
 */
struct tacutmp *fetch_tacutmp()
{
  static struct tacutmp  tacbuf[1024];		/* utmp read buffer */
  static int  bytes, wfd;
  static int  firsttime = 1;
  static int cur = 0;			/* current tacbuf number */
  static long bl;

  if (asciiwtmp)
    return(fetch_tacutmp_ascii());

  if (firsttime)
  {
    struct stat	stb;			/* stat of file for size */

    firsttime = 0;
    if (file == (char *)NULL || *file == '\0')
      file = WTMP ;

    if ((wfd = open(file, O_RDONLY, 0)) < 0 || fstat(wfd, &stb) == -1) {
      fprintf(stderr, "%s: %s: ", prognm, file);
      perror ((char *)NULL);
      exit (1);
    }

    if ( (stb.st_size % sizeof tacbuf[0]) != 0) {
      fprintf(stderr,
	      "(taclast) ERROR: WTMP record sizes not compatible with compiled tacutmp structure.\n");
      fprintf(stderr,
	      "This is controlled using the XTACUTMP compile time define.\n");
#ifdef XTACUTMP
      fprintf(stderr, "Recompile program without XTACUTMP.\n");
#else
      fprintf(stderr, "Recompile program with XTACUTMP.\n");
#endif
      exit(1);
    }

    /* extract total # of reads */
    bl = (stb.st_size + sizeof(tacbuf) - 1) / sizeof(tacbuf);

  }	/* end if (firsttime) */

  /* work backwards in the file */
  if (bl > 0 || cur > 0)
  {
    if (cur == 0)
    {			/* read in a new block of data */
      --bl;
      if (lseek(wfd, (off_t)(bl * sizeof(tacbuf)), SEEK_SET) == -1 ||
	  (bytes = read(wfd, tacbuf, sizeof(tacbuf))) == -1)
      {
	fprintf(stderr, "%s: %s: ", prognm, file);
	perror((char *)NULL);
	exit (1);
      }
      cur = bytes / sizeof(tacbuf[0]);
    }

    return (&tacbuf[--cur]);
  }

  return(NULL);
}	/* end fetch_tacutmp() */

/*
 * wtmp --
 *	read through the wtmp file
 */
void
wtmp()
{
  struct tacutmp  *bp;				/* current structure */
  TTY	*T;					/* tty list entry */
  time_t start, delta;				/* time difference */
  time_t prevtm;			/* prev entry time */
  char	*ct, *crmsg;

  time(&start);			/* store start time */

  while ( (bp = fetch_tacutmp()) != NULL)
  {
    prevtm = bp->ut_time;	/* need this at the end... */
    /*
     * if the terminal line is '~', the machine stopped.
     */
    if (bp->ut_line[0] == '~' && !bp->ut_line[1])
    {
#ifdef DEBUG
      if (debug > 3)
	fprintf(stderr,"(debug) xreload %*s entry\n", UT_HOSTSIZE,bp->ut_host);
#endif
      /* everybody just logged out. Store timestamp as negative */
      for (T = ttylist; T; T = T->next) {
	if (ignorehost || strncasecmp(T->host, bp->ut_host, UT_HOSTSIZE) == 0 )
	  T->logout = -bp->ut_time;	/* indicate stopped entry */
      }
      add_crash_entry(bp->ut_host, bp->ut_time);	/* when host crashed */

      if (strncmp(bp->ut_name, "xreload", UT_NAMESIZE) == 0)
	crmsg = "xreload";
      else if (strncmp(bp->ut_name, "shutdown", UT_NAMESIZE) == 0)
	crmsg = "shutdown";
      else
	crmsg = "crash";
      if (want(bp))
	if (!totalsonly)
	{
	  ct = ctime(&bp->ut_time);
	  printf("%-*.*s  %-*.*s %-*.*s %10.10s %5.5s \n",
		 UT_NAMESIZE, UT_NAMESIZE, bp->ut_name,
		 UT_LINESIZE, UT_LINESIZE, bp->ut_line,
		 UT_HOSTSIZE, UT_HOSTSIZE, bp->ut_host,
		 ct, ct + 11);
	  if (maxrec != -1 && !--maxrec)
	    return;
	}
      continue;
    }	/* end if machine stopped */

    /*
     * find associated tty in TTY list. A different node exists for each
     * TTY + host combination.
     */
    for (T = ttylist; T ; T = T->next) {
      if (!strncmp(T->tty, bp->ut_line, UT_LINESIZE))	/* line matches */
      {
	if (ignorehost)
	  break;
	/* Some hosts don't log a hostname on logouts */
	if (*(T->host) == '\0' ||
	    strncasecmp(T->host, bp->ut_host, UT_HOSTSIZE) == 0)
	  break;
      }
    }

    if (T && want(bp))
    {
      if (!islogout(bp))	    /* if this is a login entry */
      {
	int isaccurate = 0;	/* assume incorrect */

	/*
	 * Check for record correctness- if prev was a logout and:
	 *	if prev username == NULL (old style logouts)
	 *     OR if prev username == curr name
	 *     OR	if prev username == LOGOUT_CHAR + curr name
	 */
	if (!T->waslogin)
	  if (*(T->name) == '\0' ||
	      !strncasecmp(bp->ut_name, T->name, sizeof(bp->ut_name)) ||
	      (*(T->name) == UT_LOGOUT_CHAR &&
	       !strncasecmp((bp->ut_name)+1, (T->name)+1, sizeof(bp->ut_name)-1)) )
	    isaccurate = 1;

	if (!T->logout)         /* person is still online, set isaccurate */
	  isaccurate = 1;
	if (debug > 1 && !isaccurate)
	  fprintf(stderr,
		  "(debug) Inaccurate login entry %.*s@%-.*s, no matching logout\n",
		  UT_NAMESIZE, bp->ut_name, UT_HOSTSIZE, bp->ut_host);

	delta = pr_entry(bp, T, isaccurate, crmsg);

	if (dototals)  addsecs(bp, delta, isaccurate);
	if (maxrec != -1 && !--maxrec)
	  return;
      }	/* end if login entry */
      else
      {	/* we got a logout entry. Check if previous also a logout... */
	if (!T->waslogin && bp->ut_name[0])
	{		/* prev also a logout & we have a name to check with */
	  /*
	   * If this username is same as previous, then we CAN assume this is
	   * an extra logout and skip. However, it is more logical to
	   * flag this entire entry as inaccurate and simply assume that
	   * the prev user logged in right when this user logged out. e.g.
	   *	vikas 10:15 pm logout
	   *	vikas 10:10 pm logout
	   *	vikas 10:05 pm logout << probably missing login entries
	   *	joe   10:00 pm logout
	   * We might be missing the first character of this user in older
	   * wtmp data files...
	   */
	  if (debug > 1)
	    fprintf(stderr,
		    "(debug) Extra logout entry  %.*s@-%.*s, prev also logout\n",
		    UT_NAMESIZE, bp->ut_name, UT_HOSTSIZE, bp->ut_host);
	  delta = pr_entry(bp, T, /*isaccurate*/0, crmsg);
	  if (dototals)  addsecs(bp, delta, /*isaccurate*/0); /* */
	}	/* end if (T->waslogin) */

      }	/* end if else (login) */

    }	/* end if want(bp) */

    if (!T)		/* add new TTY to linked list */
    {
      T = addtty(bp->ut_line, bp->ut_host, bp->ut_name);
      if (!islogout(bp) && want(bp))	/* person still online */
      {
	delta = pr_entry(bp, T, /*isaccurate*/ 1, crmsg);	/* online */
	if (dototals) addsecs(bp, delta, /*isaccurate*/ 1);
      }
    }

    /* 	We always store this bp->ut_time. If was login entry, then
     *	serves as the logout in case the logout entry is lost for the
     *  previous person
     */
    T->logout = bp->ut_time;
    /* Update this at the very end since its earlier value was required */
    if (islogout(bp))    T->waslogin = 0 ;
    else  T->waslogin = 1;
    strncpy(T->name, bp->ut_name, UT_NAMESIZE);	/* save prev name */
  }	/* end while(bp != NULL) */

  /* At end of the wtmp file, close out all LOGOUT's in the TTY list */
  for (T = ttylist; T != NULL; T = T->next)
    if (!T->waslogin)	/* logout entry */
    {
      struct tacutmp tbp;		/* we recreate the struct from TTY */

      if (debug)
	fprintf(stderr, "(debug) No login entry for %s@%s at end of file\n",
		T->name, T->host);
      strncpy(tbp.ut_name, T->name, UT_NAMESIZE);	/* from T to tbp */
      strncpy(tbp.ut_line, T->tty, UT_LINESIZE);
      strncpy(tbp.ut_host, T->host, UT_HOSTSIZE);
#ifdef XTACUTMP
      strncpy(tbp.ut_comment, "Login before start of file", UT_COMMENTSIZE);
#endif
      tbp.ut_time = prevtm;
      if (want(&tbp))
      {
	delta = pr_entry(&tbp, T, /*isaccurate*/0, "");
	if (dototals)  addsecs(&tbp, delta, /*isaccurate*/0);
      }
    }

  /* print out the total times */
  if (dototals)  pr_totaltimes();

  printf("\nwtmp processing began %24.24s\n", ctime(&start));
  printf("        (first record %24.24s)\n", ctime(&prevtm));

}	/* end: wtmp() */


/*
 * want --
 *	see if want this entry (in case some hosts or username were
 * specified on the command line).
 */
int
want(bp)
  struct tacutmp *bp;
{
  ARG *step;

  if (noargs)
    return (YES);
    
  for (step = HOSTarglist; step; step = step->next)
    if (!strncasecmp(step->name, bp->ut_host, UT_HOSTSIZE))
      return (YES);

  for (step = USERarglist; step; step = step->next)
    if (!strncasecmp(step->name, bp->ut_name, UT_NAMESIZE))
      return (YES);
    
  return (NO);

}

/*
 * add_crash_entry --
 *	add the time that a host crashed into the linked list or update
 * the previous value. these are used by addTTY() as logout values.
 * Store the values as -ve so that pr_time() can detect that the server
 * had crashed.
 */
void add_crash_entry(host, crtime)
  char *host;
  time_t crtime;
{
  struct _crashtm *T;

  for (T = crashtm; T ; T = T->next)
    if (strncasecmp(T->host, host, UT_HOSTSIZE) == 0)
    {
      T->crash = -crtime;
      return;
    }
  /* Add a new entry */
  T = (struct _crashtm *)malloc(sizeof (struct _crashtm));
  strncpy(T->host, host, UT_HOSTSIZE);
  T->crash = -crtime;
  T->next = crashtm;
  crashtm = T;

  return ;
}	/* add_crash_entry() */

/*
 * addarg --
 *	add an entry to a linked list of arguments (used if a list of
 * username or hostname is specified on the command line or  we need
 * to add to these linked lists).
 */
void
addarg(type, arg)
  int type;
  char *arg;
{
  ARG *cur, **whichlist;

  if (!(cur = (ARG *)malloc((u_int)sizeof(ARG)))) {
    fprintf(stderr, "%s: malloc failure.\n", prognm);
    exit (1);
  }
  bzero(cur, sizeof (ARG));

  switch (type) {
  case HOST_TYPE:
    whichlist = &HOSTarglist;	
    break;
  case USER_TYPE:
    whichlist = &USERarglist;
    break;
  default:
    fprintf(stderr, "addarg: unknown type %d\n", type);
  }
  cur->next = *whichlist;
  cur->name = arg;
  *whichlist = cur;

#ifdef TACACCT
  cur->ac = (struct acctstats *)malloc(sizeof(struct acctstats));
  bzero(cur->ac, sizeof(struct acctstats));
#endif
}	/* addarg() */

/*
 * addtty --
 *	add an entry to a linked list of ttys. Search in crashtm linked list
 * for hostname, and use that as the logout time (the crash time of the
 * server = logout time).
 */
TTY *
addtty(ttyname, hostname, username)
  char *ttyname, *hostname, *username;
{
  TTY *cur;
  struct _crashtm *T;

  if (!(cur = (TTY *)malloc((u_int)sizeof(TTY)))) {
    fprintf(stderr, "%s: malloc failure.\n", prognm);
    exit (1);
  }
  bzero(cur, sizeof(TTY));

  for (T = crashtm ; T ; T = T->next)
    if (strcasecmp(T->host, hostname) == 0)
    {
      cur->logout = T->crash;
      break;
    }

  cur->next = ttylist;
  cur->waslogin = 0;	/* initialization- logout. Changed later... */

  /* since the size of the TTY fields are UT_ + 1, we can use strncpy */
  strncpy(cur->tty,  ttyname,  UT_LINESIZE);
  strncpy(cur->host, hostname, UT_HOSTSIZE);
  strncpy(cur->name, username, UT_NAMESIZE);
  return (ttylist = cur);
}


#ifndef TACACCT
/*
 * Add the delta seconds to the total number of seconds. Add to the arglist
 * if this is a new host or name and nothing was supplied on the command line.
 */
addsecs(bp, delta, isaccurate)
  struct tacutmp *bp;
  time_t delta;
  int isaccurate;		/* whether delta is confirmed accurate */
{
  ARG *steph, *stepu;

  /* ** HOST list ** */
  for (steph = HOSTarglist; steph; steph = steph->next)
    if (!strncasecmp(steph->name, bp->ut_host, UT_HOSTSIZE))
      break ;
  if (!steph && noargs) {		/* new host, add to linked list */
    char *s = (char *)malloc(UT_HOSTSIZE + 1);
    strncpy (s, bp->ut_host, UT_HOSTSIZE);
    addarg(HOST_TYPE, s);
    steph = HOSTarglist;
  }
  /* ** USER list ** */
  for (stepu = USERarglist; stepu; stepu = stepu->next)
    if (!strncasecmp(stepu->name, bp->ut_name, UT_NAMESIZE))
      break ;
  if (!stepu && noargs) {
    char *s = (char *)malloc(UT_NAMESIZE + 1);
    strncpy (s, bp->ut_name, UT_NAMESIZE);
    addarg(USER_TYPE, s);
    stepu = USERarglist;
  }

  if (steph) {
    steph->maxsecs = MAX(steph->maxsecs, delta);	/* update max time */
    ++(steph->count);
  }
  if (stepu) {
    stepu->maxsecs = MAX(stepu->maxsecs, delta);	/* update max time */
    ++(stepu->count);
  }

  if   (isaccurate) {
    if (steph) steph->oksecs += (long)delta;
    if (stepu) stepu->oksecs += (long)delta;
  }
  else {
    if (steph) steph->errsecs += (long)delta;
    if (stepu) stepu->errsecs += (long)delta;
  }

}	/* end: add_secs() */

#endif  /* TACACCT */

/*
 * print the total time for each user supplied argument.
 */
#ifndef TACACCT
pr_totaltimes()
{
  ARG *step;
    
  printf("\nTOTAL TIMES (d+hh:mm) (Total/Err)\n===========\n");
    
  for (step = USERarglist; step; step = step->next)
  {
    time_t tsecs = (step->errsecs + step->oksecs) ;

    printf("USER %s", step->name);
    if (tsecs < SECSPERDAY)
      printf("  (%5.5s)", asctime(gmtime(&tsecs))+11);
    else
      printf(" (%ld+%5.5s)",
	     tsecs / SECSPERDAY, asctime(gmtime(&tsecs))+11);
    printf(" %ld / %ld\n", tsecs, step->errsecs);
  }	/* for() */
  printf("\n");

  for (step = HOSTarglist; step; step = step->next)
  {
    time_t tsecs = (step->errsecs + step->oksecs) ;

    printf("HOST %s", step->name);
    if (tsecs < SECSPERDAY)
      printf("  (%5.5s)", asctime(gmtime(&tsecs))+11);
    else
      printf(" (%ld+%5.5s)",
	     tsecs / SECSPERDAY, asctime(gmtime(&tsecs))+11);
    printf(" %ld / %ld\n", tsecs, step->errsecs);
  }	/* for() */

}	/* pr_totaltimes() */

#endif /* TACACCT */

/*
 * Print out the entry. Also calculates the delta time (secs the user
 * was logged in) and returns that.
 * Extracts and sets the global 'tstzone' timezone of the terminal server.
 */

time_t
pr_entry(bp, T, isaccurate, crmesg)
  struct tacutmp *bp;
  TTY *T;
  int isaccurate;	/* this login entry and times are accurate */
  char *crmesg;		/* crash message */
{
  char *ct;
  time_t delta, tmlogin, tmlogout;
  int crash = 0;

#ifdef TACACCT
  if ( (tstzone = get_tsgmtoffset(bp->ut_host)) == -1 )
    tstzone = loctzone;		/* use local timezone */
#endif

  if (T->logout < 0)	/* stored as -ve on reboot */
  {
    crash = 1;
    T->logout = -T->logout;
  }
  delta = ( T->logout ? T->logout:time(NULL) ) - bp->ut_time;

  if (totalsonly)
  {
    if (delta < 0)
      return (0);
    else
      return (delta);
  }

  tmlogin  = bp->ut_time + tstzone;
  tmlogout = T->logout + tstzone;

#ifdef TACACCT
  ct = asctime(gmtime(&tmlogin));	/* use term server's timezone */
#else
  ct = ctime(&tmlogin);		/* use local timezone */
#endif
  printf("%-*.*s  %-*.*s %-*.*s %10.10s %5.5s ",
	 UT_NAMESIZE, UT_NAMESIZE, bp->ut_name,
	 UT_LINESIZE, UT_LINESIZE, bp->ut_line,
	 UT_HOSTSIZE, UT_HOSTSIZE, bp->ut_host,
	 ct, ct + 11);

  if (!T->logout)		/* indicates that person is still online */
    printf(" online");
  else if (crash)
    printf("- %s", crmesg);	/* shutdown or crash */
  else
  {
#ifdef TACACCT
    ct = asctime(gmtime(&tmlogout));
#else
    ct = ctime(&tmlogout);		/* use local timezone */
#endif
    printf("- %5.5s", ct+11);
  }

  if (delta < 0) {	/* should not happen */
    printf(" (negative)");
    delta = 0;
  }
  else if (delta < SECSPERDAY)
    printf("  (%5.5s)", asctime(gmtime(&delta))+11);
  else
    printf(" (%ld+%5.5s)", delta / SECSPERDAY, asctime(gmtime(&delta))+11);

  printf(" %ld%s\n", delta, isaccurate ? " ":" inaccurate");

  return (delta);

}	/* end pr_entry() */


/*
 * hostconv --
 *	convert the hostname to search pattern; if the supplied host name
 *	has a domain attached that is the same as the current domain, rip
 *	off the domain suffix since that's what login(1) does.
 */
void
hostconv(arg)
  char *arg;
{
  static int first = 1;
  static char *hostdot, name[MAXHOSTNAMELEN];
  char *argdot;

  if (!(argdot = strchr(arg, '.')))
    return;
  if (first) {
    first = 0;
    if (gethostname(name, sizeof(name))) {
      perror("last: gethostname");
      exit (1);
    }
    hostdot = strchr(name, '.');
  }
  if (hostdot && !strcasecmp(hostdot, argdot))
    *argdot = '\0';
}

/*
 * Following are 'who' routines for printing out the data in the
 * UTMP file to show users currently logged in.
 */

doutmp()
{
  time_t now, delta;
  struct tacutmp u ;
  FILE   *ut;
  int    nusers = 0, tusers = 0;	/* wanted users, total users */
  int ib;

  now = time(NULL);

  if (file == (char *)NULL || *file == '\0')
    file = UTMP ;
  if ((ut = fopen(file, "r")) == NULL) {
    fprintf(stderr, "%s: %s: ", prognm, file);
    perror ((char *)NULL);
    exit (1);
  }

  /* This little mumbo-jumbo to align the header -furio@spin.it */
  (void)printf("USER");
  for (ib = UT_NAMESIZE-4; ib > 0 ; ib--)
    printf(" ");
  printf(" TTY      FROM             LOGIN@ ONLINE(d+h:m)\n");

  while (fread(&u, sizeof(u), 1, ut)) {
    if (islogout(&u))
      continue;
    ++tusers;
    if (want(&u))  {
      (void)printf("%-*.*s %-*.*s %-*.*s ",
		   UT_NAMESIZE, UT_NAMESIZE, u.ut_name,
		   UT_LINESIZE, UT_LINESIZE, u.ut_line,
		   UT_HOSTSIZE, UT_HOSTSIZE, *(u.ut_host) ? u.ut_host : "-");
      delta = now - u.ut_time;
      pr_attime(&delta, &(u.ut_time), &now); /* */
      if (delta < SECSPERDAY)
	printf("  (%5.5s)",
	       asctime(gmtime(&delta))+11);
      else
	printf(" (%ld+%5.5s)",
	       delta / SECSPERDAY,
	       asctime(gmtime(&delta))+11);
      printf("\n");
      ++nusers;
      continue;
    }

    (void)fclose(ut);
    (void)printf("\nTotal users: %d (%d)\n", nusers, tusers);

  }	/* end: for */

}
/*
 * pr_attime --
 *	Print the time that the user logged in
 */

pr_attime(delta, started, now)
  time_t *delta, *started, *now;
{
  struct tm *tp;

  tp = localtime(started);

  if (*delta > (SECSPERDAY * DAYSPERWEEK))	/* more than a week */
    printf("%2.2d-%2.2d-%2.2d",
	   tp->tm_mday, tp->tm_mon + 1, tp->tm_year);

  else if (*now / SECSPERDAY != *started / SECSPERDAY) {	/* not today */
    printf("%s-%2.2d:%2.2d",
	   weekdays[tp->tm_wday], tp->tm_hour, tp->tm_min);
  }

  else {	/* today */
    printf("%2.2d:%2.2d", tp->tm_hour, tp->tm_min);
  }

}	/* end pr_attime() */

