/*
 * $Header: /home/vikas/src/xtacacsd/RCS/perm.c,v 1.9 1998/03/08 21:35:48 vikas Exp $
 */
/*
 * Password & authorization interface routines for xtacacs.
 *
 * Note that unlike other password routines, these functions do NOT
 * overwrite the cleartext password as soon as we are done using them.
 * This is because the clear text password is used repeatedly, etc. etc.
 *
 * authent_pw:
 *	Given the tacacs username and password, calls the routines for
 *	checking against the password files and PH routines. Returns A_FAIL
 *	A_PASS, A_NOUSER, A_ERROR.
 *	After calling this routine, the password structure MUST be filled
 *	or be NULL.
 *
 * authent_files:
 *	Called by authent_pw()
 *	Extracts the user info from the various password files and
 * 	compares the password with the one supplied. Searches through all
 * 	the listed password files for a match.
 *
 * authent_qi:
 *	Called by authent_pw for checking against the CSI QI/PH database
 *
 * search_pwname:
 *	Given a name, searches through all the listed password files for a
 * 	matching username.
 *
 * check_perm:
 *	For the user/group/geco combination given, it checks the permissions
 * 	of the user against the configured lines in the 'permlist' structure.
 *	It calls the check_key() program which modifies the tacacs response
 *	fields if necessary (such as access list, denies, number of logins,
 *	etc.)
 *
 * read_config:
 *	Read the configuration file.
 *
 *
 *  COPYRIGHT (c) by Vikas Aggarwal, vikas@navya.com
 *
 * $Log: perm.c,v $
 * Revision 1.9  1998/03/08 21:35:48  vikas
 * - added debug messages during comparison of mask
 * - checks alternate password files only if supplied
 *
 * Revision 1.8  1998/01/05 04:41:36  vikas
 * Changed #ifdef DCE to OSFDCE since the new DEC Unix seems to define
 * DCE somewhere in the include files.
 *
 * Revision 1.7  1997/05/31 11:03:43  vikas
 * Fixed small bug in report() (missing argument)
 *
 * Revision 1.6  1997/05/29 11:51:54  vikas
 * Just changed the indentation (good old xemacs).
 *
 * Revision 1.5  1997/05/28 20:16:59  vikas
 * Added support for OSF DCE authentication (pbhenson@csupomona.edu).
 * Added new config file keyword QUIETNOUSER which sets the -Quiet flag.
 *
 * Revision 1.4  1996/05/19  17:32:35  vikas
 * Changed authent() routines for use with result_codes (A_ERROR, A_xx).
 * Separated system authent, file authent into much cleaner independent
 * routines.
 * Added support for DEC SIA, QI. Now runs external getok program
 * with more arguments on the command line.
 *
 * Revision 1.3.1.1  1996/04/21  08:20:12  vikas
 * Interim version before fixing support for DEC SIA routines.
 *
 * Revision 1.3  1996/01/17  13:54:51  vikas
 * Added enable levels for Cisco IOS 10.3
 * Renamed keys to uinfo for readability. Moved Getpwnam() to Getpw.c
 * Easier parsing of boolean options in read_config()
 * Accepts range of lines in config file.
 *
 * Revision 1.2  1995/11/10  22:11:44  vikas
 * Created a structure for all boolean options; cleaner code.
 *
 * Revision 1.1  1995/07/14  16:19:37  vikas
 * Initial revision
 *
 *
 */


#ifndef lint
 static char rcsid[] = "$RCSfile: perm.c,v $ $Revision: 1.9 $ $Date: 1998/03/08 21:35:48 $" ;
 static char sccsid[] = "$RCSfile: perm.c,v $ $Revision: 1.9 $ $Date: 1998/03/08 21:35:48 $" ;
#endif


#include <sys/types.h>
#include <sys/wait.h>		/* for Wxxx macros used after popen() */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <syslog.h>
#include <unistd.h>
#include <fcntl.h>
#include <netinet/in.h>		/* htonl() definitions */
#ifdef SHADOW_PW
# include <shadow.h>		/* shadow password files on SYSV */
#endif
#ifdef SIA			/* DEC's Security Integration Arch. SIA */
# include <sia.h>		/* Enhanced security */
# include <siad.h>
#endif
#ifdef OSFDCE			/* OSF's DCE authentication standard */
# include <dce/sec_login.h>
#endif

#include "tacacs.h"
#include "common.h"

#ifdef QI
extern char *QI_host1;		/* defined in ph.c */
extern char *QI_host2;
extern char *QI_timeoutstr;
extern char *QI_type;
extern char *QI_uid;
extern char *QI_gid;
extern char *QI_shell;
extern char *QI_gecos;
#endif

static int curpwindex;			/* current password file */
static int authtokenct;			/* count of authtoken keywords */

static char *monthname[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
		     "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
static long days_ere_month[] = {0, 31, 59, 90, 120, 151,
			 181, 212, 243, 273, 304, 334};

/*
 * Following structure allows special keywords in the password field
 * of a user. For that user, the corresponding program is executed to
 * get an okay. Used for the Enigma Logic, SDI support.
 */
#define MAX_AUTHTOKENS  32
struct _authtoken {
    char *keyword;		/* in password field */
    char *exec;			/* program to exec for password auth */
} authtoken[MAX_AUTHTOKENS];

/*
 * Following is a list of 'actions' and 'permissions' for the 
 * configuration file. Note that the first match which explicity permits
 * or denies access will return.
 *
 * Thus, an explicit PERMIT or DENY or EXEC{PERMIT/DENY} will return
 * immediately on matching. NOROUTING, NUMLOGIN will deny if the request
 * violates (else leave the request untouched), and ACL inserts the
 * access-list number (but does not affect permit/deny).
 */

/* The various ACTIONS that can be performed for a request */
enum {ACL, NOROUTING, NUMLOGIN, GETOK, PERMIT, DENY, EXECPERMIT, EXECDENY};
struct _acts {
    int  enumi;			/* list of actions */
    char *str;			/* keywords in the config file */
};

struct _acts  acts[] = {
  { ACL, "acl" },		/* send access-list in reply */
  { NOROUTING, "norouting"},	/* deny slip routing requests */
  { NUMLOGIN, "numlogin" },	/* restrict number of logins via utmp */
  { GETOK, "getok" },		/* getOK from external program */
  { PERMIT, "permit" },		/* permit the request */
  { DENY, "deny" },		/* deny the request */
  { EXECPERMIT, "exec" },	/* execute the program and permit */
  { EXECPERMIT, "execpermit" },	/* .. alternate name */
  { EXECDENY, "execdeny" },	/* exec and deny access */
  { EXECDENY, "denyexec" },	/* .. just alternate name */
  { 0, "" }
};

/* The various REQUEST types from the terminal server */
struct _req {
    int type;		/* request types for which permission checked */
    char *str;		/* keywords in the config file */
};

struct _req req[]= {
  {XTA_LOGIN,	 "login"},
  {XTA_CONNECT,  "connect"},	/* permits all connect requests */
  {XTA_ENABLE,   "enable"},
  {XTA_SLIPADDR, "slipaddr"},
  {XTA_SLIPON,   "slipon"},
  {XTA_SLIPON,   "slip"},
  {XTA_ARAP_AUTH, "arap"},
  {XTA_CHAP_AUTH, "chap"},
  {ALL_REQUESTS, "any"},
  {ALL_REQUESTS, "all"},
  {OLD_REQUEST,  "old"},
  {0 , ""}
};

/* A linked list of line numbers to be stored in _perm. */
struct _lineno
{
   int line;
   struct _lineno *next;
};


/*
 * The permission's struct. to create list of requests & actions.
 * The information in the config file is stored in this structure.
 * If the key + host + request_type match, then 'action' is performed.
 * Obviously need a sensible combination of request+action (i.e.
 * request_type = LOGIN  && act = NOROUTING  make no sense (unless the
 * tserver starts keeping track of the xta_f_<flags>)
 */
struct _perm
{
  int		keytype;	/* Type of entry- USER_K/GROUP_K/GECOS_K */
  char	*strkey;	/* key value (username/gid/gecos) */
  char	*host;		/* to check hostname for incoming request */
  u_long	hostip;		/* IP address as a 'long' */
  u_long	mask;		/* IP mask 0.0.255.255 for IP addr */
  struct _lineno *line;       /* to check linenumber for incoming request */
  int		request_type;	/* from the req[] array: login/connect/etc. */
  char	*strreq_type;	/* req_type in string format, for logging */
  int		action;		/* from the act[] array: deny/acl/etc. */
  ushort	enlevel;       	/* enable level, in IOS 10.3 and higher */
  char	*straction;	/* action in string form, easier in logging */
  char	*args;		/* if any arguments for the 'action' */
};

#define MAXCFLINES 256			/* max number of config lines */
struct _perm permlist[MAXCFLINES];


/*
 * The user information that has to be matched against the configuration
 * lines can be of 3 types:
 *	USER for username,  GROUP for GID,  GECOS for gecos field
 * We store this info in a simple structure to make matching easy.
 */

struct _uinfo
{
  int		type;	/* from enumerated list */
  char	*str;	/* user,group,geco string to match in config file */
  char	*val;	/* value (username or group or geco) */
};

/*
 * Order is important here, since uinfo[USER_K] must be a 'user' entry,
 * hence this bit of extraneous work in assigning values...
 */
struct _uinfo uinfo[] = {
#define USER_K	0
  { USER_K, "user", NULL },
#define GROUP_K	1
  { GROUP_K, "group", NULL },
#define GECOS_K	2
  { GECOS_K, "geco", NULL },
#define EO_K	3
  { EO_K, "", NULL }
};


/*+ authent_pw:
 *	Calls the various verification routines that have to return a filled
 * passwd structure. Returns A_ERROR, A_NOUSER, A_FAIL, A_PASS
 */
authent_pw(name, passwd, ppw, checkpw)
  char *name, *passwd;
  struct passwd **ppw;
  int checkpw;		/* check passwd or just return filled ppw */
{
  int status = A_NOUSER;	/* assume no_such_user at start */
  int tstatus;

  /* No null usernames */
     
  if (!name || *name == '\0' || 
      (checkpw && (passwd == NULL || *passwd == '\0') ) )
  {
    *ppw = NULL;
    return (A_FAIL);
  }

  if (do_system_pw)	/* try system calls and passwd files */
    if ((status = authent_system(name, passwd, ppw, checkpw)) == A_PASS)
      return (A_PASS);


  if (pwfile[0] != NULL)    /* try alternate passwd files if given */
  {
    if ( (tstatus = authent_files(name, passwd, ppw, checkpw)) == A_PASS)
      return(A_PASS);
    if (tstatus == A_FAIL || tstatus == A_ERROR)
      status = tstatus ;	/* else leave it NOUSER or ERROR */
  }
      
#ifdef QI
  if ( (tstatus = authent_qi(name, passwd, ppw, checkpw)) == A_PASS)
    return(A_PASS);
  if (tstatus == A_FAIL || tstatus == A_ERROR)
    status = tstatus;
#endif

  return (status);
}

#ifdef SIA	/* DEC Enhanced Security */
audgenl()
{
  /* define this null routine on osf/1 3.2 to make linker happy */
}
#endif

/*
 * Verify the username and password in the default system files. Note that
 * this routine cannot do case insensitive matches and any special tweakings.
 * Only used if the do_system_pw has been set.
 */
authent_system(name, passwd, ppw, checkpw)
  char *name, *passwd;
  struct passwd **ppw;
  int checkpw;
{
  char tpasswd[PASSWD_LENGTH+1];/* compare only first n chars of passwd */
  char *epasswd;		/* encrypted password string for comparing */
  struct passwd *pw;
#ifdef SHADOW_PW
  struct spwd *spw;
#endif
#ifdef SIA
  SIAENTITY *who = NULL;
#endif

#ifdef DEBUG
  report (LOG_DEBUG, "authent_system: checking user=\"%s\"", name );
#endif /* DEBUG */

  setpwent();
  if ( (pw = getpwnam(name)) == NULL) 	/* case sensitive */
    if (ignorecase)	    /* If ignorecase, lowercase and try again */
    {
      char lname[128];
      register char *lp = lname, *np = name;

      while (*lp = tolower(*np++))        /* convert query to lower */
	lp++;
      *lp = '\0';			/* just being extra careful */
      setpwent();
      pw = getpwnam(lname);		/* try lowercase username */
    }

  *ppw = pw;
    
  if (pw == NULL) {
    report(LOG_INFO, "getpwnam() returned no such user %s", name);
    return(A_NOUSER);
  }


  if (pw->pw_uid == 0)
    return(A_FAIL);			/* no access to uid root */
      
  if (!checkpw) {
#ifdef DEBUG
    report(LOG_DEBUG, "authent_system: not checking passwd since checkpw=0");
#endif
    return(A_PASS);	/* all okay */
  }

  if (!(pw->pw_passwd) || *pw->pw_passwd == '\0') /* blank passwd in file */
  {
    if (blankpassword) {
#ifdef DEBUG
      report(LOG_DEBUG, "authent_system: permitting blank password");
#endif
      return(A_PASS);	/* all okay */
    }
    else {
#ifdef DEBUG
      report(LOG_DEBUG, "authent_system: denying blank password");
#endif
      return(A_FAIL);
    }
  }	/* end if(blank password) */
	  

  if (authtokenct)	    /* Now check if the password is a special token */
  {
    register tk = 0;

    for ( ; tk < authtokenct && authtoken[tk].keyword != NULL; ++tk)
      if (strcmp(pw->pw_passwd,authtoken[tk].keyword) == 0)
	return (authtoken_stub(authtoken[tk].exec, pw->pw_name, passwd));
  }

  /*
     * So far we have just extracted the user entry from the password
     * file and checked if the password is a special token or blank, etc.
     * Now we try and match the encrypted passwords (might need to extract
     * the password from the shadow password file).
     */
  strncpy(tpasswd, passwd, PASSWD_LENGTH);	/* the cleartext password */
  epasswd = (char *)crypt(tpasswd, pw->pw_passwd);
  if (strlen(epasswd) == 0) {
    report(LOG_WARNING,"authent_system: Null encrypted passwd, shouldnt happen");
    return(A_ERROR);
  }

  if (strcmp(epasswd, pw->pw_passwd) == 0)   /* match found */
  {
#ifdef DEBUG
    if (debug)
      report (LOG_DEBUG, "authent_system: Password matched (%s)", epasswd);
#endif
    return(A_PASS);					/* Okay */
  }
  else if (debug)
    report(LOG_DEBUG,
	   "authent_system: Password match failed (real %s, query %s)",
	   pw->pw_passwd, epasswd);

  /*
     * Okay, check in system special routines like shadow or DEC's SIA
     */

#ifdef SHADOW_PW
# ifdef DEBUG
  report (LOG_DEBUG,"search_pwname: looking in shadow pwd file");
# endif
  /* The user does exist since we have reached here. Extract any
     * shadow password entry... use the already extracted 'pw->pw_name'
     * instead of 'name' since it might be mixed case.
     */
  setspent();
  spw=(struct spwd *)getspnam(pw->pw_name);	/* Dont use 'name' */
  if (spw != NULL)
  {
    pw->pw_passwd = spw->sp_pwdp;
    /* spw->sp_expire contains the expiration date in days (on Linux
     * it is the number of days since Jan 1 1970)
     */
    expiredays = spw->sp_expire;
#ifdef DEBUG
    report(LOG_DEBUG,"authent_system (shadow): expiredays = %ld",expiredays);
#endif
  }
  else
    pw->pw_passwd = "NotInFil" ;	/* no password */

  strncpy(tpasswd, passwd, PASSWD_LENGTH);
  epasswd = (char *)crypt(tpasswd, pw->pw_passwd);
  if (strlen(epasswd) == 0) {
    report(LOG_WARNING,"authent_system (shadow): Null encrypted passwd, shouldnt happen");
    return(A_ERROR);
  }

  if (strcmp(epasswd, pw->pw_passwd) == 0)   /* match found */
  {
#ifdef DEBUG
    if (debug)
      report (LOG_DEBUG, "authent_system: Password matched (%s)", epasswd);
#endif
    endspent();	/* do this in the end in case it wipes out spw */
    return(A_PASS);					/* Okay */
  }
  else if (debug)
    report(LOG_DEBUG,
	   "authent_system: Password match failed (real %s, query %s)",
	   pw->pw_passwd, epasswd);
  endspent();		/* do this in the end in case it wipes out spw */

#endif /* SHADOW_PW */


#ifdef SIA	/* DEC OSF1 Enhanced security, no crypt() required */
  if (pw)
  {
    char *info[2];
    info[0] = "xtacacsd";
    info[1] = NULL;

# ifdef DEBUG
    report (LOG_DEBUG,"authent_system: calling SIA auth routines");
# endif

    if (sia_ses_init(&who, (1), info, /*host*/NULL, /*user*/name,
		     /*tty*/NULL, /*can collect*/0,
		     /*unused*/NULL) != SIASUCCESS)
    {
      report(LOG_ALERT, "ERROR: sia_ses_init() failed %s",
	     sys_errlist[errno]);
      return(A_ERROR);
    }
    /* succeeded in init, now check password */
    if (sia_ses_authent(NULL, passwd, who) == SIASUCCESS) {
      report(LOG_INFO,
	     "authent_system: sia_ses_authent OK for user '%s'", name);
      sia_ses_release(&who);	/* frees up all resources */
      return(A_PASS);
    }
    sia_ses_release(&who);	/* frees up all resources */

  }	/* if (pw) */

#endif /* DEC SIA */

  /* DCE authentication code by Paul Henson <henson@acm.org> */
#ifdef OSFDCE
  if (pw)
  {
    sec_login_handle_t login_context;
    sec_login_auth_src_t auth_src;
    sec_passwd_rec_t pw_entry;
    boolean32 reset_passwd;
    sec_passwd_str_t tmp_pw;
    error_status_t dce_st;
      
# ifdef DEBUG
    report (LOG_DEBUG,"authent_system: calling DCE auth routines");
# endif
    if (sec_login_setup_identity(name, sec_login_no_flags,
				 &login_context, &dce_st))
    {
      pw_entry.version_number = sec_passwd_c_version_none;
      pw_entry.pepper = NULL;
      pw_entry.key.key_type = sec_passwd_plain;
      strncpy((char *)tmp_pw, passwd, sec_passwd_str_max_len);
      tmp_pw[sec_passwd_str_max_len] = '\0';
      pw_entry.key.tagged_union.plain = &(tmp_pw[0]);

      if (sec_login_valid_and_cert_ident(login_context, &pw_entry,
					 &reset_passwd, &auth_src, &dce_st))
      {
	report(LOG_INFO,
	       "authent_system: dce cert_ident OK for user '%s'", name);
	sec_login_purge_context(&login_context, &dce_st);
	return(A_PASS);	/* login succeeded */
      }
      else
      {
	/* cert_ident failed */
	if (debug)
	  report(LOG_DEBUG,
		 "authent_system: dce cert_ident FAILED for '%s'", name);
	sec_login_purge_context(&login_context, &dce_st);
      }
    }	/* if sec_login_setup_identity() */
    else
    {
      /* login_setup failed */
    }
  }	/* if (pw) */
#endif	/* OSFDCE */

  /*
     * At this point, all methods have been exhausted. Note that if there
     * was no such user, this routine would have already returned -1 above.
     */
  report(LOG_INFO,
	 "authent_system: system authentication failed for '%s'", name);
  return(A_FAIL);
} 		/* end: authent_system */



/*+ authent_files:
 *
 * 	Verify the provided name/password in all the supplied password
 * filenames. If no alternate file is listed, the do_system_pw will
 * have been set in main().
 *	- get pw struct for 'name'
 *	- check if password is a special AUTHTOKEN
 *	- check if encrypted password matches
 *
 * The supplied filenames are in 'pwfile[]'.
 * It calls an external program for authentication if the password field
 * has a special token (specified in the config file using AUTHTOKEN,
 * stored in the authtoken structure).
 *
 * To prevent password anomalies, does not accept if the password field 
 * is blank in the passwd file (like gopher or sync). Also, do not accept
 * if the uid is '0'.
 *
 * Return filled 'pw' without checking the password if checkpw == 0
 * (this can be used by ARAP and CHAP authentication where they just
 * want the structure filled in and dont use password authentication.
 *
 *  Return's A_PASS, A_FAIL, A_NOUSER or A_ERROR
 */
authent_files(name, passwd, ppw, checkpw)
  char *name, *passwd;
  struct passwd **ppw;
  int checkpw;
{
  char tpasswd[PASSWD_LENGTH+1];	/* compare only first n chars of passwd */
  char *epasswd;			/* encrypted password string for comparing */
  struct passwd *pw;

#ifdef DEBUG
  report (LOG_DEBUG, "authent_files: checking user=\"%s\"", name );
#endif /* DEBUG */

  curpwindex = 0;	/* reset ptr to alternate files */
  *ppw = NULL;	/* initial value */

  while ((pw = search_pwname(name)) != NULL)
  {				/* until searched thru all password files */
    *ppw = pw;

    if (pw->pw_uid == 0)
      continue;		/* no access or checks for this uid */

    if (!checkpw) {
#ifdef DEBUG
      report(LOG_DEBUG, "authent_files: not checking passwd since checkpw=0");
#endif
      return(A_PASS);	/* all okay */
    }

    if (*pw->pw_passwd == '\0')	/* blank passwd in file */
    {
      if (blankpassword) {
#ifdef DEBUG
	report(LOG_DEBUG, "authent_files: permitting blank password");
#endif
	return(A_PASS);/* all okay */
      }
      else {
#ifdef DEBUG
	report(LOG_DEBUG, "authent_files: ignoring blank password");
#endif
	continue;
      }
    }	/* end if(blank password) */
	
	
    if (authtokenct)    /* Now check if the password is a special token */
    {
      register tk = 0;

      for ( ; tk < authtokenct && authtoken[tk].keyword != NULL; ++tk)
	if (strcmp(pw->pw_passwd,authtoken[tk].keyword) == 0)
	  return (authtoken_stub(authtoken[tk].exec, pw->pw_name, passwd));
    }
	
    /*
     * Okay, finally try encryption, etc.
     */
    strncpy(tpasswd, passwd, PASSWD_LENGTH);
    epasswd = (char *)crypt(tpasswd, pw->pw_passwd);
    if (strlen(epasswd) == 0) {
      report(LOG_WARNING,"authent_files: Null encrypted passwd, shouldnt happen");
      continue;
    }
	
    if (strcmp(epasswd, pw->pw_passwd) == 0)   /* match found */
    {
#ifdef DEBUG
      if (debug)
	report (LOG_DEBUG, "authent_files: Password matched (%s)", epasswd);
#endif
      return(A_PASS);/* Okay */
    }
    else if (debug)
      report(LOG_DEBUG,
	     "authent_files: Password match failed (real %s, query %s)",
	     pw->pw_passwd, epasswd);
	
    /* Get next password entry for the same username */
	
  }		/* end: while(search_pwname) */

  if (*ppw == NULL)	/* no user was found at all */
  {
    report (LOG_INFO, "authent_files: no such user '%s'", name);
    return(A_NOUSER);
  }

  report(LOG_INFO,"authent_files: authentication failed for '%s'", name);
  return(A_FAIL);				/* passwords failed */
} 		/* end: authent_files */


/*+ 
 * FUNCTION:
 * 	Search for a password entry for the name given from the entire
 * list of password files. Returns null pointer at end of all files.
 */
struct passwd * 
search_pwname(name)
  char *name;
{
  struct passwd *pw;

  if (!name || *name == '\0')
    return (NULL);

  /*
   * Now search in the password files using the alternate password
   * routines.
   */
  while (pwfile[curpwindex] != NULL)
  {
    reset_pwfile(pwfile[curpwindex]);	/* reset current pwd file */

    pw = Getpwnam(name, ignorecase);
    ++curpwindex;			/* search next file in the next pass */
    if (pw != NULL)
    {
#ifdef DEBUG
      report (LOG_DEBUG, "search_pwname: Match for %s found in %s",
	      pw->pw_name, pwfile[curpwindex - 1]);
#endif
      if (pw && pw->pw_passwd == NULL)
	report(LOG_WARNING,
	       "search_pwname: Warning, null password pointer for %s",
	       name);
      return (pw);
    }
  }	/* end: while(pwfile[] != NULL) */

  return (NULL);	/* nothing found */
}	/* end: search_pwname */
    



/*+ authtoken_stub:
 * 	Routine to call an external program to verify passwords.
 * The routine checks for the existance of the program, and if any problems,
 * will return 0 with the appropiate error message...
 *
 * Since the return value from pclose() has many high order bits set to
 * reflect the exit status of the child process, we *must* use the WIF
 * macros to get the real exit values (see 'man 2 wait')
 *
 */

authtoken_stub(auth_prog, name, password)
  char	*auth_prog, *name, *password;
{
  FILE 	*p;
  char	buf[256];
  int	ret = 0;		/* Not OK */
    
  /* Check to see if the stub program is there... */
  if (access(auth_prog, R_OK | X_OK)) {
    report(LOG_ERR, "authtoken_stub: invalid auth program: %s: %s", 
	   auth_prog, "read/execute access denied");
    return (A_ERROR);
  }
  sprintf(buf, "%s '%s' '%s'\0", auth_prog, name, password);
#ifdef DEBUG
  report(LOG_INFO, "authtoken_stub: execing: %s", buf);
#endif
  if ((p = (FILE *)popen(buf, "r")) == NULL) {
    report(LOG_ERR, "authtoken_stub: cannot exec auth program: %s: %m",
	   auth_prog, "access denied?");
    return (A_ERROR);
  }
  ret = pclose(p);

  /* Handle most cases for the child process going wrong - TpB 10/22/96
   * Check for signals, special child exits, etc.
   */
  if (WIFEXITED(ret))	/* exited normally by calling an 'exit' */
  {
#ifdef DEBUG
    report(LOG_INFO, "authtoken_stub: return status: %d", WEXITSTATUS(ret));
#endif
    if (WEXITSTATUS(ret) == 1)
      return(A_PASS);
    else
      return(A_FAIL);     /* if exit() isn't 1, then assume a failure */
  }	/* if WIFEXITED() */

  /* Here if process did not exit normally by calling an 'exit'. Give info */
  report(LOG_ERR, "authtoken_stub: child returned but didn't exit.");
  if (WIFSIGNALED(ret))
  {
    report(LOG_ERR, "authtoken_stub: child signalled %d",WTERMSIG(ret));
#ifdef WCOREDUMP
    if (WCOREDUMP(ret))
      report(LOG_ERR, "authtoken_stub: child dumped core");
#endif
  }
  if (WIFSTOPPED(ret))	/* not terminated, but has stopped */
    report(LOG_ERR, "authtoken_stub: child stopped %d",WSTOPSIG(ret));

  return(A_FAIL);
}	/* end: authtoken_stub() */




/*+
 * Convert the date string into days since 1970.
 * The date shold be in the form:  Dec 22 1995
 * Does a simple check to see if the first character is a '/' to indicate
 * if this is a shell or a date.
 *
 */
long strtodate(datestr)
  char *datestr;
{
  long age;
  long day, month, year, leaps;
  char monthstr[10];

  /* If no date or a shell, let it pass.  (Backward compatability.) */

  if (!datestr || *datestr == '\0' || *datestr == '/')
    return(0);
    
  /* Parse date string.  Fail it upon error.  */

  if (sscanf(datestr, "%s %ld %ld", monthstr, &day, &year) != 3)
  {
    report (LOG_WARNING, "strtodate: Unparseable date value %s",
	    datestr);
    return(2);
  }

  /* Compute the date in days.  */

  for (month = 0; month < 12; month++)
    if (strncmp(monthstr, monthname[month], 3) == 0)
      break;
  if (month > 11)
    return(2);
  leaps = (year-1969)/4 + (((year % 4) == 0) && (month >= 2));
  age = (((year-1970)*365) + days_ere_month[month] + (day-1) + leaps);

  if (age == 0)	/* in case someone entered Jan 1 1970, return a low... */
    return (2);	/* ... number. It will get rejected later anyway */
  else
    return (age);	/* days */

}	/* end: strtodate() */

/*+ check_expiration:
 * Given expiration date (in number of days since 1970), it checks if the
 * expiration is past the current date or within the warning period.
 *
 *	Returns 0 if OK, 1 if warning, 2 if expired
 *
 * If using shadow passwords, the age is in the format of number of days
 * since 1970 in the shadow password file. If using sysV, there is a
 * pw_age() field, else look for the age in the shell field. What a mess!!
 */

check_expiration(expiration)
  long expiration;		/* num days since 1970 */
{
  long now, warning;

  if (expiration == 0)
    return(0);		/* okay */

  if (expiration > 315360000)  /* 1980, hence probably in seconds */
    expiration = expiration / SEC_IN_DAY;

  warning = expiration - WARNING_PERIOD;	/* in days */

  /*
   * Get the current time (to the day)
   */
  now = querytime/SEC_IN_DAY;	/* convert to days from seconds */
    
  if (now > expiration)
    return(2);
  if (now > warning)
    return(1);
  return(0);

} 	/* end: check_expiration() */

/*
 * Compare the supplied hostname/hostip/mask with the client's hostname
 * or IP address (in the global variable 'from'). Check for special
 * configuration keywords 'all' or 'any'.
 */

ishostmatch(hoststr, hostip, hostmask)
  char *hoststr;
  unsigned long hostip, hostmask;
{
  if (strcasecmp(hoststr, "all") == 0 ||  strcasecmp(hoststr, "any") == 0)
    return(1);	/* matched */

  if (hostip == 0)
  {
    if (strcasecmp(from_hname, hoststr) == 0)
      return(1);	/* host matched */
    else
      return(0);
  }

  /*
   * the hostmask is in Cisco notation. i.e. 0.0.0.255 indicates ignore
   * the last 8 bits
   */
  report(LOG_DEBUG, "ishostmatch() checking  %lu == %lu & ~%lu",
	 hostip, (unsigned long)from.sin_addr.s_addr, hostmask);
  if (hostip == (from.sin_addr.s_addr & (~hostmask)) )
    return(1);	/* matched */
  else
    return(0);

}	/* end ishostmatch() */
	
/*+ check_perm:
 *	Extended checking only for extended xtacacs requests. Only checks the
 * line number for old type requests. Saves the response on entry,
 * and calls 'check_key' with each permission list. If check_key returns
 * a non-zero response, it return's immediately (does not check the other
 * lists).
 *
 */

check_perm(pw, tp)
  struct passwd *pw ;
xtacacstype *tp;
{
#ifdef HAVEGIDTYPE
#define gid_type  gid_t       	/* working on SYSV */
#else
#define gid_type  int		/* broken on BSD even though defined */
#endif
  register int i, j;
  char strgid[8];	/* to convert the numeric gid into string */
  char *gecostr;
  int gidarraylen = -1 ;	/* number of groups for the user */
  static gid_type ngroups_max ;
  static gid_type *gidarray;	/* suppl gid's set */

  if (tp->response == XTA_A_REJECTED)
    return (0);

  if (pw->pw_gid)
    sprintf (strgid, "%d\0", pw->pw_gid);
  else
    strgid[0] = '\0';

  gecostr = (char *)strrchr(pw->pw_gecos, ',');
  if (gecostr == NULL || *gecostr == '\0')
    gecostr = pw->pw_gecos;	/* if comma not found */
  else
    ++gecostr ;		/* skip the comma */

#ifdef DEBUG
  report (LOG_DEBUG,"check_perm: User- %s/GID- %s/Geco- %s, host- %s, req_type- %d",
	  pw->pw_name, strgid, gecostr, from_hname, tp->type);
#endif
  /*
   * The uinfo[] array is indexed by the type (user/group/geco). Hence,
   * we store the user information in the uinfo[] array for fast access
   * depending on the key type.
   */
  uinfo[USER_K].val  = pw->pw_name;
  uinfo[GROUP_K].val = strgid;
  uinfo[GECOS_K].val = gecostr;

  /* Check if pre-10.3 request or new one.
   * pre-10.3 Cisco IOS sends uninitialized tp->accesslist when
   * authenticating 'enable' command. This allows us to identify
   * pre-10.3 requests (simple check, possible failure small)
   * You should explicitly disable the 'enable' action on pre-10.3
   * hosts in the config file.
   */
  if (tp->type == XTA_ENABLE && tp->accesslist > MAX_ENABLE_LEVEL)
    tp->accesslist = MAX_ENABLE_LEVEL;  /* or should it be zero CSCdi28362 */

  /*
   * Now go through the permissions list (which is info from the config
   * file). If the -
   *		req_type, host, line_number, key
   * of the permission list matches that of the incoming request, then
   * the corresponding 'action' is performed.
   *
   * The user's information is in the uinfo struct, and the config lines
   * are in the permlist struct.
   */
  for (i=0; permlist[i].strkey && *(permlist[i].strkey) ; ++i)
  {
    int permlistgid;
    char *k = uinfo[permlist[i].keytype].val ;
    struct _lineno *lineno;

    if (k == NULL || *k == '\0')		/* nothing in pw field */
    {
#ifdef DEBUG
      report(LOG_DEBUG, "check_perm: null %s for user, skipping check",
	     uinfo[permlist[i].keytype].str);
#endif
      continue ;
    }

#ifdef DEBUG
    if (debug > 1)
      report(LOG_DEBUG,
	     "check_perm: checking with %s- %s host- %s req_type- %d",
	     uinfo[permlist[i].keytype].str,
	     permlist[i].strkey, permlist[i].host,
	     permlist[i].request_type);
#endif

    /*
     * For v10.3 ENABLE requests, check if the requested level matches
     * config level
     */
    if (tp->type == XTA_ENABLE &&
	permlist[i].request_type == XTA_ENABLE &&
	permlist[i].enlevel != ALL_ENABLE_LEVEL)
      if (permlist[i].enlevel != tp->accesslist)	/* should it be < */
	continue ;    /* skip */

    if (permlist[i].request_type != ALL_REQUESTS &&
	tp->type != permlist[i].request_type)
      continue ;	/* skip */

    if (!ishostmatch(permlist[i].host,permlist[i].hostip,permlist[i].mask))
      continue;	/* skip */

    if (strcasecmp(permlist[i].strkey, "all") == 0 ||
	strcasecmp(permlist[i].strkey, "any") == 0 ||
	strcasecmp(k, permlist[i].strkey) == 0)
      goto donekeys;	/* 'goto' seems the easiest solution */

    if (permlist[i].keytype != GROUP_K)
      continue;
    else	/* Check if the user belongs to this supplementary group */
    {
#ifdef DEBUG
      if (debug > 1)
	report(LOG_DEBUG, "check_perm: checking if user belongs to group %s",
	       permlist[i].strkey);
#endif
      if (!sscanf(permlist[i].strkey, "%d", &permlistgid))
      {
	/* Gid key isn't an integer, so can't match it */
	report(LOG_NOTICE, "check_perm: group %s not a number",
	       permlist[i].strkey);
	continue;
      }

      if (!gidarray)		/* malloc memory for array */
      {
	ngroups_max = sysconf(_SC_NGROUPS_MAX);
	gidarray = (gid_type *)malloc(ngroups_max * sizeof(gid_type));
	if (!gidarray) {
	  report(LOG_ERR,"check_perm malloc: %s, exiting",
		 sys_errlist[errno]);
	  exit (1);
	}
      }/* end if !gidarray */
	    
      if (gidarraylen < 0)    /* Init gidarray for this user */
      {
#ifndef NOINITGROUPS
	if (initgroups(pw->pw_name, pw->pw_gid) >= 0)
#endif
	  gidarraylen = getgroups(ngroups_max, gidarray);
	      
	if (gidarraylen < 0)  {
	  gidarraylen = 1;
	  gidarray[0] = pw->pw_gid;
	}
#ifdef DEBUG
	if (debug > 2) {
	  report(LOG_DEBUG, "check_perm: User belongs to %d groups:",
		 gidarraylen);
	  for (j=0; j < gidarraylen; j++)
	    report(LOG_DEBUG, " \t%d", gidarray[j]);
	}
#endif /* DEBUG */

      }	/* end: if (gidarraylen not inited) */

      for(j=0; j < gidarraylen && gidarray[j] != permlistgid; j++)
	;	/* do nothing */

      if (j == gidarraylen)
	/* No match in gid set */
	continue;

    }	/* end if checking supplementary groups */

  donekeys:
    /*
     * check to see if permlist[i].line is NULL or tp->lport
     * matches a line in the line list.
     */
    if (tp->type != OLD_REQUEST && permlist[i].line != NULL) {
      lineno = permlist[i].line ;
      do {
	if (lineno->line == tp->lport)
	  break;
	lineno = lineno->next;
      } while (lineno != NULL);
      if (lineno == NULL)	/* no match in entire list */
	continue;
    }

    /* if we get here, we're ok'ed on everything so far */

 
    if (tp->type == OLD_REQUEST) {
      /* Only action PERMIT is valid for an old-style query. */
      switch (permlist[i].action) {
      case PERMIT:
	return (1);
      case DENY:
	return (0);
      case ACL:
      case NOROUTING:
      case GETOK:
      case EXECPERMIT:
      case EXECDENY:
	report(LOG_INFO,
	       "old_process: warning: option ignored for old process.");
	continue;
      }

    }	/* if OLD_REQUEST */
    /* From here, only new process style query are handled */

    if (k == NULL || *k == '\0')
    {
      report(LOG_WARNING, "check_perm: Key was null ?");
      continue;			/* should not happen */
    }
    if (check_key(&permlist[i], pw, tp) == 1)
      return (1);

  }	/* end:  for() */

  return (0);			/* response not modified, flags might be! */

}	/* end: check_perm() */


/*+ check_key:
 * Return non-zero indicates that it changed the tp->response and the
 * calling routine should not call check_key again.
 */
check_key(perm, pw, tp)
  struct _perm *perm;
  struct passwd *pw;
  xtacacstype *tp;
{
  int current_logins;
  char runstr[128];		/* tmp string for doing a 'system' */

  if (pw == NULL)
    return (0);

#ifdef DEBUG
  if (debug)
    report(LOG_DEBUG,
	   "check_key: User '%s', matched permission line '%s %s' req '%s', doing action %s",
	   pw->pw_name, uinfo[perm->keytype].str, perm->strkey, 
	   perm->strreq_type, perm->straction);
#endif
  switch (perm->action)
  {
  case ACL:				/* ***************** */
    if (perm->args == NULL)	
      break ;
    if (tp->type == XTA_ENABLE)
      break;   /* save privelege level in tp->accesslist in new queries */
    /*
     * arg is access list. For new style SLIPON queries, send back the ACL
     */
    if (newstyle_query && tp->type == XTA_SLIPON)
    {
      struct xta_slip_acl acl;
      char s[8], *t = perm->args;
      int i = 0;

      bzero(&acl, sizeof acl);
      while ( isdigit(*t) )
	s[i++] = *t++;
      s[i] = '\0' ;
      if (i != 0)
	acl.in = atol(s);

      while (*t != '\0' && (!isdigit(*t)) )
	++t;
      i = 0;
      while ( isdigit(*t) )
	s[i++] = *t++ ;
      s[i] = '\0' ;
      if (i != 0)
	acl.out = atol(s);
      else
	acl.out = 0;
#ifdef DEBUG
# ifdef __alpha
      report (LOG_DEBUG,
	      "check_key: access_list set to %d/%d (%s in config file)\n",
	      acl.in, acl.out, perm->args);
# else
      report (LOG_DEBUG,
	      "check_key: access_list set to %ld/%ld (%s in config file)\n",
	      acl.in, acl.out, perm->args);
# endif	/* __alpha */
#endif	/* DEBUG */
      acl.in  = htonl(acl.in);
      acl.out = htonl(acl.out);
      tp->namelen = 0; tp->pwlen = sizeof(acl);
      bcopy((char *)&acl, ((char *)tp) + XTACACSSIZE, tp->pwlen);
    }
    else		/* old style query */
    {
      tp->accesslist = atoi (perm->args);
#ifdef DEBUG
      report (LOG_DEBUG,"check_key: access_list set to %d (%s in config file)",
	      tp->accesslist, perm->args);
#endif
    }
    break;

  case NOROUTING:			/* ***************** */
    if ((tp->type == XTA_SLIPON || tp->type == XTA_SLIPADDR)
	&& (tp->flags & XTA_F_ROUTING))
    {
      report(LOG_INFO, "denying routing request for %s",
	     pw->pw_name);
      tp->response = XTA_A_REJECTED;
      tp->reason   = XTA_A_NOROUTE;
      return (1);/* no more checks */
    }
    else if (tp->type != XTA_SLIPADDR)
      report (LOG_WARNING, 
	      "check_key: Bad config for %s- routing and no slip", pw->pw_name);
    break;

  case NUMLOGIN:			/* # of logins restricted */
    if (tp->type != XTA_LOGIN) {
#ifdef DEBUG
      report (LOG_DEBUG, "check_key: skipping numlogin check on non-login request");
#endif
      break;
    }
    if (utmpfd < 0)
    {
      report(LOG_ERR, "check_key: No utmp file descriptor- %s, denying request",
	     utmp_file);
      tp->response = XTA_A_REJECTED;
      tp->reason   = XTA_A_DENIED;
      return (1);/* deny */
    }
    current_logins = cur_login_count(pw->pw_name, from_hname, tp->lport);
	    
#ifdef DEBUG
    report(LOG_DEBUG, "check_key: %s- cur logins= %d, allowed= %s",
	   pw->pw_name, current_logins, perm->args );
#endif
    if ( current_logins >= atoi(perm->args) )
    {
      report(LOG_INFO, "check_key: denying request from %s, curlogins = %d",
	     from_hname, current_logins);
      tp->response = XTA_A_REJECTED;
      tp->reason   = XTA_A_DENIED;/* login exceed */
      return (1);
    }

    break;

  case GETOK:			/* ***************** */
    if (access(perm->args, R_OK | X_OK))
    {
      report(LOG_ERR, "check_key: invalid GETOK program: %s: %s", 
	     perm->args, "access denied");
      break;			/* assume OK */
    }
    else
    {
      char buf[256];
      FILE *p ;
      int ret = 0;

      /* Run the program with username, gid, hostname, etc as args */
      sprintf(buf, "%s '%s' %d '%s' %u %u %lu '%s'\0",
	      perm->args, pw->pw_name, pw->pw_gid, from_hname, tp->lport,
	      tp->type, tp->flags, pw->pw_gecos);
#ifdef DEBUG
      report(LOG_INFO, "check_key: execing GETOK program: %s", buf);
#endif
      if ((p = (FILE *)popen(buf, "r")) == NULL)
      {
	report(LOG_ERR, "check_key: cannot popen GETOK prog: %s",
	       perm->args);
	break ;
      }
      ret = pclose(p);
      if (! WIFEXITED(ret)) ret = 0;	/* not checking for signals */
      else  ret = WEXITSTATUS(ret);

#ifdef DEBUG
      report(LOG_INFO, "check_key: GETOK returned: %d", ret);
#endif
      if (ret == 0)		/* denied */
      {
	report(LOG_INFO,
	       "check_key: GETOK denied request for %s from %s",
	       pw->pw_name, from_hname);
	tp->response = XTA_A_REJECTED;
	tp->reason   = XTA_A_DENIED;
	return (1);/* no more checks */
      }
      else
      {
	tp->response = XTA_A_ACCEPTED;
	if (tp->reason != XTA_A_EXPIRING) tp->reason = XTA_A_NONE;
	return (1);/* no more checks */
      }
    }
    break;

  case DENY:				/* ***************** */
    report(LOG_INFO, "check_key: Denying request from %s", from_hname);
    tp->response = XTA_A_REJECTED;
    tp->reason   = XTA_A_DENIED;
    return (1);/* no more checks */

  case PERMIT:			/* ***************** */
    report(LOG_INFO,"check_key: Permitting request from %s", from_hname);
    tp->response = XTA_A_ACCEPTED;
    if (tp->reason != XTA_A_EXPIRING) tp->reason = XTA_A_NONE;
    return (1);/* no more checks */

  case EXECDENY:	/* tack on the username, host, line and an & */
    sprintf (runstr, "(%s %s %s %d)&\0", 
	     perm->args, pw->pw_name, from_hname, tp->lport);
    report (LOG_DEBUG,"ExecDeny: running system '%s' and denying request",
	    runstr);
    system(runstr);/* ouch */
    tp->response = XTA_A_REJECTED;/* note REJECT response */
    tp->reason   = XTA_A_NONE;/* no reason */
    return(1);

  case EXECPERMIT:			/* ***************** */
    sprintf (runstr, "(%s %s %s %d)&\0", 
	     perm->args, pw->pw_name, from_hname, tp->lport);
    report (LOG_DEBUG,"ExecPermit: running system '%s'", runstr);
    system(runstr);/* ouch */
    tp->response = XTA_A_ACCEPTED;
    if (tp->reason != XTA_A_EXPIRING) tp->reason = XTA_A_NONE;
    return(1);/* no more checks */
	    
  default:				/* ***************** */
    report(LOG_ERR, "Programming error, unrecognized action %d",
	   perm->action);
    break;

  }/** end switch **/

  return (0);			/* zero indicates the response not modified */

}	/** end check_key() **/


/*+ read_config:
 *
 */
struct _boolopts {
  char *str;
  int  *boolval;
} ;
struct _boolopts boolopts[] = {
  {"BLANKPASSWORD", &blankpassword},
  {"HOSTWTMP", &update_host_wtmp},
  {"IGNORECASE", &ignorecase},
  {"LOGGING", &logging},
  {"NONAMESERVER", &nonameserver},
  {"QUIETNOUSER", &Quiet},	/* DENY if user exists and bad password */
  {"QUIET", &quiet},		/* never send DENY back */
  {"", NULL}
} ;

read_config(cfile)
  char *cfile;
{
  register int i, j;
  char line[BUFSIZ];
  int pct = 0;		/* counter index for the permission lists */
  FILE *cf = NULL;

  if (debug)
    report(LOG_DEBUG, "read_config: reading config file %s", cfile);

  if ((cf = fopen(cfile, "r")) == NULL)
  {
    report(LOG_ERR,"read_config: fopen() error for file %s- %s, exiting",
	   cfile, sys_errlist[errno]);
    return (0);
  }

  for (i = 0; i < MAX_AUTHTOKENS; ++i)
    authtoken[i].keyword = NULL ;

  while (fgets(line, sizeof (line), cf) != NULL)
  {
    int permflag = 0;		/* reset every time */
    static int linenum = 0;
    char *s;

    ++linenum;
    if ( (s = (char *)strchr(line, '\n')) != NULL )
      *s = '\0';                      /* rid of newline */
    s = line;

    s = skip_spaces(s);
    if (*s == '#' || *s == '\0')	/* ignore comments */
      continue;

	/*
	 * All the BOOLEAN options
	 */
    for (j=0 ; *(boolopts[j].str) != '\0'; ++j)
      if (strncasecmp(s, boolopts[j].str, strlen(boolopts[j].str)) == 0)
	break;

    if (*(boolopts[j].str) != '\0')
    {
      *(boolopts[j].boolval) = 1;
#ifdef DEBUG
      if (debug)
	report(LOG_DEBUG, "read_config: %s = %d",
	       boolopts[j].str, *(boolopts[j].boolval));
#endif
      continue;	/* do while() and read next config line */
    }

    /*
	 * Other STRING options
	 */
    if (strncasecmp(s, "DEBUGLEVEL", strlen("DEBUGLEVEL")) == 0)
    {
      s = strtok(s, " \t=");
      sscanf(skip_spaces(strtok(NULL, " \t\n")), "%d", &debug);
      if (debug)
	report(LOG_DEBUG, "read_config: debug level set to %d", debug);
      continue;
    }
    if (strncasecmp(s, "ABORTSECS", strlen("ABORTSECS")) == 0)
    {
      s = strtok(s, " \t=");
      sscanf(skip_spaces(strtok(NULL, " \t\n")), "%ld", &abort_timeout);
      if (debug)
	report(LOG_DEBUG, "read_config: ABORTSECS set to %ld",
	       abort_timeout);
      continue;
    }

#define CmpStoreStrOpt(stR, loC) { \
      if (strncasecmp(s, stR, strlen(stR)) == 0) { \
						     s = strtok(s, " \t"); \
						     loC = Strdup(skip_spaces(strtok(NULL, " \t\n"))); \
						     continue; \
      } \
    }	/* end define macro */

    CmpStoreStrOpt("WTMP", wtmp_file) ;
    CmpStoreStrOpt("UTMP", utmp_file) ;
#ifdef QI
    CmpStoreStrOpt("QI_HOST1", QI_host1) ;
    CmpStoreStrOpt("QIHOST1",  QI_host1) ;
    CmpStoreStrOpt("QI_HOST2", QI_host2) ;
    CmpStoreStrOpt("QIHOST2",  QI_host2) ;
    CmpStoreStrOpt("QI_TIMEOUT", QI_timeoutstr) ;
    CmpStoreStrOpt("QITIMEOUT",  QI_timeoutstr) ;
    CmpStoreStrOpt("QI_TYPE",  QI_type) ;
    CmpStoreStrOpt("QITYPE",   QI_type) ;
    CmpStoreStrOpt("QI_UID",   QI_uid) ;
    CmpStoreStrOpt("QIUID",    QI_uid) ;
    CmpStoreStrOpt("QI_GID",   QI_gid) ;
    CmpStoreStrOpt("QIGID",    QI_gid) ;
    CmpStoreStrOpt("QI_SHELL", QI_shell);
    CmpStoreStrOpt("QISHELL",  QI_shell);
    CmpStoreStrOpt("QI_GECOS", QI_gecos);
    CmpStoreStrOpt("QIGECOS",  QI_gecos);
#endif

    if (strncasecmp(s, "PASSW", strlen("PASSW")) == 0)
    {
      char *t;
      s = strtok(s, " \t");
      t = skip_spaces(strtok(NULL, " \t\n"));	/* filename */

#ifndef MAX_PASSWD_FILES
# define MAX_PASSWD_FILES 5		/* XXXXX */
#endif
      if (strcasecmp(t, "DEFAULT") == 0)
	do_system_pw = 1;			/* getpwname, etc. */
      else if (pwfilect < MAX_PASSWD_FILES)
	pwfile[pwfilect++] = Strdup(t);
      else
	report (LOG_ERR,
		"read_config [%d]: Exceed max number of passwd files, skipping %s",
		linenum, t);
      continue;
    }

    /*
	 * Special keywords in password fields that cause a program to
	 * execute:
	 *	AUTHTOKEN  keyword   program-to-execute
	 */
    if (strncasecmp(s, "AUTHTOKEN", strlen("AUTHTOKEN")) == 0)
    {	
      s = strtok(s, " \t");
      if (authtokenct < MAX_AUTHTOKENS)
	authtoken[authtokenct].keyword = Strdup(skip_spaces(strtok(NULL, " \t\n")));
      else
      {
	report (LOG_ERR,
		"read_config [%d]: Exceed max number of authtokens, skipping %s",
		linenum, skip_spaces(strtok(NULL, " \t\n")) );
	continue;
      }

      s = strtok(NULL, " \t");
      if (s != NULL)
      {
	authtoken[authtokenct].exec = Strdup(skip_spaces(s));
	++authtokenct;
      }
      else
	report (LOG_ERR,
		"read_config [%d]: Missing exec program in AUTH line, skipping",
		linenum);
      continue;
    }


    /*
	 * Here are the permission lines (user /group /gecos). Store the
	 * line information in the permlist structure.
	 *  USER name HOST name [mask] [request-type|ALL]  [action-type] <args>
	 *  GROUP gid  ...
	 *  GECO  string ...
	 *
	 *	[action] is one of  deny/acl/norouting/ etc. (act[] array)
	 */
    for (i=0; i < EO_K; ++i)
    {
      if (strncasecmp(s, uinfo[i].str, strlen(uinfo[i].str)) == 0)
      {
	if (pct >= MAXCFLINES)
	  report (LOG_ERR, 
		  "read_config [%d]: Exceeding max config lines, ignoring '%s'",
		  linenum, s);
	else
	  permflag++ ;	/* this was reset at the beginning each time */

	break;
      }
    }	/* end:  for()  */


    if (permflag)		/* was indeed a permission line */
    {
      char *line = NULL;

      bzero(permlist+pct, sizeof(struct _perm));

      permlist[pct].keytype = uinfo[i].type;

      s = strtok(s, " \t");	/* skip past first keyword */

      permlist[pct].strkey = Strdup(strtok(NULL, " \t\n") );  /* key */

      s = strtok(NULL, " \t\n");		/* HOST name */
      if (strcasecmp("host", s) != 0)
      {
	report(LOG_WARNING,
	       "read_config [%d]: Missing HOST, ignoring '%s'",
	       linenum, s);
	continue;
      }
      permlist[pct].host = Strdup(strtok(NULL, " \t\n"));

      s= strtok(NULL, " \t\n");	/* request type or LINE or MASK */
      if (strcasecmp("mask", s) == 0)
      {
	char *mask;
	mask = strtok(NULL, " \t\n");
	if ( *(permlist[pct].host) > 0 && *(permlist[pct].host) <= 9)
	{
	  unsigned long a, b, c, d;
	  sscanf(mask, "%ul.%ul.%ul.%ul", &a, &b, &c, &d);
	  permlist[pct].mask = (a << 24) + (b << 16) + (c << 8) + d;
	  sscanf(permlist[pct].host, "%ul.%ul.%ul.%ul", &a, &b, &c, &d);
	  permlist[pct].hostip = (a << 24) + (b << 16) + (c << 8) + d;
	  report(LOG_DEBUG, "read_config [%d]: Scanned ip %s/%s (%lu/%lu)",
		 linenum, permlist[pct].host, permlist[pct].mask,
		 permlist[pct].hostip, permlist[pct].mask);
	}
	s= strtok(NULL, " \t\n");			/* request type */
      }

      if (strcasecmp("line", s) == 0)
      {
	/* defer parsing of `line' to later because the `strtok' buffer
	 * is in use now.
	 */
	line = Strdup(strtok(NULL, " \t\n"));
	if (strcasecmp("any", line) == 0 ||
	    strcasecmp("all", line) == 0)
	  line = NULL;
	s= strtok(NULL, " \t\n");		/* request type */
      }

	        
      for (i = 0; *(req[i].str) != '\0'; ++i)	/* process request type */
	if (strcasecmp(req[i].str, s) == 0)
	{
	  permlist[pct].request_type = req[i].type;
	  permlist[pct].strreq_type = req[i].str;
	  break;
	}

      if (logging && *(req[i].str) == '\0')
      {
	syslog(LOG_ERR, "read_config [%d]: unknown request_type: '%s'",
	       linenum, s);
	continue;	/* ignore the line */
      }

      s= strtok(NULL, " \t\n");  		/* action or LEVEL */
      if (strcasecmp("level", s) == 0)
      {
	s= strtok(NULL, " \t\n");
	permlist[pct].enlevel = atoi(s);
	s= strtok(NULL, " \t\n");               /* action */
      }
      else
	permlist[pct].enlevel = ALL_ENABLE_LEVEL;
               
      for (i = 0; *(acts[i].str) != '\0'; ++i)
	if (strcasecmp(acts[i].str, s) == 0) {
	  permlist[pct].action =  acts[i].enumi ;
	  permlist[pct].straction = acts[i].str ;
	  break;
	}

      if (logging && *(acts[i].str) == '\0') {
	syslog(LOG_ERR, "read_config [%d]: unknown action: '%s'", linenum, s);
	continue ;	/* ignore line */
      }

      s= strtok(NULL, "\0");  		/* remainder of line */
      if (s && *s != '\0')
	permlist[pct].args = Strdup(skip_spaces(s));	/* arguments */
      else
	permlist[pct].args = "" ;

      /* Here we have a valid line... print out for debugging */
      report (LOG_DEBUG,
	      "read_config: %s %s @ %s, REQUEST %s %s ARGS= %s", 
	      uinfo[permlist[pct].keytype].str, permlist[pct].strkey,
	      permlist[pct].host, permlist[pct].strreq_type, 
	      permlist[pct].straction, permlist[pct].args);
	    
      /* now strtok can be used to parse the `LINE' sequence */
      if (line != NULL)
      {
	struct _lineno *lineno;
	int i, j;

	s = strtok(line, ",");
	while (s) 
	{       /* kissg 951112  range accepted in LINE list */
	  switch (sscanf(s,"%d-%d",&i,&j))
	  {
	  case 0:		/* wrong input */
	    i=0; j= -1;
	    break;
	  case 1:		/* only one number */
	    j=i;
	    break;
	  case 2:		/* two numbers, assume range */
	    break;
	  }

	  for ( /*EMPTY*/ ; i<=j; i++)
	  {
	    lineno= (struct _lineno *)malloc(sizeof(struct _lineno));
	    lineno->line = i;
	    lineno->next = permlist[pct].line;
	    permlist[pct].line = lineno;
	  }

	  s = strtok(NULL, ",");
	}
      }
	    
      ++pct;		/* increment the list pointer */
      continue;
    }	/* end if permflag */

    /* shouldn't get to this point */

    report(LOG_ERR, "read_config [%d]: invalid line: '%s'", linenum, s);
  }

  bzero(&permlist[pct], sizeof (struct _perm));	/* null last entry */

#ifdef DEBUG
  if (debug)
  {
    for(i=0; i < authtokenct; i++)
      report(LOG_DEBUG,"read_config: AUTHTOKEN <%s> <%s>",
	     authtoken[i].keyword, authtoken[i].exec);
    for(i=0; i < pwfilect; i++)
      report(LOG_DEBUG,"read_config: password file <%s>", pwfile[i]);
  }
#endif
  return(1);	/* all okay ? */
}

