/*
 * $Header: /home/vikas/src/xtacacsd/RCS/Getpw.c,v 1.9 1997/01/11 06:48:41 vikas Exp $
 */

/*
 * Since all this POSIX-ing doesn't even seem to help the 'getpw...'
 * routines, heres a simple set that will extract the password fields
 * from any given filename.
 *
 * Have added support for creating and using DBM databases from passwd files.
 * It stores the passwd file keyed by name and also by lowercase name
 * so that it can extract names using the 'ignorecase' option in Getpwnam()
 *
 * Just wanted it for the xtacacsd package.
 *
 * USAGE:
 *		reset_pwfile(char *filename)
 *		Getpwnam(name, ignorecase)  tries to use DBM database
 *		Fgetpwent(NULL)	fetches next entry in password file
 *		Endpwent()	to close password file
 *
 * BUGS:
 *	By the time you wake up, 'they' could have changed the
 * structure of the password file so the fields might not be filled in
 * properly. However, the location of the username/password/uid/gid is
 * usually the same, so the 'important' fields needed for xtacacs
 * should still be salvagable.
 *
 * 	Vikas Aggarwal, vikas@navya.com, Dec 1994
 *
 * $Log: Getpw.c,v $
 * Revision 1.9  1997/01/11 06:48:41  vikas
 * Added extra paren's to avoid macro bugs. Added logic to
 * try and detect BSD style passwd file on SysV machines in linetopw()
 *
 * Revision 1.7  1996/05/19  17:06:58  vikas
 * Added support for old style dbm routines in addition to the
 * new style ndbm routines.
 *
 * Revision 1.6  1996/03/19  20:27:45  vikas
 * Now uses dbm format database for the passwd files. Also created
 * easy macro for extracting the password fields.
 *
 * Revision 1.5  1996/03/15  18:19:24  vikas
 * Now using a macro definition to get the next password field.
 * Instead of returning null string ptrs, now returns ptr to null string
 *
 * Revision 1.4  1996/03/14  22:33:25  vikas
 * Added the Getpwnam() function (it was in misc.c before).
 *
 * Revision 1.3  1995/11/20  19:16:54  vikas
 * Fixed bug in parsing of directory and shell (kissg@sztaki.hu)
 *
 * Revision 1.2  1995/07/14  16:10:59  vikas
 * Added OSF and FreeBSD 2.0 customizations (grr).
 *
 * Revision 1.1  1994/12/15  21:40:50  vikas
 * Initial revision
 *
 *
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <pwd.h>
#include <ctype.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifndef BUFSIZ
# define BUFSIZ 1024
#endif

#ifdef USE_NDBM
# include <ndbm.h>
#else
# ifdef USE_DBM		/* older library */
#  include <dbm.h>
#  if defined (sun)
#   undef NULL		/* dbm.h sets it to char *, undo */
#   define NULL 0
#  endif /* sun */
# endif	/* USE_DBM */
#endif	/* USE_NDBM */

#if defined(SYSV) || defined(SVR4) || defined(__svr4__)
#  define index strchr
#  define rindex strrchr
#  define bcopy(a, b, c) memmove(b, a, c)
#  define bzero(a, b)    memset(a, 0, b)
#endif

extern int 	errno;
#if !defined(__FreeBSD__) && !defined(__BSDI__) && !defined(HAVESYSERR)
extern char 	*sys_errlist[];		/* ansi definition in stdio.h */
#endif	/* not HAVESYSERR */

static int	getpw_debug;
static char 	curpwfile[BUFSIZ];
static struct	passwd *linetopw();
static FILE	*fcurpw;

struct passwd 	*Fgetpwent(), *Getpwnam();

#ifdef USE_NDBM
  static DBM	*dbmfile;		/* database file pointer */
# define isdbm_open(F)	((F == NULL) ? 0:1)
#else
# ifdef USE_DBM				/* create NDBM equivalents */
  static int	dbmfile = -1;		/* 0 indicates its open */
# define isdbm_open(F)	((F == 0) ? 1:0)

# define dbm_open(F,M,X) dbminit(F)
# define dbm_close(F)	dbmclose()
# define dbm_fetch(F,K)	fetch(K)
# define dbm_store(F,K,D,I)	store(K,D)
# define DBM_INSERT 0
# endif	/* USE_DBM */
#endif	/* USE_NDBM */

#define NAMELEN	32		/* for consistent char array allocation */


/*+ Getpwnam:
 * 	Replacement for getpwnam() POSIX function but implemented using
 * getpwent() instead to allow matching case insensitive usernames
 */
struct passwd *
Getpwnam (name, ignorecase)
     char *name;
     int ignorecase;		/* 1 to ignorecase, 0 to look at case */
{
    struct passwd *pw;
    static char line[BUFSIZ+1];

#if defined(USE_NDBM) || defined(USE_DBM)
    if (isdbm_open(dbmfile))	/* was opened by reset_pwfile() */
    {
	char lname[NAMELEN+1];
	datum	key, data;

	pw = NULL;		/* initial value */

	if (ignorecase) {
	    register char *lp = lname, *np = name;

	    while ( (*lp = tolower(*np++)))	/* convert query to lower */
	      lp++;
	    *lp = '\0';				/* terminate string */
	    key.dptr = lname;
	}
	else
	  key.dptr = name;
	key.dsize= strlen(name);

#ifdef DEBUG
	    if (getpw_debug > 1)
	      fprintf(stderr, "(dbg) Calling dbm_fetch()\n"); /* */
#endif
	data = dbm_fetch(dbmfile, key);

	if (data.dptr != NULL) {
#ifdef DEBUG
	    if (getpw_debug)
	      fprintf(stderr, "(dbg) dbm_fetched() '%.*s'\n", data.dsize, data.dptr); /* */
#endif
	    bcopy(data.dptr, line, data.dsize);	/* save to static storage */
	    line[data.dsize] = '\0';	/* terminating NULL to be safe */
	    pw = linetopw(line);
	}

	if (pw)
	  return(pw);		/* search using regular method if not found */

    }	/* end if (dbmfile) */
#endif	/* USE_NDBM */

    while ( (pw = Fgetpwent(NULL)) != NULL)
      if ( ((ignorecase && strcasecmp(pw->pw_name, name) == 0)) ||
	  (strcmp(pw->pw_name, name) == 0) )
	break ;

    return (pw);	/* can be NULL */

}	/* end: Getpwnam */
    


/*+ Fgetpwent
 * Return next password file entry if pwfile is NULL. Close the current file
 * if password filename is empty string.
 * Does not use the DBM file format since there is no advantage in using it
 * (we are looking at all entries anyway), and we would rather not use DBM
 * files in case the DBM files are corrupted (note that Getpwnam() uses this
 * routine as a fallback if extracting via DBM fails).
 */
struct passwd *
Fgetpwent(pwFile)
     char *pwFile;
{
    static char line[BUFSIZ+1];		/* needs to be static */

    if (pwFile == NULL)	{	/* continue working in current file */
	if (fcurpw == NULL)	/* at end of file */
	  return(NULL);
    }
    else if (*pwFile == '\0') {	/* close the current file */
	Endpwent();
	return(NULL);
    }
    else 			/* new or old filename, close and open */
      reset_pwfile(pwFile);

    /* Here if the 'fcurpw' file pointer is okay */

    if (fgets(line, BUFSIZ, fcurpw) == NULL)
      return (NULL);		/* end of file */

    return (linetopw(line));

}	/* end Fgetpwent() */


/*+ linetopw
 * Convert line to an passwd structure. Note that there is no limitation on the
 * length of the passwords, etc. (unlike the typical Unix getpwent() call)
 */
static struct passwd *
linetopw(line)
     char *line;	/* this should be a static char array */
{
    register char *p, *q;
    char *uid = NULL, *gid = NULL;
    static struct passwd pwp;

/* Macro to replace separator with a '\0' and position 'p' to next field */
#define  getfield(Fld, Sep) \
    { \
	Fld = p; \
	if (!p) \
	  Fld = ""; \
	if ( p && (q = strchr(p, Sep)) != NULL) { \
	   *q = '\0'; ++q; \
	} \
	p = q; \
    }

    bzero(&pwp, sizeof pwp);
    if ((q = strrchr(line, '\n')))
      *q = '\0' ;			/* replace \n */
    p = line ;

    getfield(pwp.pw_name, ':');
    getfield(pwp.pw_passwd, ':');
    getfield(uid, ':');
    getfield(gid, ':');

#if defined(SYSV) || defined(SVR4) || defined(__svr4__)
    getfield(pwp.pw_age, ':');
    getfield(pwp.pw_comment, ':');
#endif /* SYSV */

    getfield(pwp.pw_gecos, ':');

#if defined(SYSV) || defined(SVR4) || defined(__svr4__)
    /* If we reached end of line, probably no age/comments (i.e. even though
     * we are sysV, we have a BSD style password file. Its the best logic one
     * can use...
     */
    if (p == '\0')
    {
      pwp.pw_shell = pwp.pw_gecos;
      pwp.pw_dir   = pwp.pw_comment;
      pwp.pw_gecos = pwp.pw_age;

      pwp.pw_age = "";		/* set to empty strings */
      pwp.pw_comment = "";
    }
    else
    {
      getfield(pwp.pw_dir, ':');
      getfield(pwp.pw_shell, ':');
    }
#endif /* SYSV */

    getfield(pwp.pw_dir, ':');
    getfield(pwp.pw_shell, ':');
    
    /*
     * Note that if there was no UID or GID string, then value 0 is returned
     */
    if (uid)
      pwp.pw_uid = strtol(uid, (char **)NULL, 10);
    if (gid)
      pwp.pw_gid = strtol(gid, (char **)NULL, 10);

#if defined(OSF) || defined(__osf__)
#ifdef PW_QUOTA
    pwp.pw_quota   = 0;
#endif
    pwp.pw_comment = "";
#endif	/* __osf__ */

    return (&pwp);

#undef getfield
}


/*+ reset_pwfile
 * Close any current password file and reopen the supplied password file.
 * If using DBM, then dbm_open the curpwfile.
 */
reset_pwfile(pwFile)
     char *pwFile;
{
    if (pwFile == NULL || *pwFile == '\0')
      return (0);

    Endpwent();
    if ((fcurpw = fopen(pwFile, "r")) == NULL) {
	fprintf(stderr, "reset_pwfile: fopen of %s failed- %s\n",
		pwFile, sys_errlist[errno]);
	return (-1);
    }
    else
      strcpy(curpwfile, pwFile);

#if defined(USE_NDBM) || defined(USE_DBM)
    dbmfile = dbm_open(curpwfile, O_RDONLY, 0);	/* NULL or -ve on error */
#endif

    return (0);
}

/*+
 * Close password file. If DBM used, close database ptr.
 */

Endpwent()
{
    if (fcurpw)
      fclose(fcurpw);
    curpwfile[0] = '\0';

#if defined(USE_NDBM) || defined(USE_DBM)
    if (isdbm_open(dbmfile))
      dbm_close(dbmfile);
# ifdef USE_NDBM
    dbmfile = NULL;
# else	/* USE_DBM */
    dbmfile = -1;
# endif
#endif /* if USE_DBM || USE_DBM */
}

/*+ create_dbm_db()
 * Routine to create a hashed DBM database for the password entries. Simply
 * store the entire password lines since we will have to unravel it all
 * anyways (might as well use linetopw() after fetching the line).
 * Stores the terminating NULL of each line also.
 */

#if defined(USE_NDBM) || defined(USE_DBM)
create_dbm_db(file)
     char *file;	/* password file */
{
    long	count = 0;
    char    	line[BUFSIZ];
    char	tfile[255], xfile[255], zfile[255];	/* file names */
    struct passwd	*p;
    datum	key, data;
    
    /*
     * Open the ASCII password file- we will use the fcurpw FILE ptr
     */
    reset_pwfile(file);		/* this opens the file and sets fcurpw */
    if (curpwfile == NULL)	/* could not be opened */
      exit(1);

    sprintf(tfile, "%sX", file); /* temp filename for creating database */

#ifdef USE_DBM		/* need to create zero length files in older DBM*/
    sprintf(xfile, "%s.dir", tfile);    
    sprintf(zfile, "%s.pag", tfile);
    open(xfile, O_WRONLY| O_CREAT| O_TRUNC, 0640);
    open(zfile, O_WRONLY| O_CREAT| O_TRUNC, 0640);
    close(xfile) ; close(zfile);
#endif /* USE_DBM */
    dbmfile = dbm_open(tfile, O_CREAT| O_EXCL| O_RDWR, 0640); /* not WRONLY */

    if (isdbm_open(dbmfile) == 0) {
	fprintf(stderr, "dbm_open() of temp file ");
	perror(tfile);
	exit (1);
    }

    while (fgets(line, BUFSIZ, fcurpw))
    {
	int rcode;
	char	name[NAMELEN], lname[NAMELEN];
	register char  *lp = line, *np = name, *lnp = lname;

	for (*np = *lp++; *np && *np != ':'; *np = *lp++)  {
	    *lnp++ = tolower(*np);	/* lowercase name */
	    ++np;
	}
	*np = *lnp = '\0';		/* terminate string */
	if (strcmp(name, lname) == 0)	/* was already lowercase */
	  lname[0] = '\0';
#ifdef DEBUG
	if (getpw_debug)
	  fprintf(stderr, "%s ", name);	/* */
#endif
	data.dptr = line;
	data.dsize = strlen(line) + 1;	/* save terminating NULL also */
	key.dptr = name;
	key.dsize = strlen(name);

	rcode = dbm_store(dbmfile, key, data, DBM_INSERT);

	if (rcode == 0 && *lname)
	{
	    key.dptr = lname;		/* store with lowercase key also */
	    rcode = dbm_store(dbmfile, key, data, DBM_INSERT);
	}

	if (rcode < 0)
	{
	    fprintf(stderr, "dbm_store ");
	    perror(name);
	    exit(1);
	}
	else if (rcode == 1)
	  fprintf(stderr,
		  "(dbm_store) Could not insert duplicate entry for '%s'\n",
		  name);
	else
	  ++count;

    }	/* end while() */

    dbm_close(dbmfile);

#ifdef DEBUG
    fprintf(stderr, "\n");
    fflush(stderr);
#endif

    sprintf(zfile, "%s.dir", file);
    sprintf(xfile, "%s.dir", tfile);

    if (rename(xfile, zfile) < 0) {
	fprintf(stderr, "rename() %s ", xfile);
	perror(zfile);
	exit(1);
    }

    sprintf(zfile, "%s.pag", file);
    sprintf(xfile, "%s.pag", tfile);

    if (rename(xfile, zfile) < 0) {
	fprintf(stderr, "rename() %s ", xfile);
	perror(zfile);
	exit(1);
    }

    printf ("Stored %ld entries\n", count);

    return(0);
    
}	/* end: create_dbm_db() */

#ifdef NO_RENAME
rename(src, dest)
     char    *src, *dest;
{
    if (unlink(dest) < 0)
      if (errno != ENOENT)
	return(-1);

    if (link(src, dest) < 0)
      return(-1);

    return(unlink(src));
}
#endif	/* NO_RENAME */

#endif /* USE_NDBM */


/*+
 *
 * Main for testing and also to create the password DBM database.
 *
 */

#ifdef GETPW_MAIN

main(ac, av)
     int ac;
     char *av[];
{
    int  createdbm = 0, c;
    char *name = NULL;
    struct passwd *p;
    extern char *optarg;
    extern int optind;

    umask(002);

    while ((c = getopt(ac, av, "cd")) != EOF)
      switch (c) {
       case 'c':
	  createdbm++;
	  break;
       case 'd':
	  getpw_debug++ ;
	  break;
       case '?':
       default:
	  fprintf(stderr, "Ignoring unknown flag %c\n", c);
      }

    ac -= optind ;
    if (ac == 0) {
	fprintf (stderr,
		 "usage: %s [-d (debug)] [-c (create_dbm database)] <password file> [name]\n",
		 av[0]);
	exit (1);
    }

    if (createdbm)
    {
#if defined(USE_NDBM) || defined(USE_DBM)
	create_dbm_db(av[optind]);
#else
	fprintf(stderr, "Error: program not compiled with DBM support\n");
#endif
	exit (0);
    }

    /*
     * Following is a test for printing out the password file or
     * searching for a user name.
     */
    reset_pwfile(av[optind]);	/* opens the file */
    if (ac > 1)
      name = av[optind + 1];	/* ignore other names */

    printf ("NAME : UID : GID : GECO : DIR : SHELL\n");

    if (name) {
	p = Getpwnam(name, 1);	/* ignorecase */
	print_pwd(p);
    }
    else
      for (p = Fgetpwent(NULL); p ; p = Fgetpwent(NULL))
	print_pwd(p);

    Endpwent();

    fflush(stderr);

}	/* end main() */


print_pwd(p)
     struct passwd *p;
{
    printf ("%s : %ld : %ld : %s : %s : %s",
	    p->pw_name, p->pw_uid, p->pw_gid, p->pw_gecos,
	    p->pw_dir, p->pw_shell);
#if defined(SYSV) || defined(SVR4) || defined(__svr4__)
    printf (" age= %s: comment= %s:", p->pw_age, p->pw_comment);
#endif
    printf ("\n");
}


#endif /* GETPW_MAIN */


