/*
 * pt_sh - pt_lib.c                                                 August 2002
 */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ptrace.h>
#include <sys/wait.h>
#include <linux/user.h>
#include <signal.h>
#include <errno.h>
#include "pt_lib.h"

void pt_traceme()
{ 
  if(ptrace(PTRACE_TRACEME, 0, NULL, NULL))
  { 
    perror("pt_traceme");
    exit(-1);
  }
}

long int pt_peek(pid_t pid, unsigned long addr)
{ 
  long int result;
  if((result = ptrace(PTRACE_PEEKTEXT, pid, (void*)addr, NULL)) == -1)
    if(errno)
    {
      perror("pt_peek");
      return -1;
    }
  return result;
}

void pt_poke(pid_t pid, unsigned long addr, long int value)
{
  if(ptrace(PTRACE_POKETEXT, pid, (void*)addr, (void*)value)<0)
    perror("pt_poke");
}

struct user_regs_struct* pt_getregs(pid_t pid)
{
  struct user_regs_struct *reg = 
    (struct user_regs_struct*)(malloc(sizeof(struct user_regs_struct)));
  if(!reg)
  {
    perror("malloc");
    return NULL;
  }
  else if(ptrace(PTRACE_GETREGS, pid, NULL, (void*)reg)<0)
  {
    perror("pt_getregs");
    free((void*)reg);
    return NULL;
  }
  return reg;
}

void pt_setregs(pid_t pid, struct user_regs_struct *reg)
{
  if(ptrace(PTRACE_SETREGS, pid, NULL, (void*)reg)<0)
  {
    perror("pt_setregs");
  }
}

struct user_fxsr_struct* pt_getfpregs(pid_t pid)
{
  struct user_fxsr_struct* reg =
    (struct user_fxsr_struct*)(malloc(sizeof(struct user_fxsr_struct)));
  if(!reg)
  {
    perror("malloc");
    return NULL;
  }
  else if(ptrace(PTRACE_GETFPREGS, pid, NULL, (void*)reg)<0)
  {
    perror("pt_getfpregs");
    free((void*)reg);
    return NULL;
  }
  return reg;
}

void pt_setfpregs(pid_t pid, struct user_fxsr_struct *reg)
{
  if(ptrace(PTRACE_SETFPREGS, pid, NULL, (void*)reg)<0)
  {
    perror("pt_setfpregs");
  }
}

void pt_cont(pid_t pid)
{
  if(ptrace(PTRACE_CONT, pid, NULL, NULL))
  {
    perror("pt_cont");
    exit(-1);
  }
}

unsigned int pt_syscall(pid_t pid)
{
  int s = 0;
  if(ptrace(PTRACE_SYSCALL, pid, NULL, NULL)<0)
  {
    perror("pt_syscall");
    exit(-1);
  }
  while(((s=wait_signal(pid)) != SIGTRAP) && (s>0)) pt_cont(pid);
  if(s<0) return 0;
  if((s = ptrace(PTRACE_PEEKUSER, pid, 4*ORIG_EAX, NULL))<0)
  {
    perror("pt_syscall");
    return 0;
  }
  return s;
}

void pt_step(pid_t pid)
{
  if(ptrace(PTRACE_SINGLESTEP, pid, NULL, NULL)<0)
  {
    perror("pt_step");
    exit(-1);
  }
}

void pt_goto(pid_t pid, unsigned long addr)
{
  struct user_regs_struct reg;
  int status;
  if(ptrace(PTRACE_GETREGS, pid, NULL, (void*)&reg)<0)
  {
    perror("pt_goto");
    return;
  }
  while(reg.eip != addr)
  {
    if(ptrace(PTRACE_SINGLESTEP, pid, NULL, NULL)<0)
    {
      perror("pt_goto");
      exit(-1);
    }
    status = SIGCHLD;
    waitpid(pid, &status , 0);
    if(ptrace(PTRACE_GETREGS, pid, NULL, (void*)&reg)<0)
    {
      perror("pt_goto");
      exit(-1);
    }
  }
}

void pt_kill(pid_t pid)
{
  if(ptrace(PTRACE_KILL, pid, NULL, NULL))
  {
    perror("pt_kill");
    exit(-1);
  }
}

void pt_detach(pid_t pid)
{
  if(ptrace(PTRACE_DETACH, pid, NULL, NULL))
  {
    perror("pt_detach");
    exit(-1);
  } 
} 

pid_t run_target(char **argv)
{
  int   status;
  pid_t pid = fork();
  if(pid<0)
  {
    perror("fork");
    exit(-1);
  }
  else if(!pid)               /* il s'agit du fils  tracer */
  {
    /* lancer le tracage */
    pt_traceme();
    /* lancer le programme a dumper */
    execv(argv[0], argv);
    perror("execv");
    exit(-1);
  }
  // ici le pre trace le process fils
  return pid;
}

int wait_signal(pid_t pid)
{
  int status;
  if(waitpid(pid, &status, WUNTRACED)<0)
  {
    perror("waitpid");
    exit(-1);
  }

  return WIFEXITED(status) ? WIFSIGNALED(status) ? - WTERMSIG(status) 
           : - _NSIG 
           : WSTOPSIG(status);
}

int next_signal(pid_t pid, int sig)
{
  pt_cont(pid);

  return wait_signal(pid);
}
