/*
 * pt_sh : pt_box.c
 */

#include <stdio.h>
#include <linux/user.h>
#include <signal.h>
#include <sys/types.h>
#include "pt_lib.h"
#include "pt_symbol.h"


#define PRINT_REGISTER(s,a)   printf(s "\t0x%08x\t(%u)\n", a, a)

pid_t current;

int childrunning()
{
  return current ? 1 : 0;
}

void restart(char *argv[])
{
  int s;
  if(current)
    pt_kill(current);
  printf("Running %s...\n", argv[0]);
  current = run_target(argv);
  while(((s = wait_signal(current))>=0) && (s!=SIGTRAP)) pt_cont(current);
  printf("SIGTRAP catched...\n");
}

void get_addr(unsigned long addr)
{
  long int r = pt_peek(current, addr);
  printf("addr %p is %d (0x%x)\n", addr, r, r);
}

void set_addr(unsigned long addr, long int value)
{
  pt_poke(current, addr, value);
}

void print_map()
{
  FILE *f;
  char  buf[1024];
  printf("map file :\n");
  snprintf(buf, 1023, "/proc/%u/maps", current);
  buf[1023] = 0;
  f = fopen(buf, "r");
  if(!f)
    perror("fopen");
  else
  {
    while(!feof(f))
    {
      fgets(buf, 1023, f);
      buf[1023] = 0;
      puts(buf);
    }
    fclose(f);
  }
}

void hexdump(unsigned long addr, long size, int byteprec)
{
  long i, v;
  for(i=0; i<size; i+=sizeof(unsigned long))
  {
    v = pt_peek(current, addr+i);
    if(!(i%20)) printf("\n%08X: ", addr+i);
    if(byteprec)
      printf("%02X %02X %02X %02X ",
             ((unsigned char*)&v)[0], ((unsigned char*)&v)[1],
             ((unsigned char*)&v)[2], ((unsigned char*)&v)[3]);
    else printf("%08X ", v);
  }
  printf("\n");
}

void dumpfile(unsigned long addr, long size, char *outfile)
{
  long i;
  unsigned long a;
  FILE *f = fopen(outfile, "w");
  if(!f)
  {
    perror("fopen");
    exit(-1);
  }
  else
  {
    printf("Dumping %08X-%08X to %s...", addr, addr+size, outfile);
    for(i=0; i<size; i+=sizeof(unsigned long))
    {
      a = pt_peek(current, addr+i);
      fwrite(&a, 4, 1, f);
    }
    fclose(f);
    printf(" done !\n");
  }
}

void get_regs(char *reglist[])
{
  int i;
  struct user_regs_struct *reg = pt_getregs(current);
  struct user_fxsr_struct *reg2 = pt_getfpregs(current);
  if(reg && reg2)
  {
    if(reglist)
    {
      for(i=0; reglist[i]; i++)
      {
        if(!strcmp(reglist[i], "eax"))
          PRINT_REGISTER("eax", reg->eax);
        else if(!strcmp(reglist[i], "ebx"))
          PRINT_REGISTER("ebx", reg->ebx);
        else if(!strcmp(reglist[i], "ecx"))
          PRINT_REGISTER("ecx", reg->ecx);
        else if(!strcmp(reglist[i], "edx"))
          PRINT_REGISTER("edx", reg->edx);
        else if(!strcmp(reglist[i], "esi"))
          PRINT_REGISTER("esi", reg->esi);
        else if(!strcmp(reglist[i], "edi"))
          PRINT_REGISTER("edi", reg->edi);
        else if(!strcmp(reglist[i], "ebp"))
          PRINT_REGISTER("ebp", reg->ebp);
        else if(!strcmp(reglist[i], "eip"))
          PRINT_REGISTER("eip", reg->eip);
        else if(!strcmp(reglist[i], "esp"))
          PRINT_REGISTER("esp", reg->esp);
        else if(!strcmp(reglist[i], "ds"))
          PRINT_REGISTER("ds",  reg->ds);
        else if(!strcmp(reglist[i], "fs"))
          PRINT_REGISTER("fs",  reg->fs);
        else if(!strcmp(reglist[i], "cs"))
          PRINT_REGISTER("cs",  reg->cs);
        else if(!strcmp(reglist[i], "ds"))
          PRINT_REGISTER("ds",  reg->ds);
        else if(!strcmp(reglist[i], "flags"))
          PRINT_REGISTER("flags", reg->eflags);
	else if(!strcmp(reglist[i], "cwd"))
          PRINT_REGISTER("cwd", reg2->cwd);
        else if(!strcmp(reglist[i], "swd"))
          PRINT_REGISTER("swd", reg2->swd);
        else if(!strcmp(reglist[i], "twd"))
          PRINT_REGISTER("twd", reg2->twd);
        else if(!strcmp(reglist[i], "fop"))
          PRINT_REGISTER("fop", reg2->fop);
        else if(!strcmp(reglist[i], "fip"))
          PRINT_REGISTER("fip", reg2->fip);
        else if(!strcmp(reglist[i], "fcs"))
          PRINT_REGISTER("fcs", reg2->fcs);
        else if(!strcmp(reglist[i], "foo"))
          PRINT_REGISTER("foo", reg2->foo);
        else if(!strcmp(reglist[i], "fos"))
          PRINT_REGISTER("fos", reg2->fos);
        else fprintf(stderr, "unknow register %s !\n", reglist[i]);
      }
    }
    else
    {
      PRINT_REGISTER("eax", reg->eax);
      PRINT_REGISTER("ebx", reg->ebx);
      PRINT_REGISTER("ecx", reg->ecx);
      PRINT_REGISTER("edx", reg->edx);
      PRINT_REGISTER("esi", reg->esi);
      PRINT_REGISTER("edi", reg->edi);
      PRINT_REGISTER("ebp", reg->ebp);
      PRINT_REGISTER("eip", reg->eip);
      PRINT_REGISTER("esp", reg->esp);
      PRINT_REGISTER("ds",  reg->ds);
      PRINT_REGISTER("fs",  reg->fs);
      PRINT_REGISTER("cs",  reg->cs);
      PRINT_REGISTER("ds",  reg->ds);
      PRINT_REGISTER("flags", reg->eflags);
      PRINT_REGISTER("cwd", reg2->cwd);
      PRINT_REGISTER("swd", reg2->swd);
      PRINT_REGISTER("twd", reg2->twd);
      PRINT_REGISTER("fop", reg2->fop);
      PRINT_REGISTER("fip", reg2->fip);
      PRINT_REGISTER("fcs", reg2->fcs);
      PRINT_REGISTER("foo", reg2->foo);
      PRINT_REGISTER("fos", reg2->fos);
    }
    free(reg);
    free(reg2);
  }
}

void set_reg(char *sreg, unsigned long value)
{
  struct user_regs_struct *reg = pt_getregs(current);
  struct user_fxsr_struct *reg2 = pt_getfpregs(current);
  if(reg && reg2)
  {
    if(!strcmp(sreg, "eax"))
      reg->eax = value;
    else if(!strcmp(sreg, "ebx"))
      reg->ebx = value;
    else if(!strcmp(sreg, "ecx"))
      reg->ecx = value;
    else if(!strcmp(sreg, "edx"))
      reg->edx = value;
    else if(!strcmp(sreg, "esi"))
      reg->esi = value;
    else if(!strcmp(sreg, "edi"))
      reg->edi = value;
    else if(!strcmp(sreg, "ebp"))
      reg->ebp = value;
    else if(!strcmp(sreg, "eip"))
      reg->eip = value;
    else if(!strcmp(sreg, "esp"))
      reg->esp = value;
    else if(!strcmp(sreg, "ds"))
      reg->ds  = value;
    else if(!strcmp(sreg, "fs"))
      reg->fs  = value;
    else if(!strcmp(sreg, "cs"))
      reg->cs  = value;
    else if(!strcmp(sreg, "ds"))
      reg->ds  = value;
    else if(!strcmp(sreg, "flags"))
      reg->eflags = value;
    else if(!strcmp(sreg, "cwd"))
      reg2->cwd = value;
    else if(!strcmp(sreg, "swd"))
      reg2->swd = value;
    else if(!strcmp(sreg, "twd"))
      reg2->twd = value;
    else if(!strcmp(sreg, "fop"))
      reg2->fop = value;
    else if(!strcmp(sreg, "fip"))
      reg2->fip = value;
    else if(!strcmp(sreg, "fcs"))
      reg2->fcs = value;
    else if(!strcmp(sreg, "foo"))
      reg2->foo = value;
    else if(!strcmp(sreg, "fos"))
      reg2->fos = value;
    else fprintf(stderr, "unknow register %s !\n", sreg);
    pt_setregs(current, reg);
    pt_setfpregs(current, reg2);
    free(reg);
    free(reg2);
  }
}

void step()
{
  pt_step(current);
}

void mgoto(unsigned long addr)
{
  pt_goto(current, addr);
  printf("Stopping at address %p\n", addr);
}

void send_sig(int sig)
{
  struct user_regs_struct *reg;
  int s;
  kill(current, sig);
  if(sig != SIGKILL) pt_cont(current);
  s = wait_signal(current);
  if(s>0)
  {
    reg = pt_getregs(current);
    printf("Received %s at %p...\n", get_signalname(s), reg ? reg->eip : 0);
    if(reg) free(reg);
  }
  else
  {
    s = -s;
    current = 0;
    if(s == _NSIG)
      printf("Child has exited normally...\n");
    else printf("Child has been killed by %s...\n", get_signalname(s));
  }
}

void mkill(int sig)
{
  int s;
  kill(current, sig);
  if(sig != SIGKILL) pt_detach(current);
  while((s = wait_signal(current))>0);
  if(s==_NSIG)
    printf("Child has exited normally...\n");
  else printf("Child has been killed by %s...\n", get_signalname(sig));
  current = 0;
}

void cont()
{
  send_sig(0);
}

void next()
{
  struct user_regs_struct *reg;
  int s = pt_syscall(current);
  if(!s)
  {
    printf("Child has exited...\n");
    current = 0;
    return;
  }
  reg = pt_getregs(current);
  printf("Stopping for %s at %p...\n", get_syscallname(s), reg ? reg->eip : 0);
  if(reg) free(reg);
}

void end()
{
  pt_detach(current);
  printf("Child has been detached !\n");
  waitpid(current, NULL, 0);
  current = 0;
}

void break_syscall(int syscall)
{
  struct user_regs_struct *reg;
  int s;
  while(((s=pt_syscall(current)) != syscall) && s);
  if(!s)
  {
    printf("Child has exited...\n");
    current = 0;
    return;
  }
  reg = pt_getregs(current);
  printf("Stopping for %s at %p...\n", get_syscallname(syscall),
                                       reg ? reg->eip : 0);
  if(reg) free(reg);
}

void break_signal(int sig)
{
  struct user_regs_struct *reg;
  int s;
  while(((s=next_signal(current, 0))!=sig) && (s>0));
   
  if(s>0)
  {
    reg = pt_getregs(current);
    printf("Received %s at %p...\n", get_signalname(s), reg ? reg->eip : 0);
    if(reg) free(reg);
  }
  else
  {
    s = -s;
    current = 0;
    if(s == _NSIG)
      printf("Child has exited normally...\n");
    else printf("Child has been killed by %s...\n", get_signalname(s));
  }
}
