//////////////////////////////////////////////////////////////////////////
// Code source  :    Shaun Dore                                         //
// Fichier      :    3DCHAP4.CPP                                        //
// Date         :    29-09-1998                                         //
// Compilateur  :    Borland C++ 3.1                                    //
// Description  :    Engin de poylgones 3D                              //
//////////////////////////////////////////////////////////////////////////

// ---------------------------  INCLUDE --------------------------------//

#include <mem.h>
#include <math.h>
#include <conio.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

// ---------------------- CONSTANTES & MACROS --------------------------//

#define MX        160                        // Millieu de l'abscisse
#define MY        100                        // Millieu de l'ordonnee
#define DISTANCE  250                        // Distances de l'obervateur
#define AMBIANT   20                         // Lumiere ambiante
#define DIFFUSE   215                        // Lumiere diffuse
#define MAX_POLY  150                        // Nb max de polygones
#define MAX_SOMM  75                         // Nb max de sommets
#define SIN(x)    SinTable[x]                // Macro SIN()
#define COS(x)    CosTable[x]                // Macro COS()

// -------------------  STRUCTURES DE DONNEES --------------------------//

// type matrice reelle de 4x4
typedef float _mat4x4[4][4];

// Structure pour representer un point dans un espace 2D
typedef struct _2D
{
  int x,y;
};

// Structure pour representer un point dans un espace 3D
typedef struct _3D
{
  float x,y,z;
};

// Structure qui definie un sommet avec ses differentes coordonnees
typedef struct _sommet
{
  _2D ecran;                 // coordonnees d'ecran
  _3D local;                 // coordonnees locales
  _3D monde;                 // coordonnees dans le monde
};

// Structure pour representer une face d'un polygone
typedef struct _tri
{
  short a,b,c;               // trois points d'un triangle
  unsigned char col;         // couleur de la face
  float z;                   // profondeur z moyenne (pour tri)
  _3D normale;               // normale de la face
};

// Structure pour contenir un objet
typedef struct _objet
{
  int nbsommet, nbpolygone;  // nombre de sommets et de polygones
  _sommet sommet[MAX_SOMM];  // coordonnees des sommets
  _tri    poly[MAX_POLY];    // polygones (triangles)
};

typedef struct TScan         // Structure pour le remplissage de polygones
{
  long gauche,droite;
};


//------------------------- VARIABLES GLOBALES ------------------------//

char  *ecran   = (char *) (0xA0000000L);    // Memoire video
char  *virtuel = new char[64000L];          // Ecran virtuel
float SinTable[360], CosTable[360];         // Table des sinus et cosinus

int   ordre[MAX_POLY];                      // Tableau pour trier les polys
TScan scanline[200];                        // Largeur des lignes des polys
int   miny,maxy;                            // hauteur min et max des polys

_mat4x4 matrice;                            // mat de transformation homogene
_mat4x4 mat1, mat2;                         // matrices temporaires
_3D lumiere = {0,0,1};                      // vecteur de lumiere

_objet unObjet;                             // un objet 3D

// ------------------------- FONCTIONS --------------------------------//

/////////////////////////////////////////////////////////////////////////
// hline - Dessine une ligne horizontale                               //
/////////////////////////////////////////////////////////////////////////
void hline(int x1, int x2, int y, unsigned char coul)
{
  memset(virtuel+x1+(y<<8)+(y<<6),coul,(x2-x1));
}

/////////////////////////////////////////////////////////////////////////
// setpal - modifie la palette                                         //
/////////////////////////////////////////////////////////////////////////
void setpal(unsigned char col, unsigned char r, unsigned char g, unsigned char b)
{
   outp (0x03C8,col);
   outp (0x03C9,r);
   outp (0x03C9,g);
   outp (0x03C9,b);
}

/////////////////////////////////////////////////////////////////////////
// SetupPal() - Palette graduelle de bleu, et blanc vers la fin        //
/////////////////////////////////////////////////////////////////////////
void preparepal()
{
  for (int i=0; i<192;i++) setpal(i,0,0,(i*63/192));
  for (i=192;i<256;i++)    setpal(i,i-192,i-192,63);
}

/////////////////////////////////////////////////////////////////////////
// precalc - Calcule le tableau de sinus/cosinus                       //
/////////////////////////////////////////////////////////////////////////
void precalc()
{
  for(int angle=0; angle<360; angle++)
  {
	 SinTable[angle]=sin(angle*M_PI/180.0);
	 CosTable[angle]=cos(angle*M_PI/180.0);
  }
}

/////////////////////////////////////////////////////////////////////////
// scalaire - Produit scalaire (retourne l'angle entre v1 et v2)       //
/////////////////////////////////////////////////////////////////////////
double scalaire(_3D v1, _3D v2)
{
  return (v1.x * v2.x) + (v1.y * v2.y) + (v1.z * v2.z);
}

/////////////////////////////////////////////////////////////////////////
// vectoriel - Produit vectoriel (retourne l'orthogonal de v1 et v2)   //
/////////////////////////////////////////////////////////////////////////
void vectoriel(_3D *v, _3D v1, _3D v2)
{
  v -> x = (v1.y * v2.z) - (v2.y * v1.z);
  v -> y = (v1.z * v2.x) - (v2.z * v1.x);
  v -> z = (v1.x * v2.y) - (v2.x * v1.y);
}

/////////////////////////////////////////////////////////////////////////
// normalise - retourne un vecteur unitaire (longueur de 1)            //
/////////////////////////////////////////////////////////////////////////
void normalise(_3D *n)
{
  double longueur = sqrt(n->x*n->x + n->y*n->y + n->z*n->z);
  if(longueur==0) return;
  n -> x /= longueur;
  n -> y /= longueur;
  n -> z /= longueur;
}

/////////////////////////////////////////////////////////////////////////
// swap - Effectue l'echange entre 2 variables float                   //
/////////////////////////////////////////////////////////////////////////
void swap(float &x,float &y)
{
  float temp = x;
	   x = y;
	   y = temp;
}

/////////////////////////////////////////////////////////////////////////
// Scan - Trouve le minX et maxX d'un cote d'un polygone                //
/////////////////////////////////////////////////////////////////////////
void scan(float x1, float y1, float x2, float y2)
{
  if (y1==y2) return;
  if (y2<y1)  {swap (y1,y2); swap (x1,x2);}
  if(y1<miny) miny=y1;
  if(y2>maxy) maxy=y2;
  double Xinc = (x2-x1) / (y2-y1);
  double x = x1 += Xinc;

  for (int y=y1;y<y2;y++)
  {
    if (x < scanline[y].gauche) scanline[y].gauche = x;
    if (x > scanline[y].droite) scanline[y].droite = x;
    x+=Xinc;
  }
}

/////////////////////////////////////////////////////////////////////////
// dessinepoly - Dessine un polygone avec liste de points(listesommet) //
/////////////////////////////////////////////////////////////////////////
void dessine_poly(_2D *listesommet, unsigned char coul)
{
  _2D *ptrcour = listesommet;
  _2D *ptrsuiv = listesommet+1;

  miny=200; maxy=0;
  for (int i=0;i<200;i++)
  {
   scanline[i].gauche = 32000;
   scanline[i].droite = -32000;
  }

  for (i=1; i<3; i++)
  {
    scan(ptrcour->x, ptrcour->y, ptrsuiv->x, ptrsuiv->y);
    ptrcour++;
    ptrsuiv++;
  }
  ptrsuiv = listesommet;
  scan(ptrcour->x, ptrcour->y, ptrsuiv->x, ptrsuiv->y);

  for (int y=miny;y<maxy;y++)
    hline (scanline[y].gauche,scanline[y].droite,y,coul);
}

/////////////////////////////////////////////////////////////////////////
// copie_matrice -  copie une matrice source vers matrice destination  //
/////////////////////////////////////////////////////////////////////////
void copie_matrice(_mat4x4 source, _mat4x4 dest)
{
  memcpy(dest,source,sizeof(_mat4x4));
}


/////////////////////////////////////////////////////////////////////////
// mult_matrice - multiplie 2 matrices et mets le resultat dans dest   //            //
/////////////////////////////////////////////////////////////////////////
void mult_matrice(_mat4x4 m1, _mat4x4 m2, _mat4x4 dest)
{
  for(short i=0;i<4;i++)
    for(short j=0;j<4;j++)
	 dest[i][j] = m1[i][0]*m2[0][j]+
			    m1[i][1]*m2[1][j]+
			    m1[i][2]*m2[2][j]+
			    m1[i][3]*m2[3][j];
}

/////////////////////////////////////////////////////////////////////////
// ident_matrice - construit une matrice identite                      //
/////////////////////////////////////////////////////////////////////////
void ident_matrice(_mat4x4 m)
{
    memset(m,NULL,sizeof(_mat4x4));   // 1  0  0  0
    m[0][0] = 1.0;                    // 0  1  0  0
    m[1][1] = 1.0;                    // 0  0  1  0
    m[2][2] = 1.0;                    // 0  0  0  1
    m[3][3] = 1.0;                    // matrice identite
}

/////////////////////////////////////////////////////////////////////////
// echelle - matrice de changement d'echelle                           //
/////////////////////////////////////////////////////////////////////////
void echelle(_mat4x4 m,float ex,float ey, float ez)
{
   _mat4x4 emat;                      // matrice echelle

   ident_matrice(emat);               // initialise matrice identite
   emat[0][0]=ex;                     // ex 0  0  0
   emat[1][1]=ey;                     // 0  ey 0  0
   emat[2][2]=ez;                     // 0  0  ez 0
							   // 0  0  0  1
   mult_matrice(m,emat,mat1);         // (emat X m) -> mat1
   copie_matrice(mat1,m);             // copie le resultat dans matrice
}                                     // globale de transformation homogene

/////////////////////////////////////////////////////////////////////////
// translation - matrice de translation                                //
/////////////////////////////////////////////////////////////////////////
void translation(_mat4x4 m,float tx,float ty,float tz)
{
   _mat4x4 tmat;                      // matrice translation

   ident_matrice(tmat);               // initialise matrice identite
   tmat[3][0]=tx;                     // 1  0  0  0
   tmat[3][1]=ty;                     // 0  1  0  0
   tmat[3][2]=tz;                     // 0  0  1  0
							   // tx ty tz 1
   mult_matrice(m,tmat,mat1);         // (tmat X m) -> mat1
   copie_matrice(mat1,m);             // copie le resultat dans matrice
}                                     // globale de transformation homogene


/////////////////////////////////////////////////////////////////////////
// rotation - matrices de rotations                                    //
/////////////////////////////////////////////////////////////////////////
void rotation(_mat4x4 m,int ax,int ay,int az)
{
  _mat4x4 xmat, ymat, zmat;

  ident_matrice(xmat);
  ident_matrice(ymat);
  ident_matrice(zmat);

  xmat[1][1] =  COS(ax);  xmat[1][2] = SIN(ax);
  xmat[2][1] = -SIN(ax);  xmat[2][2] = COS(ax);

  ymat[0][0] =  COS(ay);  ymat[0][2] = -SIN(ay);
  ymat[2][0] =  SIN(ay);  ymat[2][2] =  COS(ay);

  zmat[0][0] =  COS(az);  zmat[0][1] =  SIN(az);
  zmat[1][0] = -SIN(az);  zmat[1][1] =  COS(az);

  mult_matrice(m,ymat,mat1);
  mult_matrice(mat1,xmat,mat2);
  mult_matrice(mat2,zmat,m);
}

/////////////////////////////////////////////////////////////////////////
// projection - transformation 3D -> 2D                                //
/////////////////////////////////////////////////////////////////////////
void projection(_sommet *sommet)
{
   sommet->ecran.x = sommet->monde.x * DISTANCE / sommet->monde.z + MX;
   sommet->ecran.y = sommet->monde.y * DISTANCE / sommet->monde.z + MY;
}

/////////////////////////////////////////////////////////////////////////
// transformation - multiplication de chaque sommet par la matrice     //
/////////////////////////////////////////////////////////////////////////
void transformation(_mat4x4 m)
{
   _sommet *sommet;

   for(int v=0; v<unObjet.nbsommet; v++)
   {
	 sommet = &unObjet.sommet[v];

	 sommet->monde.x = sommet->local.x*m[0][0]+
				    sommet->local.y*m[1][0]+
				    sommet->local.z*m[2][0]+
								m[3][0];

	 sommet->monde.y = sommet->local.x*m[0][1]+
				    sommet->local.y*m[1][1]+
				    sommet->local.z*m[2][1]+
								m[3][1];

	 sommet->monde.z = sommet->local.x*m[0][2]+
				    sommet->local.y*m[1][2]+
				    sommet->local.z*m[2][2]+
								m[3][2];
	 projection(sommet);
   }
}

/////////////////////////////////////////////////////////////////////////
// calcnormal - calcule les normales de face pour chaque polygones     //
/////////////////////////////////////////////////////////////////////////
void calcnormal()
{
  _3D v1,v2;

  for(int face=0; face<unObjet.nbpolygone; face++)
  {
    v1.x = unObjet.sommet[unObjet.poly[face].a].monde.x - unObjet.sommet[unObjet.poly[face].b].monde.x;
    v1.y = unObjet.sommet[unObjet.poly[face].a].monde.y - unObjet.sommet[unObjet.poly[face].b].monde.y;
    v1.z = unObjet.sommet[unObjet.poly[face].a].monde.z - unObjet.sommet[unObjet.poly[face].b].monde.z;

    v2.x = unObjet.sommet[unObjet.poly[face].c].monde.x - unObjet.sommet[unObjet.poly[face].b].monde.x;
    v2.y = unObjet.sommet[unObjet.poly[face].c].monde.y - unObjet.sommet[unObjet.poly[face].b].monde.y;
    v2.z = unObjet.sommet[unObjet.poly[face].c].monde.z - unObjet.sommet[unObjet.poly[face].b].monde.z;

    vectoriel(&unObjet.poly[face].normale,v2,v1);
    normalise(&unObjet.poly[face].normale);
  }
}

/////////////////////////////////////////////////////////////////////////
// trier_faces - trie les faces selon leurs Z moyen (Bubble Sort)      //
/////////////////////////////////////////////////////////////////////////
void trier_faces(int nb)
{
  int position = 0;
  int tempval;

  while (position < nb-1)
  {
    if (unObjet.poly[ordre[position]].z > unObjet.poly[ordre[position+1]].z)
    {
	 tempval = ordre[position+1];
	 ordre[position+1] = ordre[position];
	 ordre[position] = tempval;
	 position = -1;
    }
    position++;
  }
}

/////////////////////////////////////////////////////////////////////////
// dessine_objet - dessine les sommets de l'objet                      //
/////////////////////////////////////////////////////////////////////////
void dessine_objet()
{
  int nb=0;
  double angle;
  float Znormale;
  unsigned char col;
  _2D poly2D[3];

  // Boucle principale du rendeur
  for(int face=0; face<unObjet.nbpolygone; face++)
  {
    poly2D[0] = unObjet.sommet[unObjet.poly[face].a].ecran;
    poly2D[1] = unObjet.sommet[unObjet.poly[face].b].ecran;
    poly2D[2] = unObjet.sommet[unObjet.poly[face].c].ecran;

    Znormale = (poly2D[0].y - poly2D[2].y) *
			(poly2D[1].x - poly2D[0].x) -
			(poly2D[0].x - poly2D[2].x) *
			(poly2D[1].y - poly2D[0].y);

    if (Znormale < 0)
    {

	 unObjet.poly[face].z = unObjet.sommet[unObjet.poly[face].a].monde.z+
					    unObjet.sommet[unObjet.poly[face].b].monde.z+
					    unObjet.sommet[unObjet.poly[face].c].monde.z;
	 ordre[nb++] = face;
	 angle = scalaire(unObjet.poly[face].normale,lumiere);
	 if (angle<0)  col = AMBIANT; else col = AMBIANT + DIFFUSE * angle;
	 unObjet.poly[face].col = col;
    }
  }

  trier_faces(nb);

  for (face=0;face<nb;face++)
  {
	 poly2D[0] = unObjet.sommet[unObjet.poly[ordre[face]].a].ecran;
	 poly2D[1] = unObjet.sommet[unObjet.poly[ordre[face]].b].ecran;
	 poly2D[2] = unObjet.sommet[unObjet.poly[ordre[face]].c].ecran;

	 dessine_poly(poly2D,unObjet.poly[ordre[face]].col);
  }
}

/////////////////////////////////////////////////////////////////////////
// loadASC - charge les vertices et les polygones d'un objet 3D Studio //
/////////////////////////////////////////////////////////////////////////
void loadASC(char *nom)
{
  FILE  *fichier;
  char  chaine[200];
  char  *fin;
  int   i,j;
  char  temp[50];
  float x,y,z;
  int   a,b,c;
  int   Nb_points=0;
  int   Nb_faces=0;
  int   decalage=0;

  if ((fichier = fopen(nom,"rt"))==NULL)
  {
    perror("Impossible d'ouvrir le fichier en lecture");
    exit(-2);
  }

  do
  {
    // On lit le fichier contenant les informations sur l'objet
    fin=fgets(chaine,100,fichier);
    if (!strncmp(chaine,"Vertex",6))
    {
	 if (strncmp(chaine,"Vertex list",11))
	 {
	// Lecture des coordonnes d'un point
	i=0;

	while(chaine[i]!='X') i++;
	i+=2;
	sscanf(chaine+i,"%f",&x);

	while(chaine[i]!='Y') i++;
	i+=2;
	sscanf(chaine+i,"%f",&y);

	while(chaine[i]!='Z') i++;
	i+=2;
	sscanf(chaine+i,"%f",&z);

	unObjet.sommet[Nb_points].local.x=x;
	unObjet.sommet[Nb_points].local.y=y;
	unObjet.sommet[Nb_points].local.z=z;

	Nb_points++;
    }
    }
    else
    {
	 if (!strncmp(chaine,"Face",4))
	 {
	if (strncmp(chaine,"Face list",9))
	{
	  // Lecture d'une facette
	  i=j=0;
	  while(chaine[i]!='A') i++;
	  i+=2;
	  j=i;
	  while(chaine[j]!=' ') j++;
	  strncpy(temp,chaine+i,j-i);
	  temp[j-i]=0;

	  unObjet.poly[Nb_faces].a=atoi(temp)+decalage;

	  while(chaine[i]!='B') i++;
	  i+=2;
	  j=i;
	  while(chaine[j]!=' ') j++;
	  strncpy(temp,chaine+i,j-i);
	  temp[j-i]=0;
	  unObjet.poly[Nb_faces].b=atoi(temp)+decalage;

	  while(chaine[i]!='C') i++;
	  i+=2;
	  j=i;
	  while(chaine[j]!=' ') j++;
	  strncpy(temp,chaine+i,j-i);
	  temp[j-i]=0;
	  unObjet.poly[Nb_faces].c=atoi(temp)+decalage;

	  Nb_faces++;
	}
	 }
	 else
	if (!strncmp(chaine,"Named object",12))
	  decalage=Nb_points;
    }
  } while(fin!=NULL);

  fclose(fichier);
  unObjet.nbpolygone=Nb_faces;
  unObjet.nbsommet=Nb_points;
}

//------------------------ FONCTION PRINCIPALE --------------------------//

void main(void)
{
  char *nom;
  int angle=0;
  int choix=0;

  // choix d'un objet
  do
  {
    asm {MOV AX,0x03; INT 0x10}
    fflush(stdin);
    printf("Veuillez choisir un objet:\n\n");
    printf("1- Cube\n2- Tore\n3- Cylindre\n4- Teapot\n5- Sphere\n6- Spindle\n\n");
    printf("Choix ->");
    scanf(" %i", &choix);
  } while(choix <=0 || choix >6);

  switch(choix)
  {
   case 1: nom = "cube.asc";     break;
   case 2: nom = "torus.asc";    break;
   case 3: nom = "cylindre.asc"; break;
   case 4: nom = "teapot.asc";   break;
   case 5: nom = "sphere.asc";   break;
   case 6: nom = "spindle.asc";  break;
  }

  // charge et affiche les informations de l'objet (sommet et polygones)
  loadASC(nom);
  printf("\nNom de l'objet     :  %s\n",nom);
  printf("Nombre de sommet   :  %i\n",unObjet.nbsommet);
  printf("Nombre de polygone :  %i\n",unObjet.nbpolygone);
  printf("\n\nAppuyez sur <retour>\n");
  getch();

  asm {MOV AX,0x13; INT 0x10}
  preparepal();
  precalc();
  normalise(&lumiere);

  while(!kbhit())
  {
    memset(virtuel,0,64000L);

    ident_matrice(matrice);

    echelle(matrice,1.0,1.0,1.0);
    rotation(matrice,angle,angle,angle);
    translation(matrice,0,0,-100);

    transformation(matrice);
    calcnormal();
    dessine_objet();

    while(!(inp(0x3DA)&8));
    memcpy(ecran,virtuel,64000L);
    if (angle++ == 359) angle = 0;
  };

  delete []virtuel;
  asm {MOV AX,0x03; INT 0x10}
  printf("Shaun Dore\ndores@videotron.ca\nhttp://pages.infinit.net/shaun/");
}
