// Exemple de tunnel  direction libre (raytracing)
// DATE:	22/06/99
// AUTEUR:	Shaun Dor
//			Krashlog / Creative Impact
//			dores@videotron.ca
//			http://pages.infinit.net/shaun/
//          http://pages.infinit.net/pcdesign/creative.impact/

// MACROS ////////////////////////////////////////////////////////////////

#define WIN32_LEAN_AND_MEAN  
// Macro qui retourne vrai si une touche est enfonce
#define Touche(VkCode) (GetAsyncKeyState(VkCode) & 0x8000 ? 1 : 0)
// Rsolution maximale X
#define MAX_X 640
// Rsolution maximale Y
#define MAX_Y 480

// INCLUDES //////////////////////////////////////////////////////////////

#include <io.h>
#include <math.h>
#include <ddraw.h>
#include <stdio.h>
#include <windows.h>
#include <windowsx.h>
#include "resource.h"

// TYPES /////////////////////////////////////////////////////////////////

// Un vecteur
typedef struct _vecteur
{
  float x, y, z;
} _vecteur;

// Grille d'interpolation
typedef struct _Grille
{
  float u,v;
} _Grille;

// Image bitmap
typedef struct BITMAP_FILE_TAG
{
	BITMAPFILEHEADER bitmapfileheader;  // Header du bitmap
	BITMAPINFOHEADER bitmapinfoheader;  // palette
	PALETTEENTRY     palette[256];      // 256 couleurs
	UCHAR            *buffer;           // les graphiques
} BITMAP_FILE, *BITMAP_FILE_PTR;


// GLOBALES /////////////////////////////////////////////////////////////

LPDIRECTDRAW		DirectDraw			= NULL;		// Objet DirectDraw
LPDIRECTDRAWSURFACE SurfacePrimaire		= NULL;		// Surface primaire
LPDIRECTDRAWSURFACE SurfaceSecondaire	= NULL;		// L'ecran virtuel de la surface primaire
LPDIRECTDRAWPALETTE lpddpal				= NULL;		// a pointer to the created dd palette
PALETTEENTRY        palette[256];					// color palette
DDSURFACEDESC		SurfaceDesc;					// Description de la surface
DDSCAPS				SurfaceCap;						// Capacits de la surface
HINSTANCE			InstancePrincipale	= NULL;		// Instance de l'application
HWND				HandlePrincipale	= NULL;		// Handle sur la fentre principale
BOOL				Quitter				= FALSE;	// Test pour savoir si on quitte l'app
UCHAR				*virtuel			= NULL;		// Ptr sur cran virtuel
BITMAP_FILE         bitmap8bit;						// un bitmap 8-bit


int ECRAN_X = 0;	// Rsolution X
int ECRAN_Y = 0;	// Rsolution Y
int INCR = 0;		// Incrment d'interpolation
int MID_X = 0;		// Millieu des X
int MID_Y = 0;		// Millieu des Y

// sin & cos pour rotation
float SinTable[256];
float CosTable[256];      

_Grille Grille[MAX_X+16][MAX_Y+16]; // Grille d'interpolation linaire
_vecteur camera = {0,0,0};			// vecteur de camera     
BYTE angle=0;						// Angle de rotation

// DCLARATIONS ////////////////////////////////////////////////////////////

int  InitDirectDraw();		// Initialisation de DirectDraw
void CloseDirectDraw();		// Libre la mmoire des objets DirectDraw
int  DessinePixel();		// Dessine des pixels sur l'cran secondaire
int  InitFenetre();			// Initialise la fentre


// FONCTIONS ///////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////
// Normalisation d'un vecteur
void vec_norm(_vecteur *v)
{
  float l = (float) 1.0 / (float) sqrt(v->x*v->x + v->y*v->y + v->z*v->z);
  v->x *= l;
  v->y *= l;
  v->z *= l;
}

////////////////////////////////////////////////////////////////////////////
// Calcul des tableaux sin/cos
void PreCalc()
{
	for(int angle=0; angle<256; angle++)
	{
		SinTable[angle]=(float)sin(angle*3.14159265358979323/128.0);
		CosTable[angle]=(float)cos(angle*3.14159265358979323/128.0);
	}
}

////////////////////////////////////////////////////////////////////////////
// Rotation sur l'axe X
void rotx(_vecteur *v, BYTE a)
{
  _vecteur temp = *v;
  v->y = temp.y*CosTable[a] - temp.z*SinTable[a];
  v->z = temp.y*SinTable[a] + temp.z*CosTable[a];
  v->x = temp.x;
}

////////////////////////////////////////////////////////////////////////////
// Raytracing (intersection avec tunnel)
void calc_tunnel(int x, int y)
{
  _vecteur direction = {(float)x-MID_X,(float)y-MID_Y,256.0}, intersect;
  float a, b, c, delta, t, t1, t2;

  // Normalise le vecteur et rotation
  rotx(&direction,angle);
  vec_norm(&direction);
  
  a = direction.x*direction.x + direction.y*direction.y;
  b = 2*(camera.x*direction.x + camera.y*direction.y);
  c = camera.x*camera.x + camera.y*camera.y - 256*256;

  delta = (float) sqrt(b*b - 4*a*c);
  t1 = (float) ((-b+delta) / (2*a+.0000001)); // attention aux div/0!
  t2 = (float) ((-b-delta) / (2*a+.0000001));

  t = t1 > 0 ? t1 : t2;

  // Point d'intersection
  intersect.x = camera.x + t*direction.x;
  intersect.y = camera.y + t*direction.y;
  intersect.z = camera.z + t*direction.z;

  // Rempli la grille d'interpolation  (mapping conique)
  Grille[x][y].u = (float) fabs(intersect.z*0.1);
  Grille[x][y].v = (float) fabs(atan2(intersect.y, intersect.x)*256/3.14159265358979323);
}

////////////////////////////////////////////////////////////////////////////
// Interpolation bilinaire dans la grille (version floating-point)
void interpolation()
{
	int x,y,i,j,yoff;
	float dul,dur,dbl,dbr,du,dv,ul,ur,bl,br,u,v;

	// Lock la surface virtuelle
	SurfaceSecondaire->Lock(NULL,&SurfaceDesc, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT,NULL);

	// Ramene un ptr sur la surface secondaire
	virtuel = (UCHAR *)SurfaceDesc.lpSurface; 

	for (j=0; j<ECRAN_Y; j+=INCR)
		for (i=0; i<ECRAN_X; i+=INCR)
		{
			// coin gauche suprieur, droit suprieur, gauche infrieur, droit infrieur
			dul = (Grille[i][j+INCR].u - Grille[i][j].u) / INCR;
			dur = (Grille[i+INCR][j+INCR].u - Grille[i+INCR][j].u) / INCR;
			dbl = (Grille[i][j+INCR].v - Grille[i][j].v) / INCR;
			dbr = (Grille[i+INCR][j+INCR].v - Grille[i+INCR][j].v) / INCR;
			ul	= (Grille[i][j].u);
			bl	= (Grille[i][j].v);
			ur	= (Grille[i+INCR][j].u);
			br	= (Grille[i+INCR][j].v);

			for (y=j; y<j+INCR; y++)
			{
				du = (ur - ul) / INCR;
				dv = (br - bl) / INCR;
				u = ul;
				v = bl;
				yoff=(y*ECRAN_X);
				for (x=i; x<i+INCR; x++)
				{
					virtuel[yoff+x] = bitmap8bit.buffer[(BYTE)u+(((BYTE)v)<<8)];
					u+=du;
					v+=dv;
				}
				ul += dul; 
				ur += dur;
				bl += dbl;
				br += dbr;
			}
    }
	// Libre la surface secondaire
	SurfaceSecondaire->Unlock(virtuel);	
	// Flip!
	while(SurfacePrimaire->Flip(NULL,DDFLIP_WAIT)!=DD_OK);	
	if (Touche(VK_ESCAPE))
		PostMessage(HandlePrincipale, WM_DESTROY, 0, 0);
}


////////////////////////////////////////////////////////////////////////////
// Le gestionnaire d'vnements pour notre bote de dialogue
LRESULT CALLBACK DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)  
{
	int i;
	RECT  rc;
	// Chaine de car pour le combo box
	char* interp[] = {"4","8","16"};
	// Int des chaines de ar
	int interpola[] = {4,8,16};

	switch(msg)
	{
		// Cration de la bote de dialogue
		case WM_INITDIALOG:
		// Centre de l'cran
		GetWindowRect( hwnd, &rc );
		SetWindowPos( hwnd, HWND_TOP,
						(GetSystemMetrics(SM_CXSCREEN) - rc.right)  / 2,
						(GetSystemMetrics(SM_CYSCREEN) - rc.bottom) / 2,
						0, 0, SWP_NOSIZE);
		// Bouton 1 checked
		SendDlgItemMessage( hwnd, IDC_320200, BM_SETCHECK, 1, 0 );
		for ( i=0 ; i<3 ; i++ ) 
		{
			SendDlgItemMessage( hwnd, IDC_INTERPOL, CB_ADDSTRING, 0, (int)interp[i] );
		}
		// Choisir interpolation 8x8
		SendDlgItemMessage( hwnd, IDC_INTERPOL, CB_SETCURSEL, 1, 0 );
		break;
		// Un bouton de commande (le # du bouton est contenu dans le low 8-bit de wParam)
		case WM_COMMAND: 
		{                                                       
			switch(LOWORD(wParam))
			{
				case ID_OK:
					if ( SendDlgItemMessage( hwnd, IDC_320200, BM_GETCHECK,  0, 0)) 
					{ 
						ECRAN_X = 320;
						ECRAN_Y = 200;
					}
					else if ( SendDlgItemMessage( hwnd, IDC_512384, BM_GETCHECK,  0, 0)) 
					{	
						ECRAN_X = 512;
						ECRAN_Y = 384;
					}
					else if ( SendDlgItemMessage( hwnd, IDC_640480, BM_GETCHECK,  0, 0))
					{
						ECRAN_X = 640;
						ECRAN_Y = 480;			
					}

					MID_X = ECRAN_X/2;
					MID_Y = ECRAN_Y/2;
				    INCR = interpola[ SendDlgItemMessage(hwnd,IDC_INTERPOL,CB_GETCURSEL,0,0)];
					EndDialog(hwnd, FALSE);
					break;

				case ID_ABOUT:
					{
						MessageBox(HandlePrincipale,"TUNNEL \n\nCode: Krashlog/Creative Impact\nEmail: dores@videotron.ca\nhttp://pages.infinit.net/shaun/\nhttp://pages.infinit.net/pcdesign/creative.impact/","A propos...",MB_OK | MB_ICONINFORMATION);
						break;
					}
				case ID_QUIT: 
					{
						EndDialog(hwnd, FALSE);
						Quitter = TRUE;
						break;
					}
			}
			break;
		}
        default: 	
			return FALSE; 
     }
     return TRUE; 
}


////////////////////////////////////////////////////////////////////////////
// Gestionnaire d'vnements de la fentre
LRESULT CALLBACK WindowProc(HWND hWnd,     // Handle sur la fentre
                            UINT Msg,      // Identificateur du message
                            WPARAM wParam, 
                            LPARAM lParam) 
{
	// Analyse le message
	switch (Msg)
	{	
		// La fentre va tre dtruite
		case WM_DESTROY:
		{
			// Envoie un message  la fentre pour quitter
			PostQuitMessage(0);
			return 0;
		}	break;
		
		default:
			break;
	}
	// Le reste des messages au gestionnaire par dfaut
	return DefWindowProc(hWnd, Msg, wParam, lParam);
}

////////////////////////////////////////////////////////////////////////////
// Initialise et instancie une fentre
int InitFenetre()
{
	WNDCLASS ClasseFenetre;	// Le nom de notre classe
	HWND hWnd;				// Handle sur la fentre

	// Style de la fentre
	ClasseFenetre.style = CS_DBLCLKS | CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	// Gestionnaire d'vnements de la fentre
	ClasseFenetre.lpfnWndProc = WindowProc;
	// Paramtres supplmentaires
	ClasseFenetre.cbClsExtra = 0;
	// Paramtres supplmentaires
	ClasseFenetre.cbWndExtra = 0;
	// Ptr sur instance de l'application
	ClasseFenetre.hInstance = InstancePrincipale;
	// Icne de la fentre
	ClasseFenetre.hIcon = LoadIcon(InstancePrincipale, MAKEINTRESOURCE(IDI_ICON1));
	// Curseur de la fentre
	ClasseFenetre.hCursor = LoadCursor(NULL, IDC_ARROW);
	// Palette de couleur
	ClasseFenetre.hbrBackground = (struct HBRUSH__ *)GetStockObject(BLACK_BRUSH);
	// Ptr sur menu
	ClasseFenetre.lpszMenuName = NULL;
	// Ptr sur nom de la classe
	ClasseFenetre.lpszClassName = "Fenetre";
	
	// Enregistre la classe
	if (!RegisterClass(&ClasseFenetre))
		return 0;

	// Cration de la fentre
	hWnd = CreateWindow("Fenetre",              
	                    "Tunnel",          
	                    WS_POPUP | WS_VISIBLE, 
	                    CW_USEDEFAULT,         
	                    CW_USEDEFAULT,        
	                    CW_USEDEFAULT,         
	                    CW_USEDEFAULT,         
	                    NULL,                  
	                    NULL,                  
	                    InstancePrincipale,    
	                    NULL);   
	// S'assure que tout est OK
	if (!hWnd)
		return 0;
	
	// Sauvegarde l'handle de la fentre dans une globale
	HandlePrincipale = hWnd;
	return 1;
}

////////////////////////////////////////////////////////////////////////////
// Fonction principale
int WINAPI WinMain(HINSTANCE hInstance,     // Handle sur l'instance prsente
                   HINSTANCE hPrevInstance, // Handle sur l'instance prcdente
                   LPSTR lpCmbLine,         // Ptr sur les arguments
                   int nCmdShow)            // tat de la fentre
{
	MSG	Msg;
	InstancePrincipale = hInstance;
    DialogBox(InstancePrincipale, MAKEINTRESOURCE(IDD_MENU), NULL, ( DLGPROC ) DlgProc);
	EndDialog(HandlePrincipale, FALSE);

	if (!Quitter)
	{
		if(!InitFenetre()) Quitter = TRUE;
		if(!InitDirectDraw()) Quitter = TRUE;
	}

	// Tant qu'on n'a pas quitter
	while (Quitter == FALSE)
	{
		// Regarde s'il y a un message dans la msg queue
		if (PeekMessage(&Msg, NULL, 0, 0, PM_REMOVE))
		{
			// Vrifie si le message est un WM_QUIT
			if (Msg.message == WM_QUIT)
				Quitter = true;
			else
			{
				// Traduit le message
				TranslateMessage(&Msg);
				// Envoie le message au gestionnaire d'vnements
				DispatchMessage(&Msg);
			}
		}
		// Calcule les intersections des rayons avec le tunnel
		for (int y=0; y<ECRAN_Y+INCR; y+=INCR)
			for (int x=0; x<ECRAN_X+INCR; x+=INCR)
				calc_tunnel(x,y);
		// Interpolation linaire
		interpolation();
		// Avance la camra
		camera.z+=64;
		// Angle de rotation
		angle++;
	}
	CloseDirectDraw();
	return Msg.wParam;
}


////////////////////////////////////////////////////////////////////////////
// Charge l'image bitmap en mmoire
int Load_Bitmap_File(BITMAP_FILE_PTR bitmap, char *filename)
{
	int file_handle,index;        
	OFSTRUCT file_data;  

	if ((file_handle = OpenFile(filename,&file_data,OF_READ))==-1) return(0);
	_lread(file_handle, &bitmap->bitmapfileheader,sizeof(BITMAPFILEHEADER));
	_lread(file_handle, &bitmap->bitmapinfoheader,sizeof(BITMAPINFOHEADER));
	bitmap->buffer = (UCHAR *)malloc(bitmap->bitmapinfoheader.biSizeImage);
	_lread(file_handle, &bitmap->palette,256*sizeof(PALETTEENTRY));
	_lseek(file_handle,-(int)(bitmap->bitmapinfoheader.biSizeImage),SEEK_END);
	_lread(file_handle,bitmap->buffer,bitmap->bitmapinfoheader.biSizeImage);
	_lclose(file_handle);
	for (index=0; index < 256; index++)
	{
		int temp_color = bitmap->palette[index].peRed;
		bitmap->palette[index].peRed  = bitmap->palette[index].peBlue;
		bitmap->palette[index].peBlue = temp_color;
		bitmap->palette[index].peFlags = PC_NOCOLLAPSE;
	}
	return(1);
} 


////////////////////////////////////////////////////////////////////////////
// Cette fonction libre la mmoire allou pour le bitmap
int Unload_Bitmap_File(BITMAP_FILE_PTR bitmap)
{
	if (bitmap->buffer)
		free(bitmap->buffer);
	return(1);
} 

////////////////////////////////////////////////////////////////////////////
// Initialise DirectDraw
int InitDirectDraw()
{
	// Cration d'un objet DirectDraw
	if (DirectDrawCreate(NULL, &DirectDraw, NULL) != DD_OK)
		return 0;

	// Niveau de coopration (FullScreen)
	if (DirectDraw->SetCooperativeLevel(HandlePrincipale,
	                                    DDSCL_ALLOWMODEX | DDSCL_FULLSCREEN |
								        DDSCL_EXCLUSIVE | DDSCL_ALLOWREBOOT) != DD_OK)
		return 0;
	
	// Slection du mode vido
	if (DirectDraw->SetDisplayMode(ECRAN_X, ECRAN_Y, 8) != DD_OK)
	{
		MessageBox(HandlePrincipale,
		           "Le mode vido dsir n'est pas support par votre pilote. Assurez-vous d'avoir la dernire version du pilote DirectX de votre carte vido.",
		           "Erreur mode vido",
		           MB_OK | MB_ICONERROR);
		return 0;
	}

	// Initialise la structure de description de la surface
	ZeroMemory(&SurfaceDesc,sizeof(SurfaceDesc));
	SurfaceDesc.dwSize            = sizeof(SurfaceDesc);
	SurfaceDesc.dwFlags           = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
	SurfaceDesc.ddsCaps.dwCaps    = DDSCAPS_PRIMARYSURFACE | DDSCAPS_SYSTEMMEMORY | DDSCAPS_FLIP | DDSCAPS_COMPLEX;
	SurfaceDesc.dwBackBufferCount = 1;

	// Cration de la surface primaire
	if (DirectDraw->CreateSurface(&SurfaceDesc, &SurfacePrimaire, NULL) != DD_OK)
		return 0;
	
	// Cration de la surface secondaire (virtuelle)
	SurfaceCap.dwCaps = DDSCAPS_BACKBUFFER;
	SurfacePrimaire->GetAttachedSurface(&SurfaceCap, &SurfaceSecondaire);
	
	memset(palette,0,256*sizeof(PALETTEENTRY));

	for (int index=0; index<256; index++)
		palette[index].peFlags = PC_NOCOLLAPSE;

	// cre l'objet palette
	DirectDraw->CreatePalette(DDPCAPS_8BIT | DDPCAPS_INITIALIZE | DDPCAPS_ALLOW256,palette,&lpddpal,NULL);
	// Precalculation sin/cos
	PreCalc();	
	// Attache la palette 8-bit  la surface primaire
	SurfacePrimaire->SetPalette(lpddpal);
	// Load le bitmap
	Load_Bitmap_File(&bitmap8bit, "tunnel.bmp");
	lpddpal->SetEntries(0,0,256,bitmap8bit.palette);
	// Cacher le curseur
	ShowCursor(FALSE);
	return 1;
}

////////////////////////////////////////////////////////////////////////////
// Libre les objets DirectDraw
void CloseDirectDraw()
{
	// Libre surface secondaire
	if (SurfaceSecondaire != NULL)
		SurfaceSecondaire->Release();

	// Libre surface primaire
	if (SurfacePrimaire != NULL)
		SurfacePrimaire->Release();
	
	// Libre objet COM DirectDraw
	if (DirectDraw != NULL)
		DirectDraw->Release();

	// Affiche le curseur de souris
	ShowCursor(TRUE);
}
